<?php
	/************************************************
	* Copyright (C) 2016-2019	Sylvain Legrand - <contact@infras.fr>	InfraS - <https://www.infras.fr>
	*
	* This program is free software: you can redistribute it and/or modify
	* it under the terms of the GNU General Public License as published by
	* the Free Software Foundation, either version 3 of the License, or
	* (at your option) any later version.
	*
	* This program is distributed in the hope that it will be useful,
	* but WITHOUT ANY WARRANTY; without even the implied warranty of
	* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	* GNU General Public License for more details.
	*
	* You should have received a copy of the GNU General Public License
	* along with this program.  If not, see <http://www.gnu.org/licenses/>.
	************************************************/

	/************************************************
	* 	\file		../infraspackplus/core/lib/infraspackplus.lib.php
	* 	\ingroup	InfraS
	* 	\brief		Functions used by InfraS module
	************************************************/

	function infraspackplus_admin_prepare_head ()
	{
		global $langs, $conf, $user;

		$h				= 0;
		$head			= array();
		if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramDolibarr))
		{
			$head[$h][0]	= dol_buildpath('/infraspackplus/admin/generalpdf.php',1);
			$head[$h][1]	= $langs->trans("InfraSPlusParamsGeneralPDF");
			$head[$h][2]	= 'generalpdf';
		}	// if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramDolibarr))
		if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramInfraSPlus))
		{
			$h++;
			$head[$h][0]	= dol_buildpath('/infraspackplus/admin/infrasplussetup.php',1);
			$head[$h][1]	= $langs->trans("InfraSPlusParamsPDF");
			$head[$h][2]	= 'infrasplussetup';
		}	// if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramInfraSPlus))
		if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramImages))
		{
			$h++;
			$head[$h][0]	= dol_buildpath('/infraspackplus/admin/images.php',1);
			$head[$h][1]	= $langs->trans("InfraSPlusParamsImages");
			$head[$h][2]	= 'images';
		}	// if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramImages))
		if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramAdresses))
		{
			$h++;
			$head[$h][0]	= dol_buildpath('/infraspackplus/admin/adresses.php',1);
			$head[$h][1]	= $langs->trans("InfraSPlusParamsAdresses");
			$head[$h][2]	= 'adresses';
		}	// if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramAdresses))
		if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramExtraFields))
		{
			$h++;
			$head[$h][0]	= dol_buildpath('/infraspackplus/admin/extrafields.php',1);
			$head[$h][1]	= $langs->trans("InfraSPlusParamsExtraFields");
			$head[$h][2]	= 'extrafields';
		}	// if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramExtraFields))
		if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramMentions))
		{
			$h++;
			$head[$h][0]	= dol_buildpath('/infraspackplus/admin/mentions.php',1);
			$head[$h][1]	= $langs->trans("InfraSPlusParamsMentions");
			$head[$h][2]	= 'mentions';
		}	// if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramMentions))
		if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramNotes))
		{
			$h++;
			$head[$h][0]	= dol_buildpath('/infraspackplus/admin/notes.php',1);
			$head[$h][1]	= $langs->trans("InfraSPlusParamsNotes");
			$head[$h][2]	= 'notes';
		}	// if (! empty($user->admin) || ! empty($user->rights->infraspackplus->paramNotes))
		$h++;
		$head[$h][0]	= dol_buildpath('/infraspackplus/admin/about.php',1);
		$head[$h][1]	= $langs->trans("About");
		$head[$h][2]	= 'about';

		return $head;
	}	// function infraspackplus_admin_prepare_head ()
	
	/************************************************
	*	Test if the PHP extension 'XML' is loaded
	*
	************************************************/
	function infraspackplus_test_php_ext()
	{
		require_once DOL_DOCUMENT_ROOT.'/core/lib/admin.lib.php';
		
		global $db, $conf, $langs;
		
		$langs->load('infraspackplus@infraspackplus');

		if (extension_loaded('xml'))	dolibarr_set_const($db, "INFRAS_PHP_EXT_XML",	1, 'chaine', 0, '', $conf->entity);
		else
		{
			dolibarr_set_const($db, "INFRAS_PHP_EXT_XML",	-1, 'chaine', 0, '', $conf->entity);
			setEventMessages($langs->trans('InfraSXMLextError'), null, 'warnings');
		}	// else	// if (extension_loaded('xml'))
	}	// function infraspackplus_test_php_ext()

	/************************************************
	* Function called to check module name from local changelog
	* Control of the min version of Dolibarr needed and get versions list
	*
	* @param   string	$appliname	module name
	* @return	array				[0] current version from changelog
	*								[1] Dolibarr min version
	*								[2] flag for error (-1 = KO ; 0 = OK)
	*								[3] array => versions list or errors list
	************************************************/
	function infraspackplus_getLocalVersionMinDoli($appliname)
	{
		global $langs;

		$currentversion	= array();
		$sxe			= infraspackplus_getChangelogFile($appliname);
		if ($sxe === false)
		{
			$currentversion[0]	= '<font color=red><b>'.$langs->trans("InfraSPlusChangelogXMLError").'</b></font>';
			$currentversion[1]	= $langs->trans("InfraSPlusnoMinDolVersion");
			$currentversion[2]	= -1;
			$currentversion[3]	= $langs->trans("InfraSPlusChangelogXMLError");
			foreach (libxml_get_errors() as $error)
			{
				$currentversion[3]	.= $error->message;
				dol_syslog('infraspackplus.Lib::infraspackplus_getLocalVersionMinDoli error->message = '.$error->message);
			}	// foreach (libxml_get_errors() as $error)
		}	// if ($sxe === false)
		else
		{
			$currentversion[0]	= $sxe->Version[count($sxe->Version) - 1]->attributes()->Number;
			$currentversion[1]	= $sxe->Dolibarr->attributes()->minVersion;
			$currentversion[2]	= 0;
			$currentversion[3]	= $sxe->Version;
		}	// else	// if ($sxe === false)
		return $currentversion;
	}	// function infraspackplus_getLocalVersionMinDoli($appliname)

	/************************************************
	 * Function called to check module name from local changelog
	 * Control of the min version of Dolibarr needed and get versions list
	 *
	 * @param   string			$appliname	module name
	 * @param   string			$from		sufixe name to separate inner changelog from download
	 * @return	string|boolean				changelog file contents or false
	************************************************/
	function infraspackplus_getChangelogFile($appliname, $from = '')
	{
		$file	= dol_buildpath($appliname, 0).'/docs/changelog'.$from.'.xml';
		if (is_file($file))
		{
			libxml_use_internal_errors(true);
			$context	= stream_context_create(array('http' => array('header' => 'Accept: application/xml')));
			$changelog	= @file_get_contents($file, false, $context);
			$sxe		= @simplexml_load_string(rtrim($changelog));
			dol_syslog('infraspackplus.Lib::infraspackplus_getChangelogFile appliname = '.$appliname.' context = '.$context.' changelog = '.($changelog ? 'Ok' : 'KO').' sxe = '.($sxe ? 'Ok' : 'KO'));
			return $sxe;
		}	// if (is_file($file.))
		else return false;
	}	// function infraspackplus_getChangelogFile($appliname, $from = '')

	/************************************************
	*	Tests version du module et de Dolibarr.
	*	Avertissements si chgt trouvés.
	*
	************************************************/
	function infraspackplus_chgtVersions ()
	{
		require_once DOL_DOCUMENT_ROOT.'/core/lib/admin.lib.php';

		global $db, $conf, $langs;
		
		$langs->load('infraspackplus@infraspackplus');
		
		$infrasdolversion	= isset($conf->global->INFRASPLUS_DOL_VERSION) ? $conf->global->INFRASPLUS_DOL_VERSION : '';
		$lastversion		= isset($conf->global->INFRASPLUS_MAIN_VERSION) ? $conf->global->INFRASPLUS_MAIN_VERSION : '';
		$changelogversion	= infraspackplus_getLocalVersionMinDoli('infraspackplus');
		dol_syslog('infraspackplus.Lib::infraspackplus_chgtVersions infrasdolversion = '.$infrasdolversion.' lastversion = '.$lastversion.' changelogversion = '.$changelogversion[0]);
		if ($infrasdolversion && $infrasdolversion < DOL_VERSION)
		{
			dolibarr_set_const($db, "INFRASPLUS_PDF_VALID_CORE_CHGT",	0, 'chaine', 0, '', $conf->entity);
			setEventMessages($langs->trans('InfraSPlusDolChg', $infrasdolversion, DOL_VERSION), null, 'warnings');
		}	// if ($infrasdolversion && $infrasdolversion < DOL_VERSION)
		elseif ($lastversion && $lastversion < $changelogversion[0])
		{
			dolibarr_set_const($db, "INFRASPLUS_PDF_VALID_CORE_CHGT",	0, 'chaine', 0, '', $conf->entity);
			setEventMessages($langs->trans('InfraSPlusVersionChg', $lastversion, $changelogversion[0]), null, 'warnings');
		}	// elseif ($lastversion && $lastversion < $changelogversion[0])
	}	// function infraspackplus_chgtVersions ()

	/************************************************
	*	Tests présence de champs supplémentaires dans les tables.
	*
	*	@param   string	$appliname	module name
	************************************************/
	function infraspackplus_test_new_fields($appliname)
	{
		dol_syslog('infraspackplus.Lib::infraspackplus_test_new_fields appliname = '.$appliname);
		infraspackplus_get_test_new_fields($appliname, 'societe_address', 'email');
		infraspackplus_get_test_new_fields($appliname, 'societe_address', 'url');
		infraspackplus_get_test_new_fields($appliname, 'societe', 'logo_emet');	
	}	// 	function infraspackplus_test_new_fields($appliname)
	
	/************************************************
	*	Tests présence d'un champ supplémentaire dans une table.
	*	Exécution du script de création si besoin.
	*
	*	@param   string	$appliname	module name
	*	@param   string	$table		simple table name without any prefix
	*	@param   string	$field		field name
	************************************************/
	function infraspackplus_get_test_new_fields($appliname, $table, $field)
	{
		global $conf, $db;

		$path			= dol_buildpath($appliname, 0).'/sql';
		$sql_column		= 'SHOW COLUMNS FROM '.MAIN_DB_PREFIX.$table.' LIKE "'.$field.'"';
		$result_columns	= $db->query($sql_column);
		dol_syslog('infraspackplus.Lib::infraspackplus_get_test_new_fields sql_column = '.$sql_column);
		if (! $db->num_rows($result_columns))
		{
			$filetable	= $path.'/llx_'.$table.'-'.$field.'.sql';
			$result		= run_sql($filetable, 1, '', 1);
			dol_syslog('infraspackplus.Lib::infraspackplus_get_test_new_fields filetable = '.$filetable.' result = '.$result);
		}	// 	if (! $db->num_rows($result_columns))
		$db->free($result_columns);
	}	// function infraspackplus_get_test_new_fields($appliname, $table, $field)
	
	/************************************************
	*	Modifie le fichier actions_milestone.class.php
	*	pour le rendre compatible avec le module
	*
	*	@return		void
	************************************************/
	function infraspackplus_test_milestone()
	{
		global $langs;

		$path			= dol_buildpath('milestone', 0);
		if (strtoupper(substr(PHP_OS, 0, 3)) != 'WIN')	infraspackplus_chmod_r($path, '0755', '0755', 'www-data', 'www-data');
		dol_syslog('infraspackplus.lib::infraspackplus_test_milestone path = '.$path.' PHP_OS = '.PHP_OS);
		$fileactions	= $path.'/class/actions_milestone.class.php';
		$actions		= strpos (file_get_contents ($fileactions), 'InfraSPackPlus_model') === false ? file_get_contents ($fileactions) : false;
		$mypath			= dol_buildpath('infraspackplus', 0);
		$filereplace	= $mypath.'/includes/milestone/new.txt';
		if ($actions !== false && is_file($filereplace))
		{
			$reg					= '/else(.{1,6})\{(.{1,7})\$tab_top_newpage = \(empty\(\$conf->global->MAIN_PDF_DONOTREPEAT_HEAD/s';
			$moved					= dol_copy($fileactions, $fileactions.'.old');
			dol_syslog('infraspackplus.lib::infraspackplus_test_milestone fileactions = '.$fileactions.' moved = '.$moved);
			if ($moved > 0)			$result	= file_put_contents ($fileactions, preg_replace ($reg, file_get_contents ($filereplace), $actions));
			else					$result	= false;
			if ($result	=== false)	setEventMessages($langs->trans('InfraSPlusMilestoneError'), null, 'errors');
		}	// if ($actions !== false && is_file($filereplace))
	}	// function infraspackplus_test_milestone()
	
	/************************************************
	*	Sauvegarde les paramètres du module
	*
	*	@param		string		$appliname	module name
	*	@param		integer		$origine	wich action call the function : 1 = backup button on parameters, 2 = module desactivation
	*	@return		string		1 = Ok or -1 = Ko or or 0 and error message
	************************************************/
	function infraspackplus_bkup_module ($appliname, $origine = 1)
	{
		global $db, $conf, $langs, $errormsg;
			
		// core change Process
		$validcorechg		= $conf->global->INFRASPLUS_PDF_VALID_CORE_CHGT;
		if ($origine == 2)	$validcorechg	= $validcorechg	== 1 ? 2 : $validcorechg;
		dol_syslog('infraspackplus.Lib::infraspackplus_bkup_module appliname = '.$appliname.' origine = '.$origine.' validcorechg = '.$validcorechg);
		// Set to UTF-8
		if (is_a($db, 'DoliDBMysqli'))	$db->db->set_charset('utf8');
		else
		{
			$db->query('SET NAMES utf8');
			$db->query('SET CHARACTER SET utf8');
		}	// else	// if (is_a($db, 'DoliDBMysqli'))
		// Control dir and file
		$path		= dol_buildpath($appliname, 0).'/sql';
		$bkpfile	= $path.'/update.'.$conf->entity;
		if (! file_exists($path))
		{
			if (dol_mkdir($path) < 0)
			{
				$errormsg	= $langs->transnoentities("ErrorCanNotCreateDir", $path);
				return 0;
			}	// if (dol_mkdir($path) < 0)
		}	// if (! file_exists($path))
		if (file_exists($path))
		{
			$currentversion	= infraspackplus_getLocalVersionMinDoli('infraspackplus');
			$handle			= fopen($bkpfile, 'w+');
			if (fwrite($handle, '') === FALSE)
			{
				$langs->load("errors");
				$errormsg	= $langs->trans("ErrorFailedToWriteInDir");
				return -1;
			}	// if (fwrite($handle, '') === FALSE)
			// Print headers and global mysql config vars
			$sqlhead	= '-- '.$db::LABEL.' dump via php with Dolibarr '.DOL_VERSION.'
--
-- Host: '.$db->db->host_info.'    Database: '.$db->database_name.'
-- ------------------------------------------------------
-- Server version			'.$db->db->server_info.'
-- Dolibarr version			'.DOL_VERSION.'
-- InfraSPackPlus version	'.$currentversion[0].'

SET FOREIGN_KEY_CHECKS = 0;
SET SQL_MODE = \'NO_AUTO_VALUE_ON_ZERO\';
LOCK TABLES '.MAIN_DB_PREFIX.'document_model WRITE, '.MAIN_DB_PREFIX.'const WRITE, '.MAIN_DB_PREFIX.'c_infraspackplus_mention WRITE, '.MAIN_DB_PREFIX.'c_infraspackplus_note WRITE;
';
			fwrite($handle, $sqlhead);
			$sql_model		= 'SELECT nom, entity, type, libelle';
			$sql_model		.= ' FROM '.MAIN_DB_PREFIX.'document_model';
			$sql_model		.= ' WHERE nom LIKE "INFRASPLUS\_%" AND entity = "'.$conf->entity.'"';
			$sql_model		.= ' ORDER BY name';
			$listeCols		= array ('nom', 'entity', 'type', 'libelle');
			$duplicate		= array ('3', 'libelle', 'nom');
			fwrite($handle, infraspackplus_bkup_table ('document_model', $sql_model, $listeCols, $duplicate, 0, ''));
			$sql_const		= 'SELECT name, entity, value, type';
			$sql_const		.= ' FROM '.MAIN_DB_PREFIX.'const';
			$sql_const		.= ' WHERE ((name LIKE "INFRASPLUS\_%" AND name NOT LIKE "INFRASPLUS\_PDF\_VALID\_CORE\_CHGT") OR (name LIKE "%\_ADDON\_PDF"  AND value LIKE "InfraSPlus\_%") OR name LIKE "%\_FREE\_TEXT%" OR name LIKE "%\_PUBLIC\_NOTE%")';
			$sql_const		.= ' AND entity = "'.$conf->entity.'"';
			$sql_const		.= ' ORDER BY name';
			$autoupdate		= isset($conf->global->MAIN_DISABLE_PDF_AUTOUPDATE)	? $conf->global->MAIN_DISABLE_PDF_AUTOUPDATE : 1;
			$onDuplicate	= $db->type == 'pgsql' ? ' ON CONFLICT (name) DO UPDATE SET ' : ' ON DUPLICATE KEY UPDATE ';
			$add			= 'INSERT INTO '.MAIN_DB_PREFIX.'const (name, entity, value, type) VALUES (\'MAIN_DISABLE_PDF_AUTOUPDATE\', \'__ENTITY__\', \''.$autoupdate.'\', \'chaine\')'.$onDuplicate.'value = \''.$autoupdate.'\';
INSERT INTO '.MAIN_DB_PREFIX.'const (name, entity, value, type) VALUES (\'INFRASPLUS_PDF_VALID_CORE_CHGT\', \'__ENTITY__\', \''.$validcorechg.'\', \'chaine\')'.$onDuplicate.'value = \''.$validcorechg.'\';
';
			$listeCols		= array ('name', 'entity', 'value', 'type');
			$duplicate		= array ('2', 'value', 'name');
			fwrite($handle, infraspackplus_bkup_table ('const', $sql_const, $listeCols, $duplicate, 0, $add));
			$sql_dict		= 'SELECT code, entity, pos, libelle, active';
			$sql_dict		.= ' FROM '.MAIN_DB_PREFIX.'c_infraspackplus_mention';
			$sql_dict		.= ' WHERE entity = "'.$conf->entity.'"';
			$sql_dict		.= ' ORDER BY pos';
			$listeCols		= array ('code', 'entity', 'pos', 'libelle', 'active');
			$duplicate		= array ('3', 'libelle', 'code');
			fwrite($handle, infraspackplus_bkup_table ('c_infraspackplus_mention', $sql_dict, $listeCols, $duplicate, 1, ''));
			$sql_dict		= 'SELECT code, entity, pos, libelle, active';
			$sql_dict		.= ' FROM '.MAIN_DB_PREFIX.'c_infraspackplus_note';
			$sql_dict		.= ' WHERE entity = "'.$conf->entity.'"';
			$sql_dict		.= ' ORDER BY pos';
			$listeCols		= array ('code', 'entity', 'pos', 'libelle', 'active');
			$duplicate		= array ('3', 'libelle', 'code');
			fwrite($handle, infraspackplus_bkup_table ('c_infraspackplus_note', $sql_dict, $listeCols, $duplicate, 1, ''));
			// Enabling back the keys/index checking
			$sqlfooter		= '
ALTER TABLE '.MAIN_DB_PREFIX.'document_model ENABLE KEYS;
ALTER TABLE '.MAIN_DB_PREFIX.'const ENABLE KEYS;
ALTER TABLE '.MAIN_DB_PREFIX.'c_infraspackplus_mention ENABLE KEYS;
ALTER TABLE '.MAIN_DB_PREFIX.'c_infraspackplus_note ENABLE KEYS;

UNLOCK TABLES;
SET FOREIGN_KEY_CHECKS = 1;

-- Dump completed on '.date('Y-m-d G-i-s').'
';
			fwrite($handle, $sqlfooter);
			fclose($handle);
			if (file_exists($bkpfile))	$moved	= dol_copy($bkpfile, DOL_DATA_ROOT.($conf->entity != 1 ? '/'.$conf->entity : '').'/admin/'.$appliname.'_update.'.$conf->entity);
			return 1;
		}	// if (file_exists($path))
	}	// function infraspackplus_bkup_module ($appliname, $origine = 1)

	/************************************************
	*	Recherche d'un fichier contenant un code langue dans son nom à partir d'une liste
	*
	*	@param	string	$table		table name to backup
	*	@param	string	$sql		sql query to prepare data  for backup
	*	@param	array	$listeCols	list of columns to backup on the table
	*	@param	array	$duplicate	values for 'ON DUPLICATE KEY UPDATE'
	*									[0] = column to update
	*									[1] = column name to update
	*									[2] = key value for conflict control (only postgreSQL)
	*	@param	boolean	$truncate	truncate the table before restore
	*	@param	string	$add		sql data to add on the beginning of the query
	*	@return	string				sql query to restore the datas
	************************************************/
	function infraspackplus_bkup_table ($table, $sql, $listeCols, $duplicate = array (), $truncate = 0, $add = '')
	{
		global $db, $conf, $langs, $errormsg;
		
		$sqlnewtable	= '';
		$result_sql		= $sql ? $db->query($sql) : '';
		dol_syslog('infraspackplus.Lib::infraspackplus_bkup_table sql = '.$sql);
		if ($result_sql)
		{
			$truncate		= $truncate ? 'TRUNCATE TABLE '.MAIN_DB_PREFIX.$table.';
' : '';
			$sqlnewtable	= '
-- Dumping data for table '.MAIN_DB_PREFIX.$table.'
'.$truncate.$add;
			while($row	= $db->fetch_row($result_sql))
			{
				// For each row of data we print a line of INSERT
				$colsInsert						= '';
				foreach ($listeCols as $col)	$colsInsert	.= $col.', ';
				$sqlnewtable					.= 'INSERT INTO '.MAIN_DB_PREFIX.$table.' ('.substr($colsInsert, 0, -2).') VALUES (';
				$columns						= count($row);
				$duplicateValue					= '';
				for($j = 0; $j < $columns; $j++)
				{
					// Processing each columns of the row to ensure that we correctly save the value (eg: add quotes for string - in fact we add quotes for everything, it's easier)
					if ($row[$j] == null && !is_string($row[$j]))	$row[$j]	= 'NULL';	// IMPORTANT: if the field is NULL we set it NULL
					elseif(is_string($row[$j]) && $row[$j] == '')	$row[$j]	= '\'\'';	// if it's an empty string, we set it as an empty string
					else																	// else for all other cases we escape the value and put quotes around
					{
						$row[$j]	= addslashes($row[$j]);
						$row[$j]	= preg_replace('#\n#', '\\n', $row[$j]);
						$row[$j]	= '\''.$row[$j].'\'';
					}	// else	// elseif(is_string($row[$j]) && $row[$j] == '')	// if ($row[$j] == null && !is_string($row[$j]))
					if ($j == 1)	$row[$j]	= '\'__ENTITY__\'';
					$onDuplicate	= $db->type == 'pgsql' ? ' ON CONFLICT ('.$duplicate[2].') DO UPDATE SET ' : ' ON DUPLICATE KEY UPDATE ';
					$duplicateValue .= $j == $duplicate[0] ? $onDuplicate.$duplicate[1].' = '.$row[$j] : '';
				}	// for($j = 0; $j < $columns; $j++)
				$sqlnewtable	.= implode(', ', $row).')'.$duplicateValue.';
';
			}	// while($row = $db->fetch_row($result_dict))
		}	// if ($result_sql)
		return $sqlnewtable;
	}	// function infraspackplus_bkup_table ($table, $sql, $listeCols, $duplicate = array (), $truncate = 0, $add = '')
	
	/************************************************
	*	Restaure les paramètres du module
	*
	*	@param		string		$appliname	module name
	*	@return		string		1 = Ok or -1 = Ko
	************************************************/
	function infraspackplus_restore_module ($appliname)
	{
		include_once DOL_DOCUMENT_ROOT.'/core/lib/admin.lib.php';
		global $conf, $db;
		
		$pathsql	= dol_buildpath($appliname, 0).'/sql';
		$handle		= @opendir($pathsql);
		if (is_resource($handle))
		{
			$filesql						= $pathsql.'/'.'update.'.$conf->entity;
			$moved							= dol_copy($filesql, $filesql.'.sql');
			if (is_file($filesql.'.sql'))	$result	= run_sql($filesql.'.sql', (empty($conf->global->MAIN_DISPLAY_SQL_INSTALL_LOG) ? 1 : 0), '', 1);
			$delete							= dol_delete_file($filesql.'.sql');
			dol_syslog('infraspackplus.Lib::infraspackplus_restore_module appliname = '.$appliname.' filesql = '.$filesql.' moved = '.$moved.' result = '.$result.' delete = '.$delete);
			if ($result > 0)				return 1;
		}	// if (is_resource($handle))
		return -1;
	}	// function infraspackplus_restore_module ($appliname)

	/************************************************
	*	Liste de fichier de conditions générales suivant le type recherché
	*
	*	@param	string	$type	type of file (CGV, CGI or CGA)
	*	@return	array			array of file name found for the type wanted
	************************************************/
	function infraspackplus_get_CGfiles ($type)
	{
		global $conf, $db;
		
		$CGs																		= array();
		if (glob($conf->mycompany->dir_output.'/'.$type.'_*.pdf'))
			foreach (glob($conf->mycompany->dir_output.'/'.$type.'_*.pdf') as $file)	$CGs[]	= dol_basename($file);
		return $CGs;
	}	// function infraspackplus_get_CGfiles ($type)
	
	/************************************************
	*	Recherche d'un fichier contenant un code langue dans son nom à partir d'une liste
	*
	*	@param	array	$CGs		list of file name to check
	*	@param	string	$searchLang	langage code search
	*	@return	string				file name found 
	************************************************/
	function infraspackplus_get_CGfiles_lang ($CGs, $searchLang)
	{
		for ($i = 0; $i < count($CGs); $i++)
		{
			$langCG						= explode('.', $CGs[$i]);
			$langCG						= $langCG[count($langCG) - 2];
			if ($langCG == $searchLang)	return $CGs[$i];
		}	// for ($i = 0; $i < count($CGs); $i++)
		return '';
	}	// function infraspackplus_get_CGfiles_lang ($CGs, $searchLang)

	/************************************************
	*	Conversion de fichier ttf en police TCPDF.
	*
	*	@param	String		$type			Font type. Leave empty for autodetect mode.
	*										Valid values are:
	*											TrueTypeUnicode
	*											TrueType
	*											Type1
	*											CID0JP	= CID-0	Japanese
	*											CID0KR	= CID-0	Korean
	*											CID0CS	= CID-0	Chinese Simplified
	*											CID0CT	= CID-0	Chinese Traditional
	*	@param  String		$enc     		Name of the encoding table to use.
	*										Leave empty for default mode. Omit this parameter for TrueType Unicode and symbolic font like Symbol or ZapfDingBats.
	*	@param  int	    	$flags   		Unsigned 32-bit integer containing flags specifying various characteristics of the font
	*										(PDF32000:2008 - 9.8.2 Font Descriptor Flags):
	*											+1 for fixed font;
	*											+4 for symbol;
	*											+32 for non-symbol;
	*											+64 for italic.
	*											Fixed and Italic mode are generally autodetected so you have to set it to 32 = non-symbolic font (default) or 4 = symbolic font.
	*	@param  String		$outpath		Output path for generated font files (must be writeable by the web server). Leave empty for default font folder.
	* 	@param	String		$platid			Platform ID for CMAP table to extract
	*										(when building a Unicode font for Windows this value should be 3, for Macintosh should be 1).
	* 	@param	Societe		$encid			Encoding ID for CMAP table to extract
	*										(when building a Unicode font for Windows this value should be 1, for Macintosh should be 0).
	*										When Platform ID is 3, legal values for Encoding ID are:
	*											0	= Symbol
	*											1	= Unicode
	*											2	= ShiftJIS
	*											3	= PRC
	*											4	= Big5
	*											5	= Wansung
	*											6	= Johab
	*											7	= Reserved
	*											8	= Reserved
	*											9	= Reserved
	*											10	= UCS-4
	* 	@param	Booleen		$addcbbox		Includes the character bounding box information on the php font file.
	* 	@param	Booleen		$link			Link to system font instead of copying the font data # (not transportable) - Note: do not work with Type1 font.
	* 	@param	String		$font			input font file.
	*	@return	String						Return Font name or false if error.
	************************************************/
	function infraspackplus_Add_TCPDF_Font($type = '', $enc = '', $flags = 32, $outpath, $platid = 3, $encid = 1, $addcbbox = false, $link = false, $font)
	{
		require_once TCPDF_PATH.'tcpdf.php';
		
		$options											= array();
		$typefont											= array('TrueTypeUnicode', 'TrueType', 'Type1', 'CID0JP', 'CID0KR', 'CID0CS', 'CID0CT');
		if (in_array($type, $typefont))	$options['type']	= $type;
		else												$options['type']	= '';
		$options['enc']										= $enc;
		$options['flags']									= intval($flags);
		$options['outpath']									= realpath($outpath);
		if (substr($options['outpath'], -1) != '/')			$options['outpath']	.= '/';
		$options['platid']									= min(max(1, intval($platid)), 3);
		$options['encid']									= min(max(0, intval($encid)), 10);
		$options['addcbbox']								= $addcbbox;
		$options['link']									= $link;
		$fontfile											= realpath($font);
		$fontname											= TCPDF_FONTS::addTTFfont($fontfile, $options['type'], $options['enc'], $options['flags'], $options['outpath'], $options['platid'], $options['encid'], $options['addcbbox'], $options['link']);
		return $fontname;
	}	// function infraspackplus_Add_TCPDF_Font($type = '', $enc = '', $flags = 32, $outpath, $platid = 3, $encid = 1, $addcbbox = false, $link = false, $font)

	/************************************************
	* Function called to check Logo files associate to customer
	*
	* @param	string	$socid	societe Id to check
	* @return	string			logo file name
	************************************************/
	function infraspackplus_getLogoEmet($socid)
	{
		global $conf, $db;
		
		$logo_emet			= '';
		$sql_logo_emet		= 'SELECT s.logo_emet';
		$sql_logo_emet		.= ' FROM '.MAIN_DB_PREFIX.'societe AS s';
		$sql_logo_emet		.= ' WHERE s.rowid = '.$socid;
		$result_logo_emet	= $db->query($sql_logo_emet);
		if ($result_logo_emet)
		{
			$obj_logo_emet	= $db->fetch_object($result_logo_emet);
			$logo_emet		= $obj_logo_emet->logo_emet;
		}	// if ($result_logo_emet)
		$db->free($result_logo_emet);
		return $logo_emet;
	}	// function infraspackplus_getLogoEmet($socid)

	/************************************************
	* Function called to update Logo files associate to customer
	*
	* @param	string	$socid	societe Id to update
	* @param	string	$logo	File name to update
	* @return	string			0 if OK SQL error else
	************************************************/
	function infraspackplus_setLogoEmet($socid, $logo)
	{
		global $conf, $db;
		
		$sql_upt			= 'UPDATE '.MAIN_DB_PREFIX.'societe';
		$sql_upt			.= ' SET logo_emet = "'.$logo.'"';
		$sql_upt			.= ' WHERE rowid = '.$socid;
		$result_upt			= $db->query($sql_upt);
		if ($result_upt)
		{
			$db->free($result_upt);
			return 0;
		}	// if ($result_upt)
		else	return $db->error().' sql = '.$sql_upt;
	}	// function infraspackplus_setLogoEmet($socid, $logo)

	/************************************************
	 * Function called to get downloaded changelog and compare with the local one
	 * Presentation of results on a HTML table
	 *
	 * @param   string	$appliname		module name
	 * @param   string	$resVersion		flag for error (-1 = KO ; 0 = OK)
	 * @param   string	$tblversions	array => versions list or errors list
	 * @param	int		$dwn			flag to show download button (0 = hide it ; 1 = show it)
	 * @return	string					HTML presentation
	************************************************/
	function infraspackplus_getChangeLog($appliname, $resVersion, $tblversions, $dwn = 0)
	{
		global $langs;
				
		$headerPath				= dol_buildpath('/'.$appliname.'/img/InfraSheader.png', 1);
		$logoPath				= dol_buildpath('/'.$appliname.'/img/InfraS.gif', 1);
		$gplv3Path				= dol_buildpath('/'.$appliname.'/img/gplv3.png', 1);
		$listUpD				= dol_buildpath('/'.$appliname.'/img/list_updates.png', 1);
		$urlInfraS				= 'https://www.infras.fr';
		$urldownl				= '/index.php?option=com_content&view=featured&Itemid=161';
		$urldocs				= '/index.php?option=com_jdownloads&view=category&catid=11&Itemid=116';
		$urlDoli				= 'http://www.dolistore.com/search.php?search_query=InfraS';
		$style					= 'color: white; font-size: 16px; font-weight: bold; width: 30%;';
		$InputCarac				= 'class = "butActionChangelog" name = "readmore" type = "button"';
		$ret					= array();
		$ret					= $langs->trans("InfraSParamPresent").'
									<br />
									<table  width = "100%" cellspacing = "10" style = "background: url('.$headerPath.'); background-size: cover;">
										<tr>
											<td rowspan = "3" align = "left" valign = "bottom" style="'.$style.'">
												<a href = "'.$urlInfraS.'" target = "_blank"><img border = "0" width = "220" src = "'.$logoPath.'"></a>
												<br/>&nbsp;&nbsp;'.$langs->trans("InfraSParamSlogan").'
											</td>
											<td align = "center" valign = "middle" width = "30%">
												<a href = "'.$urlInfraS.$urldownl.'" target = "_blank"><input '.$InputCarac.' value = "'.$langs->trans("InfraSParamLienModules").'" /></a>
											</td>
											<td rowspan = "3" align = "center" valign = "middle" width = "30%">
												<a href = "'.$urlDoli.'" target = "_blank"><img border = "0" width = "180" src = "'.DOL_URL_ROOT.'/theme/dolistore_logo.png"></a>
												<br />'.$langs->trans("InfraSParamMoreModulesLink").'
											</td>
										</tr>
										<tr>
											<td align = "center" valign = "middle">
												<a href = "'.$urlInfraS.$urldocs.'" target = "_blank"><input '.$InputCarac.' value = "'.$langs->trans("InfraSParamLienDocs").'" /></a>
											</td>
										</tr>
										<tr>
											<td align = "center" valign = "middle">
												<img border="0" width="120" src="'.$gplv3Path.'"/>
												<br />'.$langs->trans("InfraSParamLicense").'
											</td>
										</tr>
									</table>';
		$ret					.= load_fiche_titre($langs->trans("InfraSParamHistoryUpdates"), '', $listUpD, 1);
		$sxe					= infraspackplus_getChangelogFile($appliname);
		$sxelast				= infraspackplus_getChangelogFile($appliname, 'dwn');
		if ($sxelast === false)	$tblversionslast	= array();
		else					$tblversionslast	= $sxelast->Version;
		if ($resVersion == -1)
		{
			foreach ($tblversions as $error)	$ret	.= $error->message;
			return $ret;
		}	// if ($resVersion == -1)
		if ($conf->global->INFRAS_SKIP_CHECKVERSION == 1)	$dwnbutton	= $dwn ? $langs->trans("InfraSParamSkipCheck") : '';
		else												$dwnbutton	= $dwn ? '<button class = "button" style = "width: 190px; padding: 3px 0px;" type = "submit" value = "dwnChangelog" name = "action" title = "'.$langs->trans("InfraSParamCheckNewVersionTitle").'">'.$langs->trans("InfraSParamCheckNewVersion").'</button>' : '';
		$ret	.= '<table class = "noborder" >
						<tr class = "liste_titre">
							<th align = center width = 100px>'.$langs->trans("InfraSParamNumberVersion").'</th>
							<th align = center width = 100px>'.$langs->trans("InfraSParamMonthVersion").'</th>
							<th align = left >'.$langs->trans("InfraSParamChangesVersion").'</th>
							<th align = center width = "200px" >'.$dwnbutton.'</th>
						</tr>';
		if ($sxe !== false && count($tblversionslast) > count($tblversions))	// il y a du nouveau
		{
			for ($i = count($tblversionslast)-1; $i >=0; $i--)
			{
				$color					= "";
				$sxePath				= $sxe->xpath('//Version[@Number="'.$tblversionslast[$i]->attributes()->Number.'"]');
				dol_syslog('infraspackplus.Lib::infraspackplus_getChangeLog sxePath = '.$sxePath);
				if (empty($sxePath))	$color='bgcolor = orange';
				$lineversion			= $tblversionslast[$i]->change;
				$ret	.= '<tr class = "oddeven">
								<td align = center '.$color.' valign = top>'.$tblversionslast[$i]->attributes()->Number.'</td>
								<td align = center '.$color.' valign = top>'.$tblversionslast[$i]->attributes()->MonthVersion.'</td>
								<td align = left colspan = "2" '.$color.' valign = top style = "padding-top: 0; padding-bottom: 0;">';
				foreach ($lineversion as $changeline)
				{
					if ($changeline->attributes()->type == "fix")		$stylecolor	= ' color: red;';
					else if ($changeline->attributes()->type == "add")	$stylecolor	= ' color: green;';
					else if ($changeline->attributes()->type == "chg")	$stylecolor	= ' color: blue;';
					else												$stylecolor	= ' color: black;';
					$ret	.= '	<table>
										<tr>
											<td width = 50px style = "border: none; padding-top: 0; padding-bottom: 0;'.$stylecolor.'">'.$changeline->attributes()->type.'</td>
											<td style = "border: none; padding-top: 0; padding-bottom: 0;'.$stylecolor.'">'.$changeline.'</td>
										</tr>
									</table>';
				}	// foreach ($lineversion as $changeline)
				$ret	.= '	</td>
							</tr>';
			}	// for ($i = count($tblversionslast)-1; $i >=0; $i--)
		}	// if (count($tblversionslast) > count($tblversions))
		elseif ($sxelast !== false && count($tblversionslast) < count($tblversions) && count($tblversionslast) > 0)
		{
			for ($i = count($tblversions)-1; $i >=0; $i--)
			{
				$color						= "";
				$sxelastPath				= $sxelast->xpath('//Version[@Number="'.$tblversions[$i]->attributes()->Number.'"]');
				if (empty($sxelastPath))	$color='bgcolor = lightgreen';
				$lineversion				= $tblversions[$i]->change;
				$ret	.= '<tr class = "oddeven">
								<td align = center '.$color.' valign = top>'.$tblversions[$i]->attributes()->Number.'</td>
								<td align = center '.$color.' valign = top>'.$tblversions[$i]->attributes()->MonthVersion.'</td>
								<td align = left colspan = "2" '.$color.' valign = top style = "padding-top: 0; padding-bottom: 0;">';
				foreach ($lineversion as $changeline)
				{
					if ($changeline->attributes()->type == "fix")		$stylecolor	= ' color: red;';
					else if ($changeline->attributes()->type == "add")	$stylecolor	= ' color: green;';
					else if ($changeline->attributes()->type == "chg")	$stylecolor	= ' color: blue;';
					else												$stylecolor	= ' color: black;';
					$ret	.= '	<table>
										<tr>
											<td width = 50px style = "border: none; padding-top: 0; padding-bottom: 0;'.$stylecolor.'">'.$changeline->attributes()->type.'</td>
											<td style = "border: none; padding-top: 0; padding-bottom: 0;'.$stylecolor.'">'.$changeline.'</td>
										</tr>
									</table>';
				}	// foreach ($lineversion as $changeline)
				$ret	.= '	</td>
							</tr>';
			}	// for ($i = count($tblversions)-1; $i >=0; $i--)
		}	// elseif ($sxelast !== false && count($tblversionslast) < count($tblversions) && count($tblversionslast) > 0)
		else	//on est à jour des versions ou pas de connection internet 
		{
			for ($i = count($tblversions)-1; $i >=0; $i--)
			{
				$lineversion	= $tblversions[$i]->change;
				$color			= "";
				$ret	.= '<tr class="oddeven">
								<td align = center valign = top>'.$tblversions[$i]->attributes()->Number.'</td>
								<td align = center valign = top>'.$tblversions[$i]->attributes()->MonthVersion.'</td>
								<td align = left colspan = "2" '.$color.' valign = top style = "padding-top: 0; padding-bottom: 0;">';
				foreach ($lineversion as $changeline)
				{
					if ($changeline->attributes()->type == "fix")		$stylecolor	= ' color: red;';
					else if ($changeline->attributes()->type == "add")	$stylecolor	= ' color: green;';
					else if ($changeline->attributes()->type == "chg")	$stylecolor	= ' color: blue;';
					else												$stylecolor	= ' color: black;';
					$ret	.= '	<table>
										<tr>
											<td width = 50px style = "border: none; padding-top: 0; padding-bottom: 0;'.$stylecolor.'">'.$changeline->attributes()->type.'</td>
											<td style = "border: none; padding-top: 0; padding-bottom: 0;'.$stylecolor.'">'.$changeline.'</td>
										</tr>
									</table>';
				}	// foreach ($lineversion as $changeline)
				$ret	.= '	</td>
							</tr>';
			}	// for ($i = count($tblversions)-1; $i >=0; $i--)
		}	// else	// elseif	// if (count($tblversionslast) > count($tblversions))
		$ret	.= '	</table>';
		return $ret;
	}	// function infraspackplus_getChangeLog($appliname)

	/************************************************
	 * Function called to get support information
	 * Presentation of results on a HTML table
	 *
	 * @param   string	$currentversion	current version from changelog
	 * @return	string					HTML presentation
	************************************************/
	function infraspackplus_getSupportInformation($currentversion)
	{
		global $db, $langs;
		
		$ret	.= '<table class="noborder" >
						<tr class="liste_titre">
							<th align = center width=200px>'.$langs->trans("InfraSSupportInformation").'</th>
							<th align = center>'.$langs->trans("Value").'</th>
						</tr>
						<tr class="oddeven">
							<td width = 20opx style = "border: none; padding-top: 0; padding-bottom: 0;">'.$langs->trans("DolibarrVersion").'</td>
							<td style = "border: none; padding-top: 0; padding-bottom: 0;">'.DOL_VERSION.'</td>
						</tr>
						<tr class="oddeven">
							<td width = 20opx style = "border: none; padding-top: 0; padding-bottom: 0;">'.$langs->trans("ModuleVersion").'</td>
							<td style = "border: none; padding-top: 0; padding-bottom: 0;">'.$currentversion.'</td>
						</tr>
						<tr class="oddeven">
							<td width = 20opx style = "border: none; padding-top: 0; padding-bottom: 0;">'.$langs->trans("PHPVersion").'</td>
							<td style = "border: none; padding-top: 0; padding-bottom: 0;">'.version_php().'</td>
						</tr>
						<tr class="oddeven">
							<td width = 20opx style = "border: none; padding-top: 0; padding-bottom: 0;">'.$langs->trans("DatabaseVersion").'</td>
							<td style = "border: none; padding-top: 0; padding-bottom: 0;">'.$db::LABEL." ".$db->getVersion().'</td>
						</tr>
						<tr class="oddeven">
							<td width = 20opx style = "border: none; padding-top: 0; padding-bottom: 0;">'.$langs->trans("WebServerVersion").'</td>
							<td style = "border: none; padding-top: 0; padding-bottom: 0;">'.$_SERVER["SERVER_SOFTWARE"].'</td>
						</tr>
						<tr><td colspan = "3" style = "line-height: 1px;">&nbsp;</td></tr>
					</table>
					<br />';
		return $ret;
	}	// function infraspackplus_getSupportInformation($currentversion)

		/************************************************
		 * Function called to check the available version by downloading the last changelog file
		 * Check if the last changelog downloaded is less than 7 days if we do not do anything
		 * 
		 * @return		string		current version with information about new ones on tooltip or error message
		************************************************/
		function infraspackplus_dwnChangelog($appliname)
		{
			global $langs, $conf;
			
			$path												= dol_buildpath($appliname, 0);
			if ($conf->global->INFRAS_PHP_EXT_XML == -1)		return -1;
			$sxelasthtmlversion									= infraspackplus_getChangelogFile($appliname, 'dwn');
			if ($sxelasthtmlversion === false)	$lasthtmldate	= '19000101';
			else								$lasthtmldate	= $sxelasthtmlversion->InfraS->attributes()->Downloaded;
			if ($lasthtmldate < date('Ymd', strtotime('-7 day')))
			{
				$context						= stream_context_create(array('http' => array('header' => 'Accept: application/xml'), 'ssl' => array('verify_peer' => false, 'verify_peer_name' => false)));
				$newhtmlversion					= @file_get_contents('https://www.infras.fr/jdownloads/Technique/Modules%20Dolibarr/Changelogs/'.$appliname.'/changelog.xml', false, $context);
				if ($newhtmlversion === false)	return -1;	// not connected
				else
				{
					$newhtmlversion		= preg_replace('#Downloaded=\".+\"#', 'Downloaded="'.date('Ymd').'"', $newhtmlversion);
					file_put_contents($path.'/docs/changelogdwn.xml', $newhtmlversion);
				}	// else	// if ($newhtmlversion === false)
			}	// if ($lasthtmldate < date('Ymd', strtotime('-7 day')))
			return 1;
		}	// function infraspackplus_dwnChangelog($appliname)

	/************************************************
	*	Return list of mention
	*
	*	@param	string	$selected		Preselected type
	*	@param  string	$htmlname		Name of field in html form
	* 	@param	int		$showempty		Add an empty field
	*	@param  string	$onChange		JavaScript function for onchange event
	*	@return	string					Select html tag with all mention labels found
	************************************************/
	function select_infraspackplus_dict($dict, $selected = '', $htmlname = 'fk_infraspackplus_dict', $showempty = 0, $onChange = '')
	{
		global $db, $conf, $langs;

		$typeDict	= ucfirst(explode('_', $dict)[2]);
		$result		= '';
		$sql		= 'SELECT rowid, code, libelle';
		$sql		.= ' FROM '.MAIN_DB_PREFIX.$dict;
		$sql		.= ' WHERE active = 1 AND entity = "'.$conf->entity.'"';
		$sql		.= ' ORDER BY pos ASC';

		$resql		= $db->query($sql);
		dol_syslog('infraspackplus.Lib::select_infraspackplus_dict sql = '.$sql);
		if ($resql)
		{
			$num	= $db->num_rows($resql);
			$i		= 0;
			if ($num)
			{
				$result	.= '&nbsp;'.$langs->trans("InfraSPlusParam".$typeDict."3").'&nbsp;';
				$result	.= '<select class = "flat" name="'.$htmlname.'"'.($onChange ? 'onchange = "'.$onChange.';"' : '').'>';
				if ($showempty)
				{
					$result					.= '<option value = "-1"';
					if ($selected == -1)	$result	.= ' selected = "selected"';
					$result					.= '>&nbsp;</option>';
				}	// if ($showempty)
				while ($i < $num)
				{
					$obj							= $db->fetch_object($resql);
					$libelle						= ($langs->trans("InfraSPlusDict".$typeDict."s".$obj->code) != ("InfraSPlusDict".$typeDict."s".$obj->code) ? $langs->trans("InfraSPlusDict".$typeDict."s".$obj->code) : ($obj->libelle != '-' ? $obj->libelle : ''));
					$result							.= '<option value = "'.$obj->code.'"';
					if ($obj->code == $selected)	$result	.= ' selected';
					$result							.= '>'.$libelle.'</option>';
					$i++;
				}	// while ($i < $num)
				$result	.= '</select>';
			}	// if ($num)
			else	$result	.= '<input type = "hidden" name = "'.$htmlname.' id="'.$htmlname.'" value = -1>';	// si pas de liste, on positionne un hidden � vide
		}	// if ($resql)
		else	$result	.= '<input type = "hidden" name = "'.$htmlname.' id="'.$htmlname.'" value = -1>';	// si pas de liste, on positionne un hidden � vide
		return	$result;
	}	// function select_infraspackplus_dict($dict, $selected = '', $htmlname = 'fk_infraspackplus_mention', $showempty = 0, $hidetext = 0)

	/************************************************
	*	Changes permissions on files and directories within $dir and dives recursively into found subdirectories.
	*
	*	@param	string	$dir		Preselected type
	*	@param  string	$filePerm	The permissions any found files should get.
	* 	@param	int		$dirPerm	The permissions any found folder should get.
	* 	@param	string	$own		The owner any found file or folder should get.
	* 	@param	string	$grp		The group any found file or folder should get.
	*	@return	boolean				 TRUE if the path if found and FALSE if not.
	************************************************/
	function infraspackplus_chmod_r($path, $filePerm = '0644', $dirPerm = '0755', $own = 'www-data', $grp = 'www-data')
	{
		dol_syslog('infraspackplus.Lib::infraspackplus_chmod_r path = '.$path.' filePerm = '.$filePerm.' dirPerm = '.$dirPerm.' own = '.$own.' grp = '.$grp);
		if (!file_exists($path))	return(false);	// Check if the path exists
        // See whether this is a file
		if (is_file($path))
		{
			dol_syslog('infraspackplus.Lib::infraspackplus_chmod_r path = '.$path);
			@chmod($path, octdec($filePerm));	// Chmod the file with our given filepermissions
			@chgrp($path, $grp);				// Chgrp the file with our given group
			@chown($path, $own);				// Chown the file with our given owner
		}	// if (is_file($path))
		elseif (is_dir($path))	// If this is a directory...
		{
			$foldersAndFiles				= scandir($path);	// Then get an array of the contents
			$entries						= array_slice($foldersAndFiles, 2);	// Remove "." and ".." from the list
			foreach ($entries as $entry)	infraspackplus_chmod_r($path.'/'.$entry, $filePerm, $dirPerm, $own, $grp);	// Parse every result... And call this function again recursively, with the same permissions
			dol_syslog('infraspackplus.Lib::infraspackplus_chmod_r path = '.$path);
			@chmod($path, octdec($dirPerm));	// When we are done with the contents of the directory, we chmod the directory itself
			@chgrp($path, $grp);				// When we are done with the contents of the directory, we chgrp the directory itself
			@chown($path, $own);				// When we are done with the contents of the directory, we chown the directory itself
		}	// elseif (is_dir($path))
		return(true);	// Everything seemed to work out well, return true
	}	// function infraspackplus_chmod_r($path, $filePerm = '0644', $dirPerm = '0755', $own = 'www-data', $grp = 'www-data')
	
	/************************************************
	*	Converts shorthand memory notation value to bytes
	*	From http://php.net/manual/en/function.ini-get.php
	*
	*	@param  string	$val	Memory size shorthand notation
	*	@return	string			value .
	************************************************/
	function infraspackplus_return_bytes($val)
	{
		$val	= trim($val);
		$last	= strtolower($val[strlen($val)-1]);
		switch($last)
		{
			case 'g':
				$val *= 1024;
			case 'm':
				$val *= 1024;
			case 'k':
				$val *= 1024;
		}	// switch($last)
		return $val;
	}	// function infraspackplus_return_bytes($val)
	
	/************************************************
	*	Create a new PDF to show what the chosen font looks like
	*
	*	@return		string		1 = Ok or 0 = Ko
	************************************************/
	function infraspackplus_test_font()
	{
		global $db, $conf, $langs;
		require_once DOL_DOCUMENT_ROOT.'/core/lib/pdf.lib.php';
		dol_include_once('/infraspackplus/core/lib/infraspackplus.pdf.lib.php');
		$formatarray		= pdf_InfraSPlus_getFormat();
		$page_largeur		= $formatarray['width'];
		$page_hauteur		= $formatarray['height'];
		$format				= array($page_largeur, $page_hauteur);
		$main_umask			= isset($conf->global->MAIN_UMASK)			? $conf->global->MAIN_UMASK				: '0755';
		$font				= isset($conf->global->INFRASPLUS_PDF_FONT) ? $conf->global->INFRASPLUS_PDF_FONT	: 'Helvetica';
		$dir				= $conf->ecm->dir_output.'/temp/';
		$file				= $dir.'TEST.pdf';
		if (! file_exists($dir))
		{
			if (dol_mkdir($dir) < 0)
			{
				setEventMessages($langs->trans("ErrorCanNotCreateDir", $dir), null, 'errors');
				return 0;
			}	// if (dol_mkdir($dir) < 0)
		}	// if (! file_exists($dir))
		if (file_exists($dir))
		{
			// Create pdf instance
			$pdf				= pdf_getInstance($format, 'mm', 'L');
			$default_font_size	= pdf_getPDFFontSize($langs);																								// Must be after pdf_getInstance
			$pdf->SetAutoPageBreak(1, 0);
			if (class_exists('TCPDF'))
			{
				$pdf->setPrintHeader(false);
				$pdf->setPrintFooter(false);
			}	// if (class_exists('TCPDF'))
			$pdf->SetFont($font, '', 14);	// set font
			$tagvs						= array('p' => array(1 => array('h' => 0.0001, 'n' => 1)), 'ul' => array(0 => array('h' => 0.0001, 'n' => 1)));
			$pdf->setHtmlVSpace($tagvs);
			$pdf->Open();
			// set document information
			$pdf->SetTitle('InfraSPackPlus test font');
			$pdf->SetSubject('InfraSPackPlus');
			$pdf->SetCreator('Dolibarr '.DOL_VERSION);
			$pdf->SetAuthor('InfraS - Sylvain Legrand');
			$pdf->SetKeywords('InfraS, InfraSPack, InfraSPackPlus, PDF, example, test, guide');
			$pdf->SetMargins(10, 10, 10);   // Left, Top, Right
			$pdf->AddPage();	// add a page
			$txt						= "Font : ".$font."<br />Test :<ul><li>Normal<ul><li>&nbsp;&nbsp;a b c d e f g h i j k l m n o p q r s t u v w x y z A B C D E F G H I J K L M N O P Q R S T U V W X Y Z</li><li>&nbsp;&nbsp;0 1 2 3 4 5 6 7 8 9 + - * = ° ² é è à ù ç â ê î ô û ä ë ï ö ü , ; : ! ? . & § % µ @ $ £ € ¤ # | ( ) { } [ ] < > _ ~</li></ul></li><li><b>Gras</b><ul><li>&nbsp;&nbsp;<b>a b c d e f g h i j k l m n o p q r s t u v w x y z A B C D E F G H I J K L M N O P Q R S T U V W X Y Z</b></li><li>&nbsp;&nbsp;<b>0 1 2 3 4 5 6 7 8 9 + - * = ° ² é è à ù ç â ê î ô û ä ë ï ö ü , ; : ! ? . & § % µ @ $ £ € ¤ # | ( ) { } [ ] < > _ ~</b></li></ul></li><li><em>Italique<em><ul><li>&nbsp;&nbsp;<em>a b c d e f g h i j k l m n o p q r s t u v w x y z A B C D E F G H I J K L M N O P Q R S T U V W X Y Z</em></li><li>&nbsp;&nbsp;<em>0 1 2 3 4 5 6 7 8 9 + - * = ° ² é è à ù ç â ê î ô û ä ë ï ö ü , ; : ! ? . & § % µ @ $ £ € ¤ # | ( ) { } [ ] < > _ ~</em></li></ul></li><li><b>Gras Italique</b><ul><li>&nbsp;&nbsp;<em><b>a b c d e f g h i j k l m n o p q r s t u v w x y z A B C D E F G H I J K L M N O P Q R S T U V W X Y Z</b></em></li><li>&nbsp;&nbsp;<em><b>0 1 2 3 4 5 6 7 8 9 + - * = ° ² é è à ù ç â ê î ô û ä ë ï ö ü , ; : ! ? . & § % µ @ $ £ € ¤ # | ( ) { } [ ] < > _ ~</b></em></li></ul></li></ul>";	// set some text to print
			$pdf->writeHTMLCell($page_hauteur - 20, $page_largeur - 20, 10, 10, dol_htmlentitiesbr($txt), 0, 1);
			$pdf->Close();
			$pdf->Output($file, 'F');
			if (! empty($main_umask))	@chmod($file, octdec($main_umask));
			return 1;   // Pas d'erreur
		}	// if (file_exists($dir))
		else
		{
			setEventMessages($langs->transnoentities("ErrorCanNotCreateDir", $dir), null, 'errors');
			return 0;
		}	// else	// if (file_exists($dir))
	}	// function infraspackplus_test_font()
	
	/************************************************
	*	Check if the parent company sould be used
	*
	*	@param		Object		$object		Object we want to build document for
	*	@return		Object					Object address found
	************************************************/
	function infraspackplus_check_parent_addr_fact ($object)
	{
		global $db, $conf;
		
		$parent_adrfact	= isset($conf->global->INFRASPLUS_PDF_FACTURE_PARENT_ADDR_FACT)	? $conf->global->INFRASPLUS_PDF_FACTURE_PARENT_ADDR_FACT	: 0;
		if (! empty($parent_adrfact) && ! empty($object->thirdparty->parent))
		{
			require_once DOL_DOCUMENT_ROOT.'/societe/class/societe.class.php';
			$parent	= new Societe($db);
			$parent->fetch($object->thirdparty->parent);
			return $parent;
		}	// if (! empty($parent_adrfact) && ! empty($object->thirdparty->parent))
		else	return $object->thirdparty;
	}	// function infraspackplus_check_parent_addr_fact ($object)
?>