<?php
	/************************************************
	* Copyright (C) 2016-2019	Sylvain Legrand - <contact@infras.fr>	InfraS - <https://www.infras.fr>
	*
	* This program is free software: you can redistribute it and/or modify
	* it under the terms of the GNU General Public License as published by
	* the Free Software Foundation, either version 3 of the License, or
	* (at your option) any later version.
	*
	* This program is distributed in the hope that it will be useful,
	* but WITHOUT ANY WARRANTY; without even the implied warranty of
	* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	* GNU General Public License for more details.
	*
	* You should have received a copy of the GNU General Public License
	* along with this program.  If not, see <http://www.gnu.org/licenses/>.
	************************************************/

	/************************************************
	* 	\file		../infraspackplus/core/lib/infras.pdf.lib.php
	* 	\ingroup	InfraS
	* 	\brief		Set of functions used for InfraS PDF generation
	************************************************/

	/************************************************
	*	Return array with format properties
	*
	*	@param	Translate	$outputlangs	Output lang to use to autodetect output format if setup not done
	*	@param	string		$format			specific format to use
	*	@return	array						Array('width'=>w,'height'=>h,'unit'=>u);
	************************************************/
	function pdf_InfraSPlus_getFormat($format = '', $outputlangs = null)
	{
		global $conf, $db;

		// Default value if setup was not done and/or entry into c_paper_format not defined
		$width	= 210;
		$height	= 297;
		$unit	= 'mm';
		if ($format)
		{
			$sql	= "SELECT code, label, width, height, unit FROM ".MAIN_DB_PREFIX."c_paper_format";
			$sql	.= " WHERE code = '".$format."'";
			$resql	= $db->query($sql);
			if ($resql)
			{
				$obj	= $db->fetch_object($resql);
				if ($obj)
				{
					$width	= (int) $obj->width;
					$height	= (int) $obj->height;
					$unit	= $obj->unit;
					return array('width'=>$width, 'height'=>$height, 'unit'=>$unit);
				}	// if ($obj)
			}	// if ($resql)
			$db->free($resql);
		}	// if ($format)
		if (empty($conf->global->MAIN_PDF_FORMAT))
		{
			include_once DOL_DOCUMENT_ROOT.'/core/lib/functions2.lib.php';
			$pdfformat	= dol_getDefaultFormat($outputlangs);
		}
		else	$pdfformat	= $conf->global->MAIN_PDF_FORMAT;
		$sql	= "SELECT code, label, width, height, unit FROM ".MAIN_DB_PREFIX."c_paper_format";
		$sql	.= " WHERE code = '".$pdfformat."'";
		$resql	= $db->query($sql);
		if ($resql)
		{
			$obj	= $db->fetch_object($resql);
			if ($obj)
			{
				$width	= (int) $obj->width;
				$height	= (int) $obj->height;
				$unit	= $obj->unit;
			}	// if ($obj)
		}	// if ($resql)
		$db->free($resql);
		return array('width'=>$width, 'height'=>$height, 'unit'=>$unit);
	}	// function pdf_InfraSPlus_getFormat($format = '', $outputlangs = null)

	/************************************************
	*	Select text color from background values
	*
	*	@param	string		$bgcolor     		RGB value for background color
	*	@return	int								'255' or '0' for white (255, 255 ,255) or black (0, 0, 0)
	************************************************/
	function pdf_InfraSPlus_txt_color(&$bgcolor)
	{
		require_once DOL_DOCUMENT_ROOT.'/core/lib/functions2.lib.php';
		global $conf;

		$tmppart			= explode(',', $bgcolor);
		$tmpvalr			= (! empty($tmppart[0]) ? $tmppart[0] : 0) * 0.3;
		$tmpvalg			= (! empty($tmppart[1]) ? $tmppart[1] : 0) * 0.59;
		$tmpvalb			= (! empty($tmppart[2]) ? $tmppart[2] : 0) * 0.11;
		$tmpval				= $tmpvalr + $tmpvalg + $tmpvalb;
		if ($tmpval <= 128)	$colorauto	= '255, 255, 255';
		else				$colorauto	= '0, 0, 0';
		$textcolorauto		= isset($conf->global->INFRASPLUS_PDF_TEXT_COLOR_AUTO)	? $conf->global->INFRASPLUS_PDF_TEXT_COLOR_AUTO : 1;
		$colorman			= isset($conf->global->INFRASPLUS_PDF_TEXT_COLOR)		? $conf->global->INFRASPLUS_PDF_TEXT_COLOR : $colorauto;
		if ($textcolorauto)	return $colorauto;
		else				return $colorman;
	}	// function pdf_InfraSPlus_txt_color(&$bgcolor)

	/************************************************
	*	Display a background as a watermark
	*
	*	@param	TCPDF		$pdf            The PDF factory
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	* 	@return	void
	************************************************/
	function pdf_InfraSPlus_bg_watermark($pdf, $formatpage, $test = '')
	{
		global $conf;

		$image_watermark		= isset($conf->global->INFRASPLUS_PDF_IMAGE_WATERMARK)		? $conf->global->INFRASPLUS_PDF_IMAGE_WATERMARK		: '';
		$watermark_i_opacity	= isset($conf->global->INFRASPLUS_PDF_I_WATERMARK_OPACITY)	? $conf->global->INFRASPLUS_PDF_I_WATERMARK_OPACITY	: 1;
		$filigrane	= $conf->mycompany->dir_output.'/logos/'.$image_watermark;
		if ($filigrane && is_readable($filigrane))
		{
			$imgsize	= array();
			$imgsize	= pdf_InfraSPlus_getSizeForImage($filigrane, $formatpage['largeur'], $formatpage['hauteur']);
			if (isset($imgsize['width']) && isset($imgsize['height']))
			{
				$pdf->SetAlpha($watermark_i_opacity / 100);
				$bMargin			= $pdf->getBreakMargin();	// get the current page break margin
				$auto_page_break	= $pdf->getAutoPageBreak();	// get current auto-page-break mode
				$pdf->SetAutoPageBreak(false, 0);	// disable auto-page-break
				$posxpicture		= ($formatpage['largeur'] - $imgsize['width']) / 2;	// centre l'image dans la page
				$posypicture		= ($formatpage['hauteur'] - $imgsize['height']) / 2;	// centre l'image dans la page
				$pdf->Image($filigrane, $posxpicture, $posypicture, $imgsize['width'], $imgsize['height'], '', '', '', false, 300, '', false, false, 0);	// set bacground image
				$pdf->SetAutoPageBreak($auto_page_break, $bMargin);	// restore auto-page-break status
				$pdf->setPageMark();	// set the starting point for the page content
				$pdf->SetAlpha(1);
			}	// if (isset($imgsize['width']) && isset($imgsize['height']))
		}	// if ($filigrane && is_readable($filigrane))
	}	// function pdf_InfraSPlus_bg_watermark($pdf, $formatpage)

	/********************************************
	*	Calcul for total discount
	*
	*	@param		Object		$object			Object shown in PDF
	*	@param		int			$i				Row number
	*	@param		boolean		$only_ht		don't include taxes
	*	@return		String						Return the difference between standart price and discounted one
	********************************************/
	function pdf_InfraSPlus_getTotRem($object, $i, $only_ht = 0)
	{
		global $conf;

		if ($conf->multicurrency->enabled && $object->multicurrency_tx != 1)	$TotBrutLine	= $object->lines[$i]->multicurrency_subprice * $object->lines[$i]->qty;
		else																	$TotBrutLine	= $object->lines[$i]->subprice * $object->lines[$i]->qty;
		if (!$only_ht)
		{
			$tvalignebrut		= $TotBrutLine * $object->lines[$i]->tva_tx / 100;
			$localtax1lignebrut	= $TotBrutLine * $object->lines[$i]->localtax1_tx / 100;
			$localtax2lignebrut	= $TotBrutLine * $object->lines[$i]->localtax2_tx / 100;
			return ($TotBrutLine + $tvalignebrut + $localtax1lignebrut + $localtax2lignebrut) - $object->lines[$i]->total_ttc;
		}	// if (!$only_ht)
		else	return $TotBrutLine - $object->lines[$i]->total_ht;
	}	// function pdf_InfraSPlus_getTotRem($object, $i, $only_ht = 0)

	/************************************************
	*	Show top small header of page.
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Translate	$outputlangs	Object lang for output
	*	@param	float		$posy			Position depart (hauteur)
	*	@param	float		$w				Largeur 1ère colonne
	* 	@param	boolean		$logo			Name of logo file
	* 	@param	object		$emetteur
	* 	@param	int			$marge_gauche
	* 	@param	int			$tab_hl
	* 	@param	array		$headertxtcolor	Text color
	* 	@return	float		Return height of logo
	************************************************/
	function pdf_InfraSPlus_logo($pdf, $outputlangs, $posy, $w, $logo, $emetteur, $marge_gauche, $tab_hl, $headertxtcolor)
	{
		global $conf;

		$noMyLogo				= isset($conf->global->PDF_DISABLE_MYCOMPANY_LOGO)	? $conf->global->PDF_DISABLE_MYCOMPANY_LOGO : 0;
		$heightLogo				= 0;
		if (! empty($noMyLogo))	return $heightLogo;
		if ($logo)				$logo	= $conf->mycompany->dir_output.'/logos/'.$logo;
		else					$logo	= $conf->mycompany->dir_output.'/logos/'.$emetteur->logo;
		if ($logo)
		{
			if (is_file($logo) && is_readable($logo))
			{
				$heightLogo	= pdf_getHeightForLogo($logo);
				$pdf->Image($logo, $marge_gauche, $posy, 0, $heightLogo);	// width=0 (auto)
			}	// if (is_file($logo) && is_readable($logo))
			else
			{
				$pdf->SetTextColor(200, 0, 0);
				$pdf->SetFont('', 'B', $default_font_size - 2);
				$pdf->MultiCell($w, $tab_hl, $outputlangs->transnoentities("PDFInfraSPlusLogoFileNotFound", $logo), '', 'L', 0, 1, $marge_gauche, $posy, true, 0, 0, false, 0, 'M', false);
				$pdf->MultiCell($w, $tab_hl, $outputlangs->transnoentities("ErrorGoToGlobalSetup"), '', 'L', 0, 1, $marge_gauche, $pdf->getY() + 1, true, 0, 0, false, 0, 'M', false);
				$pdf->SetTextColor($headertxtcolor[0], $headertxtcolor[1], $headertxtcolor[2]);
				$heightLogo	= $pdf->getY() + 1;
			}	// else	// if (is_file($logo) && is_readable($logo))
		}	// if ($logo)
		else
		{
			$text		= $emetteur->name;
			$pdf->MultiCell($w, $tab_hl, $outputlangs->convToOutputCharset($text), '', 'L', 0, 1, $marge_gauche, $posy, true, 0, 0, false, 0, 'M', false);
			$heightLogo = $tab_hl;
		}	// else// if ($logo)
		return $heightLogo;
	}	// function pdf_InfraSPlus_logo($pdf, $outputlangs, $posy, $w, $logo, $emetteur, $marge_gauche, $tab_hl, $headertxtcolor)

	/************************************************
	*	Search Creator name
	*
	*	@param	Object		$object				Object
	*	@return	string							name found or empty
	************************************************/
	function pdf_InfraSPlus_creator($object, $outputlangs)
	{
		global $db;

	if (is_object($object->user_creation))
	{
		if ($object->user_creation->id)	return $object->user_creation->getFullName($outputlangs);
		else							return '';
	}	// if (is_object($object->user_creation))
	else
	{
			$userstatic				= new User($db);
			$userstatic->fetch($object->user_creation_id ? $object->user_creation_id : $object->user_creation);
			if ($userstatic->id)	return $userstatic->getFullName($outputlangs);
		else
			{
				$userstatic->fetch($object->user_author_id ? $object->user_author_id : $object->user_author);
				if ($userstatic->id)	return $userstatic->getFullName($outputlangs);
				else					return '';
			}	// else	// if ($userstatic->id)
	}	// else	// if (is_object($object->user_creation))
	}	// function pdf_InfraSPlus_creator($object)

	/************************************************
	*	Show linked objects for PDF generation
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Object		$object			Object shown in PDF
	*	@param	Translate	$outputlangs	Output langs object
	*	@param  int			$posx			Pos x
	*	@param  int			$posy			Pos y
	*	@param  int			$w				Width of cells. If 0, they extend up to the right margin of the page
	*	@param  int			$tab_hl			Cell minimum height. The cell extends automatically if needed.
	*	@param	int			$align			Align
	*	@param  string		$obj_type_avoid	Object type linked not to show
	*	@return	float						The Y PDF position
	************************************************/
	function pdf_InfraSPlus_writeLinkedObjects(&$pdf, $object, $outputlangs, $posx, $posy, $w, $tab_hl, $align, $obj_type_avoid = '')
	{
		global $conf;

		$nodatelinked	= isset($conf->global->INFRASPLUS_PDF_NO_DATE_LINKED)	? $conf->global->INFRASPLUS_PDF_NO_DATE_LINKED : 0;
		$linkedobjects	= pdf_InfraSPlus_getLinkedObjects($object, $outputlangs, $obj_type_avoid);
		if (! empty($linkedobjects))
		{
			foreach($linkedobjects as $linkedobject)
			{
				$reftoshow															= $linkedobject["ref_title"].' : '.$linkedobject["ref_value"];
				if (empty($nodatelinked) && ! empty($linkedobject["date_value"]))	$reftoshow	.= ' / '.$linkedobject["date_value"];
				$posy																+= $tab_hl - 0.5;
				$pdf->MultiCell($w, $tab_hl, $reftoshow, '', $align, 0, 1, $posx, $posy, true, 0, 0, false, 0, 'M', false);
			}	// foreach($linkedobjects as $linkedobject)
		}	// if (! empty($linkedobjects))
		return $pdf->getY();
	}	// function pdf_InfraSPlus_writeLinkedObjects(&$pdf, $object, $outputlangs, $posx, $posy, $w, $tab_hl, $align)

	/************************************************
	*	Return linked objects to use for document generation.
	*	Warning: To save space, this function returns only one link per link type (all links are concated on same record string).
	*	This function is used by pdf_InfraSPlus_writeLinkedObjects
	*
	*	@param  Object		$object			Object shown in PDF
	*	@param	Translate	$outputlangs	Output langs object
	*	@param  string		$obj_type_avoid	Object type linked not to show ('facture', 'propal', 'supplier_proposal', 'commande',
	*																		'supplier_order', 'contrat', 'fichinter', 'shipping')
	* 	@return	array                       Linked objects
	************************************************/
	function pdf_InfraSPlus_getLinkedObjects($object, $outputlangs, $obj_type_avoid = '')
	{
		global $conf, $db, $hookmanager;

		$projectlinked	= isset($conf->global->INFRASPLUS_PDF_SHOW_REF_PROJECT)		? $conf->global->INFRASPLUS_PDF_SHOW_REF_PROJECT	: 0;
		$projectdesc	= isset($conf->global->INFRASPLUS_PDF_SHOW_DESC_PROJECT)	? $conf->global->INFRASPLUS_PDF_SHOW_DESC_PROJECT	: 0;
		$linkedobjects	= array();
		$object->fetchObjectLinked();
		foreach($object->linkedObjects as $objecttype => $objects)
		{
			if ($objecttype == 'facture' && $obj_type_avoid != 'facture')
			{
			   // For invoice, we don't want to have a reference line on document. Image we are using recuring invoice, we will have a line longer than document width.
			}	// if ($objecttype == 'facture' && $obj_type_avoid != 'facture')
			elseif (($objecttype == 'propal' && $obj_type_avoid != 'propal') || ($objecttype == 'supplier_proposal' && $obj_type_avoid != 'supplier_proposal'))
			{
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("RefProposal");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref);
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("DatePropal");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->date, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif (($objecttype == 'propal' && $obj_type_avoid != 'propal') || ($objecttype == 'supplier_proposal' && $obj_type_avoid != 'supplier_proposal'))
			elseif (($objecttype == 'commande' && $obj_type_avoid != 'commande') || ($objecttype == 'supplier_order' && $obj_type_avoid != 'supplier_order'))
			{
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("RefOrder");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref).($elementobject->ref_client ? ' ('.$elementobject->ref_client.')' : '') . ($elementobject->ref_supplier ? ' ('.$elementobject->ref_supplier.')' : '');
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("OrderDate");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->date, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif (($objecttype == 'commande' && $obj_type_avoid != 'commande') || ($objecttype == 'supplier_order' && $obj_type_avoid != 'supplier_order'))
			elseif ($objecttype == 'contrat' && $obj_type_avoid != 'contrat')
			{
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("RefContract");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref);
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("DateContract");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->date_contrat, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif ($objecttype == 'contrat' && $obj_type_avoid != 'contrat')
			elseif ($objecttype == 'fichinter' && $obj_type_avoid != 'fichinter')
			{
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("PDFInfraSPlusRefInter");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref);
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("Date");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->datec, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif ($objecttype == 'fichinter' && $obj_type_avoid != 'fichinter')
			elseif ($objecttype == 'shipping' && $obj_type_avoid != 'shipping')
			{
				foreach($objects as $x => $elementobject)
				{
					$order														= null;
					// We concat this record info into fields xxx_value. title is overwrote.
					if (empty($object->linkedObjects['commande']) && $object->element != 'commande')	// There is not already a link to order and object is not the order, so we show also info with order
					{
						$elementobject->fetchObjectLinked();
						if (! empty($elementobject->linkedObjects['commande']))	$order	= reset($elementobject->linkedObjects['commande']);
					}	// if (empty($object->linkedObjects['commande']) && $object->element != 'commande')
					if (! is_object($order))
					{
						$linkedobjects[$objecttype]['ref_title']				= $outputlangs->transnoentities("RefSending");
						if (! empty($linkedobjects[$objecttype]['ref_value']))	$linkedobjects[$objecttype]['ref_value']	.=' / ';
						$linkedobjects[$objecttype]['ref_value']				.= $outputlangs->transnoentities($elementobject->ref);
					}	// if (! is_object($order))
					else
					{
						$linkedobjects[$objecttype]['ref_title']				= $outputlangs->transnoentities("RefOrder").' / '.$outputlangs->transnoentities("RefSending");
						if (empty($linkedobjects[$objecttype]['ref_value']))	$linkedobjects[$objecttype]['ref_value']	= $outputlangs->convToOutputCharset($order->ref) . ($order->ref_client ? ' ('.$order->ref_client.')' : '');
						$linkedobjects[$objecttype]['ref_value']				.= ' / '.$outputlangs->transnoentities($elementobject->ref);
					}	// else	// if (! is_object($order))
				}	// foreach($objects as $x => $elementobject)
			}	// elseif ($objecttype == 'shipping' && $obj_type_avoid != 'shipping')
		}	// foreach($object->linkedObjects as $objecttype => $objects)
		if ($projectlinked && ! empty($conf->projet->enabled) && ! empty($object->fk_project))
		{
					require_once DOL_DOCUMENT_ROOT.'/projet/class/project.class.php';
					$proj		= new Project($db);
					$proj->fetch($object->fk_project);
					$linkedobjects['project']['ref_title']	= $outputlangs->transnoentities("RefProject");
					$linkedobjects['project']['ref_value']	= $proj->ref.($projectdesc ? ' - '.$proj->title : ''); // simplification => only the reference remains
		}	// if ($projectlinked && ! empty($conf->projet->enabled) && ! empty($object->fk_project))
		// For add external linked objects
		if (is_object($hookmanager))
		{
			$parameters								= array('linkedobjects' => $linkedobjects, 'outputlangs' => $outputlangs);
			$action									= '';
			$hookmanager->executeHooks('pdf_getLinkedObjects', $parameters, $object, $action);
			if (! empty($hookmanager->resArray))	$linkedobjects	= $hookmanager->resArray;
		}	// if (is_object($hookmanager))
		return $linkedobjects;
	}	// function pdf_InfraSPlus_getLinkedObjects($object, $outputlangs)

	/************************************************
	*	Return linked shipping objects to use for document generation.
	*
	*	@param  Object		$object			Object shown in PDF
	*	@param	Translate	$outputlangs	Output langs object
	* 	@return	array                       Linked shippings
	************************************************/
	function pdf_InfraSPlus_getLinkedshippings($object, $outputlangs)
	{
		global $conf, $db;

		$linkedshippings	= array();
		$sql				 = 'SELECT *';
		$sql				.= ' FROM '.MAIN_DB_PREFIX.'element_element AS ee';
		$sql				.= ' INNER JOIN '.MAIN_DB_PREFIX.'expedition AS e';
		$sql				.= ' ON ee.fk_source = e.rowid';
		$sql				.= ' WHERE ee.sourcetype = "shipping"';
		$sql				.= ' AND ee.fk_target = "'.$object->id.'"';
		$resql				= $db->query($sql);
		if ($resql)
		{
			$num	= $db->num_rows($resql);
			for ($i = 0; $i < $num; $i++)
			{
				$obj		= $db->fetch_object($resql);
				$linkedshippings['ref']					= $obj->ref;
				$linkedshippings['date_delivery']		= $obj->date_delivery;
				$linkedshippings['fk_shipping_method']	= $obj->fk_shipping_method;
				$linkedshippings['tracking_number']		= $obj->tracking_number;
			}	// for ($i = 0; $i < $num; $i++)
		}	// if ($resql)
		$db->free($resql);
		return $linkedshippings;
	}	// function pdf_InfraSPlus_getLinkedshippings($object, $outputlangs)

	/************************************************
	*	Get all addresses needed
	*
	*	@param  object		$object			Object shown in PDF
	*	@param  Translate	$outputlangs	Object lang for output
	*	@param	array		$arrayidcontact	List of contact ID
	*	@param	string		$adr			Alternative address for sender
	* 	@param	string		$adrlivr		Shipping address
	* 	@param	object		$emetteur		Object company
	* 	@param	boolean		$invLivr		Inversion shipping address and recepient one
	* 	@param	string		$typeadr		for supplier documents if we use both internal and external adrresses (shipping to customer)
	*	@param	object		$adrfact		Object address
	*	@param	int			$ticket			set to 1 for special format (small page)
	*	@param	object		$adrSst			Object address
	* 	@return	array		Return all addresses found
	************************************************/
	function pdf_InfraSPlus_getAddresses($object, $outputlangs, $arrayidcontact, $adr, $adrlivr, $emetteur, $invLivr = 0, $typeadr = '', $adrfact = null, $ticket = 0, $adrSst = null)
	{
		global $db, $conf;

		$show_emet_details		= isset($conf->global->INFRASPLUS_PDF_SHOW_EMET_DETAILS) && empty($ticket)		? $conf->global->INFRASPLUS_PDF_SHOW_EMET_DETAILS		: 0;
		$use_comp_name_contact	= isset($conf->global->MAIN_USE_COMPANY_NAME_OF_CONTACT)						? $conf->global->MAIN_USE_COMPANY_NAME_OF_CONTACT		: 0;
		$includealias			= isset($conf->global->PDF_INCLUDE_ALIAS_IN_THIRDPARTY_NAME)					? $conf->global->PDF_INCLUDE_ALIAS_IN_THIRDPARTY_NAME	: 0;
		$show_sender_alias		= isset($conf->global->INFRASPLUS_PDF_SHOW_SENDER_ALIAS)						? $conf->global->INFRASPLUS_PDF_SHOW_SENDER_ALIAS		: 0;
		$show_recep_details		= isset($conf->global->INFRASPLUS_PDF_SHOW_RECEP_DETAILS) && empty($ticket)		? $conf->global->INFRASPLUS_PDF_SHOW_RECEP_DETAILS		: 0;
		$showadrlivr			= isset($conf->global->INFRASPLUS_PDF_SHOW_ADRESSE_RECEPTION)					? $conf->global->INFRASPLUS_PDF_SHOW_ADRESSE_RECEPTION	: 0;
		$showadrSsT				= isset($conf->global->INFRASPLUS_PDF_ADRESSE_SOUS_TRAITANT)					? $conf->global->INFRASPLUS_PDF_ADRESSE_SOUS_TRAITANT	: 0;
		// Sender properties
		$sender_Alias			= '';
		$carac_emetteur			= '';
		// Add internal contact if defined
		if (is_array($arrayidcontact['I']) && count($arrayidcontact['I']) > 0)
		{
			$object->fetch_user($arrayidcontact['I'][0]);
			$carac_emetteur	.= $outputlangs->convToOutputCharset($object->user->getFullName($outputlangs))."\n";
		}	// if (is_array($arrayidcontact['I']) && count($arrayidcontact['I']) > 0)
		if ($adrlivr && $typeadr == 'supplierInvoice')	$carac_emetteur .= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $emetteur, $object->client, '', 0, ($show_emet_details ? 'source' : 'sourcewithnodetails'), $object, 1, false, $ticket);
		elseif ($adr)
		{
			$sender_Alias	= $show_sender_alias ? $adr->name : '';
			$carac_emetteur .= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $adr, '', '', 0, ($show_emet_details ? 'source' : 'sourcewithnodetails'), $object, 1, false, $ticket);
		}	// elseif ($adr)	// if ($adrlivr && $typeadr == 'supplierInvoice')
		else	$carac_emetteur .= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $emetteur, '', '', 0, ($show_emet_details ? 'source' : 'sourcewithnodetails'), $object, 1, false, $ticket);
		if (is_object ($arrayidcontact['U']))
		{
			$carac_client_name	= $outputlangs->transnoentities("AUTHOR").' : '.$arrayidcontact['U']->getFullName($outputlangs);
			$carac_client		= $outputlangs->transnoentities("DateCreation").' : '.dol_print_date($object->date_create, 'day', false, $outputlangs)."\n";
			$det_client			= $outputlangs->convToOutputCharset(dol_format_address($arrayidcontact['U']->address, 0, "\n", $outputlangs))."\n";
			$det_client			.= $arrayidcontact['U']->email ? $outputlangs->transnoentities("Email").' : '.$outputlangs->convToOutputCharset($arrayidcontact['U']->email)."\n" : '';
			$det_client			.= $arrayidcontact['B']->iban ? $outputlangs->transnoentities("IBAN").' : '.$outputlangs->convToOutputCharset($arrayidcontact['B']->iban)."\n" : '';
			$carac_client		.= $det_client ? $det_client."\n" : '';
			if ($object->fk_statut == 99 && $object->fk_user_refuse > 0)
			{
				$userfee		= new User($db);
				$userfee->fetch($object->fk_user_refuse);
				$carac_client	.= $outputlangs->transnoentities("REFUSEUR").' : '.$userfee->getFullName($outputlangs)."\n";
				$carac_client	.= $outputlangs->transnoentities("MOTIF_REFUS").' : '.$outputlangs->convToOutputCharset($object->detail_refuse)."\n";
				$carac_client	.= $outputlangs->transnoentities("DATE_REFUS").' : '.dol_print_date($object->date_refuse, 'day', false, $outputlangs);
			}	// if ($object->fk_statut == 99 && $object->fk_user_refuse > 0)
			elseif ($object->fk_statut == 4 && $object->fk_user_cancel > 0)
			{
				$userfee		= new User($db);
				$userfee->fetch($object->fk_user_cancel);
				$carac_client	.= $outputlangs->transnoentities("CANCEL_USER").' : '.$userfee->getFullName($outputlangs)."\n";
				$carac_client	.= $outputlangs->transnoentities("MOTIF_CANCEL").' : '.$outputlangs->convToOutputCharset($object->detail_cancel)."\n";
				$carac_client	.= $outputlangs->transnoentities("DATE_CANCEL").' : '.dol_print_date($object->date_cancel, 'day', false, $outputlangs);
			}	// elseif ($object->fk_statut == 4 && $object->fk_user_cancel > 0)
			elseif ($object->fk_user_approve > 0)
			{
				$userfee		= new User($db);
				$userfee->fetch($object->fk_user_approve);
				$carac_client	.= $outputlangs->transnoentities("VALIDOR").' : '.$userfee->getFullName($outputlangs)."\n";
				$carac_client	.= $outputlangs->transnoentities("DateApprove").' : '.dol_print_date($object->date_approve, 'day', false, $outputlangs);
			}	// elseif ($object->fk_user_approve > 0)
			if ($object->fk_statut == 6 && $object->fk_user_paid > 0)
			{
				$userfee		= new User($db);
				$userfee->fetch($object->fk_user_paid);
				$carac_client	.= $outputlangs->transnoentities("AUTHORPAIEMENT").' : '.$userfee->getFullName($outputlangs)."\n";
				$carac_client	.= $outputlangs->transnoentities("DATE_PAIEMENT").' : '.dol_print_date($object->date_paiement, 'day', false, $outputlangs);
			}	// if ($object->fk_statut == 6 && $object->fk_user_paid > 0)
		}	// if (is_object ($arrayidcontact['U']))
		else
		{
			// Recipient properties - If contact defined, we use it
			if ($adrfact)
			{
				$carac_client		= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $emetteur, $adrfact, '', 0, ($show_recep_details ? 'targetwithdetails' : 'target'), $object, 2, false, $ticket);
				if ($carac_client)	$carac_client_name	= $outputlangs->convToOutputCharset($adrfact->name);
			}	// if ($adrfact)
			else
			{
				$usecontact		= false;
				$norepeatname	= false;
				if (is_array($arrayidcontact['E']) && count($arrayidcontact['E']) > 0)
				{
					$usecontact	= true;
					$result		= $object->fetch_contact($arrayidcontact['E'][0]);
				}	// if (is_array($arrayidcontact['E']) && count($arrayidcontact['E']) > 0)
				//Recipient name
				// On peut utiliser le nom de la societe du contact
				if ($usecontact && !empty($use_comp_name_contact))
				{
					$norepeatname	= true;
					$thirdparty		= $object->contact;
				}	// if ($usecontact && !empty($use_comp_name_contact))
				else				$thirdparty	= $typeadr == 'supplierInvoice' ? $adrlivr : infraspackplus_check_parent_addr_fact ($object);
				$carac_client_name	= pdf_InfraSPlus_Build_Third_party_Name($thirdparty, $outputlangs, $includealias);
				$carac_client		= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $emetteur, $thirdparty, ($usecontact ? $object->contact : ''), $usecontact, ($show_recep_details ? 'targetwithdetails' : 'target'), $object, 1, $norepeatname, $ticket);
			}	// else	if ($adrfact)
			// Shipping address
			if ($showadrlivr && $adrlivr)
			{
				if ($adrlivr == 'Default')	$livrshow	= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $emetteur, $object->thirdparty, '', 0, ($show_recep_details ? 'targetwithdetails' : 'target'), $object, 0, false, $ticket);
				else						$livrshow	= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $emetteur, $adrlivr, '', 0, ($show_recep_details ? 'targetwithdetails' : 'target'), $object, 0, false, $ticket);
			}	// if ($showadrlivr && $adrlivr)
			// Subcontractor address
			if ($showadrSsT && $adrSst)	$SsTshow	= pdf_InfraSPlus_build_address($outputlangs, $emetteur, $emetteur, $adrSst, '', 0, ($show_recep_details ? 'targetwithdetails' : 'target'), $object, 0, false, $ticket);
		}	// else	// if (count($arrayidcontact['U']) > 0)
		return	array(	'sender_Alias'		=> $sender_Alias,
						'carac_emetteur'	=> $carac_emetteur,
						'carac_client_name'	=> $invLivr && $livrshow ? $adrlivr->name : $carac_client_name,
						'carac_client'		=> $invLivr && $livrshow ? $livrshow : $carac_client,
						'livrshow'			=> $livrshow,
						'SsTshow'			=> $SsTshow
					 );
	}	// function pdf_InfraSPlus_getAddresses($object, $outputlangs, $arrayidcontact, $adr, $adrlivr, $emetteur, $invLivr = 0, $typeadr = '', $adrfact = '', $ticket = 0, $adrSst = '')

	/************************************************
	*	Return a string with full address formated for output on documents
	*
	*	@param	Translate			$outputlangs		Output langs object
	*	@param  Societe				$sourcecompany		Source company object
	*	@param  Address				$sourceaddress		Source address object
	*	@param  Societe|string|null	$targetcompany		Target company object
	*	@param  Contact|string|null	$targetcontact		Target contact object
	*	@param	int					$usecontact			Use contact instead of company
	*	@param	string				$mode				Address type ('source', 'sourcewithnodetails', 'target', 'targetwithnodetails', 'targetwithdetails',
	*																	'targetwithdetails_xxx': target but include also phone/fax/email/url)
	*	@param  Object   		   $object				Object we want to build document for
	*	@param  Bolean				$profids			Display profesionnal IDs
	*	@param  Bolean				$norepeatname		Display name of contact
	*	@return	string									String with full address
	************************************************/
	function pdf_InfraSPlus_build_address($outputlangs, $sourcecompany, $sourceaddress = '', $targetcompany = '', $targetcontact = '', $usecontact = 0, $mode = 'source', $object = null, $profids = 0, $norepeatname = false, $ticket = 0)
	{
		global $conf, $hookmanager;

		if (strpos($mode, 'source') === 0 && ! is_object($sourcecompany))		return -1;
		if (strpos($mode, 'source') === 0 && ! is_object($sourceaddress))		return -1;
		if (strpos($mode, 'target') === 0 && ! is_object($targetcompany))		return -1;
		if (! empty($sourceaddress->state_id) && empty($sourceaddress->state))	$sourceaddress->state	= getState($sourceaddress->state_id);
		if (! empty($targetcompany->state_id) && empty($targetcompany->state))	$targetcompany->state	= getState($targetcompany->state_id);
		$targetcompanyIDs														= $profids == 2 ? infraspackplus_check_parent_addr_fact ($object) : $targetcompany;
		$reshook																= 0;
		$stringaddress															= '';
		if (is_object($hookmanager))
		{
			$parameters		= array('sourcecompany'=>&$sourcecompany, 'targetcompany'=>&$targetcompany, 'targetcontact'=>&$targetcontact, 'outputlangs'=>$outputlangs, 'mode'=>$mode, 'usecontact'=>$usecontact);
			$action			= '';
			$reshook		= $hookmanager->executeHooks('pdf_build_address', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			$stringaddress	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			$withCountry	= (!empty($sourceaddress->country_code) && ($targetcompany->country_code != $sourceaddress->country_code)) ? 1 : 0;	// Country
			if ($mode == 'source' || $mode == 'sourcewithnodetails')
			{
				$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($sourceaddress, $withCountry, "\n", $outputlangs)).($ticket ? '' : "\n");
				if ( $mode != 'sourcewithnodetails')
				{
					if (empty($conf->global->MAIN_PDF_DISABLESOURCEDETAILS))
					{
						// Phone
						if ($sourceaddress->phone)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("PhoneShort").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($sourceaddress->phone)));
						// Fax
						if ($sourceaddress->fax)	$stringaddress	.= ($stringaddress ? ($sourceaddress->phone ? " - " : "\n") : '' ).$outputlangs->transnoentities("Fax").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($sourceaddress->fax)));
						// EMail
						if ($sourceaddress->email)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Email").": ".$outputlangs->convToOutputCharset($sourceaddress->email);
						// Web
						if ($sourceaddress->url)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Web").": ".$outputlangs->convToOutputCharset($sourceaddress->url);
					}	// if (empty($conf->global->MAIN_PDF_DISABLESOURCEDETAILS))
				}	// if ( $mode != 'sourcewithnodetails')
				if ($profids)
				{
					if ((! empty($conf->global->INFRASPLUS_PDF_TVAINTRA_IN_SOURCE_ADDRESS) || ! empty($ticket)) && ! empty($sourcecompany->tva_intra))
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("VATIntraShort").' : '.$outputlangs->convToOutputCharset($sourcecompany->tva_intra);
					if ((! empty($conf->global->INFRASPLUS_PDF_PROFID1_IN_SOURCE_ADDRESS) || ! empty($ticket)) && ! empty($sourcecompany->idprof1))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId1",$sourcecompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($sourcecompany->idprof1);
					}	// if (! empty($conf->global->INFRASPLUS_PDF_PROFID1_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof1))
					if (! empty($conf->global->INFRASPLUS_PDF_PROFID2_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof2))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId2",$sourcecompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($sourcecompany->idprof2);
					}	// if (! empty($conf->global->INFRASPLUS_PDF_PROFID2_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof2))
					if (! empty($conf->global->INFRASPLUS_PDF_PROFID3_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof3))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId3",$sourcecompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($sourcecompany->idprof3);
					}	// if (! empty($conf->global->INFRASPLUS_PDF_PROFID3_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof3))
					if (! empty($conf->global->INFRASPLUS_PDF_PROFID4_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof4))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId4",$sourcecompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($sourcecompany->idprof4);
					}	// if (! empty($conf->global->INFRASPLUS_PDF_PROFID4_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof4))
					if (! empty($conf->global->INFRASPLUS_PDF_PROFID5_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof5))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId5",$sourcecompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($sourcecompany->idprof5);
					}	// if (! empty($conf->global->INFRASPLUS_PDF_PROFID5_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof5))
					if (! empty($conf->global->INFRASPLUS_PDF_PROFID6_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof6))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId6",$sourcecompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($sourcecompany->idprof6);
					}	// if (! empty($conf->global->INFRASPLUS_PDF_PROFID6_IN_SOURCE_ADDRESS) && ! empty($sourcecompany->idprof6))
				}	// if ($profids)
			}	// if ($mode == 'source')
			if ($mode == 'target' || $mode == 'targetwithnodetails' || preg_match('/targetwithdetails/',$mode))
			{
				if ($usecontact)
				{
					if (! $norepeatname)					$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset($targetcontact->getFullName($outputlangs, 1, -1));
					if (!empty($targetcontact->address))	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($targetcontact, $withCountry, "\n", $outputlangs))."\n";
					else
					{
						$companytouseforaddress	= $targetcompany;
						if ($targetcontact->socid > 0 && $targetcontact->socid != $targetcompany->id)	// Contact on a thirdparty that is a different thirdparty than the thirdparty of object
						{
							$targetcontact->fetch_thirdparty();
							$companytouseforaddress	= $targetcontact->thirdparty;
						}	// if ($targetcontact->socid > 0 && $targetcontact->socid != $targetcompany->id)
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($companytouseforaddress, $withCountry, "\n", $outputlangs))."\n";
					}	// else	// if (!empty($targetcontact->address))
					if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
					{
						// Phone
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_phone/',$mode))
						{
							if (! empty($targetcontact->phone_pro) || ! empty($targetcontact->phone_mobile))	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("PhoneShort").": ";
							if (! empty($targetcontact->phone_pro))												$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcontact->phone_pro)));
							if (! empty($targetcontact->phone_pro) && ! empty($targetcontact->phone_mobile))	$stringaddress	.= " / ";
							if (! empty($targetcontact->phone_mobile))											$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcontact->phone_mobile)));
						}	//  if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_phone/',$mode))
						// Fax
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_fax/',$mode))
							if ($targetcontact->fax)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Fax").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcontact->fax)));
						// EMail
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_email/',$mode))
							if ($targetcontact->email)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Email").": ".$outputlangs->convToOutputCharset($targetcontact->email);
						// Web
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_url/',$mode))
							if ($targetcontact->url)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Web").": ".$outputlangs->convToOutputCharset($targetcontact->url);
					}	// if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
				}	// if ($usecontact)
				else
				{
					$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($targetcompany, $withCountry, "\n", $outputlangs)).($ticket ? '' : "\n");
					if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
					{
						// Phone
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_phone/',$mode))
						{
							if (! empty($targetcompany->phone) || ! empty($targetcompany->phone_mobile))	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("PhoneShort").": ";
							if (! empty($targetcompany->phone))												$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcompany->phone)));
							if (! empty($targetcompany->phone) && ! empty($targetcompany->phone_mobile))	$stringaddress	.= " / ";
							if (! empty($targetcompany->phone_mobile))										$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcompany->phone_mobile)));
						}
						// Fax
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_fax/',$mode))
							if ($targetcompany->fax)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Fax").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcompany->fax)));
						// EMail
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_email/',$mode))
							if ($targetcompany->email)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Email").": ".$outputlangs->convToOutputCharset($targetcompany->email);
						// Web
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_url/',$mode))
							if ($targetcompany->url)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Web").": ".$outputlangs->convToOutputCharset($targetcompany->url);
					}	// if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
				}	// else	// if ($usecontact)
				$listElementsCli	= array('propal', 'commande', 'facture', 'contrat', 'shipping', 'fichinter');
				if (in_array($object->element, $listElementsCli) && ! empty($conf->global->INFRASPLUS_PDF_SHOW_NUM_CLI) && $mode != 'targetwithnodetails')
					if ($object->thirdparty->code_client)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("PDFInfraSPlusCodeCli").' : '.$outputlangs->convToOutputCharset($object->thirdparty->code_client);
				// Intra VAT
				if (empty($conf->global->MAIN_TVAINTRA_NOT_IN_ADDRESS) && $mode != 'targetwithnodetails')
					if ($targetcompanyIDs->tva_intra)	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("VATIntraShort").' : '.$outputlangs->convToOutputCharset($targetcompanyIDs->tva_intra);
				// Professionnal Ids
				if ($profids)
				{
					if (! empty($conf->global->MAIN_PROFID1_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof1))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId1",$targetcompanyIDs->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($targetcompanyIDs->idprof1);
					}	// if (! empty($conf->global->MAIN_PROFID1_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof1))
					if (! empty($conf->global->MAIN_PROFID2_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof2))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId2",$targetcompanyIDs->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($targetcompanyIDs->idprof2);
					}	// if (! empty($conf->global->MAIN_PROFID2_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof2))
					if (! empty($conf->global->MAIN_PROFID3_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof3))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId3",$targetcompanyIDs->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($targetcompanyIDs->idprof3);
					}	// if (! empty($conf->global->MAIN_PROFID3_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof3))
					if (! empty($conf->global->MAIN_PROFID4_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof4))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId4",$targetcompanyIDs->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($targetcompanyIDs->idprof4);
					}	// if (! empty($conf->global->MAIN_PROFID4_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof4))
					if (! empty($conf->global->MAIN_PROFID5_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof5))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId5",$targetcompanyIDs->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($targetcompanyIDs->idprof5);
					}	// if (! empty($conf->global->MAIN_PROFID5_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof5))
					if (! empty($conf->global->MAIN_PROFID6_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof6))
					{
						$tmp			= $outputlangs->transcountrynoentities("ProfId6",$targetcompanyIDs->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$tmp.' : '.$outputlangs->convToOutputCharset($targetcompanyIDs->idprof6);
					}	// if (! empty($conf->global->MAIN_PROFID6_IN_ADDRESS) && ! empty($targetcompanyIDs->idprof6))
				}	// if ($profids)
				// Public note
				if (! empty($conf->global->MAIN_PUBLIC_NOTE_IN_ADDRESS))
				{
					if ($mode == 'source' && ! empty($sourcecompany->note_public))
						$stringaddress	.= ($stringaddress ? "\n" : '' ).dol_string_nohtmltag($sourcecompany->note_public);
					if (($mode == 'target' || preg_match('/targetwithdetails/',$mode)) && ! empty($targetcompany->note_public))
						$stringaddress	.= ($stringaddress ? "\n" : '' ).dol_string_nohtmltag($targetcompany->note_public);
				}	// if (! empty($conf->global->MAIN_PUBLIC_NOTE_IN_ADDRESS))
			}	// if ($mode == 'target' || $mode == 'targetwithnodetails' || preg_match('/targetwithdetails/',$mode))
		}	// if (empty($reshook))
		return $stringaddress;
	}	// function pdf_InfraSPlus_build_address($outputlangs, $sourcecompany, $sourceaddress = '', $targetcompany = '', $targetcontact = '', $usecontact = 0, $mode = 'source', $object = null, $profids = 0, $norepeatname = false, $ticket = 0)

	/************************************************
	*	Returns the name of the thirdparty
	*
	*	@param	Societe|Contact     $thirdparty     Contact or thirdparty
	*	@param	Translate           $outputlangs    Output language
	*	@param	int                 $includealias   1 = Include alias name before name
	*	@return	string				String with name of thirdparty (+ alias if requested)
	************************************************/
	function pdf_InfraSPlus_Build_Third_party_Name($thirdparty, $outputlangs, $includealias = 0)
	{
		global $conf;

		$statusWithName	= ! empty($conf->global->INFRASPLUS_PDF_SHOW_STATUS_WITH_CLIENT_NAME) && $thirdparty->forme_juridique_code ? ' '.$outputlangs->convToOutputCharset(getFormeJuridiqueLabel($thirdparty->forme_juridique_code)) : '';
		$socname		= '';
		if ($thirdparty instanceof Societe)
		{
			$socname												.= $thirdparty->name.$statusWithName;
			if ($includealias && !empty($thirdparty->name_alias))	$socname	.= "\n".$thirdparty->name_alias;
		}	// if ($thirdparty instanceof Societe)
		elseif ($thirdparty instanceof Contact)						$socname	= $outputlangs->convToOutputCharset($thirdparty->getFullName($outputlangs, 1, -1));//$thirdparty->socname;
		else														throw new InvalidArgumentException('Parameter 1 = $thirdparty is not a Societe nor Contact');

		return $outputlangs->convToOutputCharset($socname);
	}	// function pdf_InfraSPlus_Build_Third_party_Name($thirdparty, $outputlangs, $includealias = 0)

	/************************************************
	*	Show top small header of page.
	*
	*	@param	TCPDF		$pdf			The PDF factory
	*	@param  Object		$object			Object shown in PDF
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	* 	@param	array		$dimCadres		Frame dimensions => 'R', 'S', 'Y'
	* 	@param	int			$tab_hl			Line height
	* 	@param	object		$emetteur		Object company
	* 	@param	array		$addresses		All addresses found
	* 	@param	int			$Rounded_rect	Radius corner value
	* 	@param	boolean		$ndf			Frames title for expense report
	* 	@return	float		return frame height
	************************************************/
	function pdf_InfraSPlus_writeAddresses(&$pdf, $object, $outputlangs, $formatpage, $dimCadres, $tab_hl, $emetteur, $addresses, $Rounded_rect, $ndf = false)
	{
		global $conf;

		$invert_sender_recipient			= isset($conf->global->MAIN_INVERT_SENDER_RECIPIENT)	? $conf->global->MAIN_INVERT_SENDER_RECIPIENT		: 0;
		$hide_recep_frame					= isset($conf->global->INFRASPLUS_PDF_HIDE_RECEP_FRAME)	? $conf->global->INFRASPLUS_PDF_HIDE_RECEP_FRAME	: 0;
		$frmeLineW							= isset($conf->global->INFRASPLUS_PDF_FRM_E_LINE_WIDTH)	? $conf->global->INFRASPLUS_PDF_FRM_E_LINE_WIDTH	: 0.2;
		$frmeLineDash						= isset($conf->global->INFRASPLUS_PDF_FRM_E_LINE_DASH)	? $conf->global->INFRASPLUS_PDF_FRM_E_LINE_DASH		: '0';
		$frmeLineColor						= isset($conf->global->INFRASPLUS_PDF_FRM_E_LINE_COLOR)	? $conf->global->INFRASPLUS_PDF_FRM_E_LINE_COLOR	: '';
		$frmeLineColor						= explode(',', $frmeLineColor);
		$frmeBgColor						= isset($conf->global->INFRASPLUS_PDF_FRM_E_BG_COLOR)	? $conf->global->INFRASPLUS_PDF_FRM_E_BG_COLOR		: '';
		$frmeBgColor						= explode(',', $frmeBgColor);
		$frmeAlpha							= isset($conf->global->INFRASPLUS_PDF_FRM_E_OPACITY)	? $conf->global->INFRASPLUS_PDF_FRM_E_OPACITY		: 30;
		$frmrLineW							= isset($conf->global->INFRASPLUS_PDF_FRM_R_LINE_WIDTH)	? $conf->global->INFRASPLUS_PDF_FRM_R_LINE_WIDTH	: 0.2;
		$frmrLineDash						= isset($conf->global->INFRASPLUS_PDF_FRM_R_LINE_DASH)	? $conf->global->INFRASPLUS_PDF_FRM_R_LINE_DASH		: '0';
		$frmrLineColor						= isset($conf->global->INFRASPLUS_PDF_FRM_R_LINE_COLOR)	? $conf->global->INFRASPLUS_PDF_FRM_R_LINE_COLOR	: '';
		$frmrLineColor						= explode(',', $frmrLineColor);
		$frmrBgColor						= isset($conf->global->INFRASPLUS_PDF_FRM_R_BG_COLOR)	? $conf->global->INFRASPLUS_PDF_FRM_R_BG_COLOR		: '';
		$frmrBgColor						= explode(',', $frmrBgColor);
		$frmrAlpha							= isset($conf->global->INFRASPLUS_PDF_FRM_R_OPACITY)	? $conf->global->INFRASPLUS_PDF_FRM_R_OPACITY		: 30;
		$frmeLineCap						= 'butt';	// fin de trait : butt = rectangle/lg->Dash ; round = rond/lg->Dash + width : square = rectangle/lg->Dash + width
		$frmeLineStyle						= array('width'=>$frmeLineW, 'dash'=>$frmeLineDash, 'cap'=>$frmeLineCap, 'color'=>$frmeLineColor);
		$frmrLineCap						= 'butt';	// fin de trait : butt = rectangle/lg->Dash ; round = rond/lg->Dash + width : square = rectangle/lg->Dash + width
		$frmrLineStyle						= array('width'=>$frmrLineW, 'dash'=>$frmrLineDash, 'cap'=>$frmrLineCap, 'color'=>$frmrLineColor);
		$default_font_size					= pdf_getPDFFontSize($outputlangs);
		$decal_round						= $Rounded_rect > 0 ? $Rounded_rect : 0;
		if ($formatpage['largeur'] < 210)	$dimCadres['R']	= 84;	// To work with US executive format
		if (! empty($invert_sender_recipient))
		{
			$dimCadres['xS']	= $formatpage['largeur'] - $formatpage['mdroite'] - $dimCadres['S'];
			$dimCadres['xR']	= $formatpage['mgauche'];
		}	// if (! empty($invert_sender_recipient))
		else
		{
			$dimCadres['xS']	= $formatpage['mgauche'];
			$dimCadres['xR']	= $formatpage['largeur'] - $formatpage['mdroite'] - $dimCadres['R'];
		}	// else	// if (! empty($invert_sender_recipient))
		$pdf->startTransaction();
		$hauteurcadre	= pdf_InfraSPlus_writeFrame($pdf, $outputlangs, $default_font_size, $tab_hl, $dimCadres, $emetteur, $addresses);
		$pdf->rollbackTransaction(true);
		if (! $hide_recep_frame)
		{
			$pdf->SetFont('', '', $default_font_size - 2);
			$pdf->MultiCell($dimCadres['S'], $tab_hl + 1, $outputlangs->transnoentities($ndf ? "TripSociete" : "BillFrom").' : ', '', 'L', 0, 1, $dimCadres['xS'] + $decal_round, $dimCadres['Y'] - 4, true, 0, 0, false, 0, 'M', false);
			$pdf->MultiCell($dimCadres['R'], $tab_hl + 1, $outputlangs->transnoentities($ndf ? "TripNDF" : "BillTo").' : ', '', 'L', 0, 1, $dimCadres['xR'] + $decal_round, $dimCadres['Y'] - 4, true, 0, 0, false, 0, 'M', false);
		}	// if (! $hide_recep_frame)
		// Show sender frame
		$pdf->SetAlpha($frmeAlpha / 100);
		$frme	= (implode(',', $frmeLineColor) == '255, 255, 255' ? '' : 'D').( implode(',', $frmeBgColor) == '255, 255, 255' ? '' : 'F');
		$pdf->RoundedRect($dimCadres['xS'], $dimCadres['Y'], $dimCadres['S'], $hauteurcadre, $Rounded_rect, '1111', $frme, $frmeLineStyle, $frmeBgColor);
		// Show recipient frame
		$pdf->SetAlpha($frmrAlpha / 100);
		$frmr	= (implode(',', $frmrLineColor) == '255, 255, 255' ? '' : 'D').( implode(',', $frmrBgColor) == '255, 255, 255' ? '' : 'F');
		$pdf->RoundedRect($dimCadres['xR'], $dimCadres['Y'], $dimCadres['R'], $hauteurcadre, $Rounded_rect, '1111', $frmr, $frmrLineStyle, $frmrBgColor);
		$pdf->SetAlpha(1);
		pdf_InfraSPlus_writeFrame($pdf, $outputlangs, $default_font_size, $tab_hl, $dimCadres, $emetteur, $addresses);
		return $hauteurcadre;
	}	// function pdf_InfraSPlus_writeAddresses(&$pdf, $object, $outputlangs, $formatpage, $dimCadres, $tab_hl, $emetteur, $addresses, $Rounded_rect, $ndf = false)

	/************************************************
	*	Show top small header of page.
	*
	*	@param	TCPDF		$pdf				The PDF factory
	*	@param  Translate	$outputlangs		Object lang for output
	*	@param	int			$default_font_size	Font size
	* 	@param	int			$tab_hl				Line height
	* 	@param	array		$dimCadres			Frame dimensions => 'R', 'S', 'Y', 'xS', 'xR'
	* 	@param	object		$emetteur			Object company
	* 	@param	array		$addresses			All addresses found
	* 	@param	float		return frame height
	************************************************/
	function pdf_InfraSPlus_writeFrame(&$pdf, $outputlangs, $default_font_size, $tab_hl, $dimCadres, $emetteur, $addresses, $ticket = 0)
	{
		global $conf;

		$frmeTxtColor							= isset($conf->global->INFRASPLUS_PDF_FRM_E_TEXT_COLOR)	? $conf->global->INFRASPLUS_PDF_FRM_E_TEXT_COLOR	: '';
		$frmeTxtColor							= explode(',', $frmeTxtColor);
		$frmrTxtColor							= isset($conf->global->INFRASPLUS_PDF_FRM_R_TEXT_COLOR)	? $conf->global->INFRASPLUS_PDF_FRM_R_TEXT_COLOR	: '';
		$frmrTxtColor							= explode(',', $frmrTxtColor);
		$statusWithName							= ! empty($conf->global->INFRASPLUS_PDF_SHOW_STATUS_WITH_SENDER_NAME) && $emetteur->forme_juridique_code ? ' '.$outputlangs->convToOutputCharset(getFormeJuridiqueLabel($emetteur->forme_juridique_code)) : '';
		// Show sender
		$posy									= $dimCadres['Y'];
		if (empty($ticket))						$pdf->SetTextColor($frmeTxtColor[0], $frmeTxtColor[1], $frmeTxtColor[2]);
		// Show sender name
		$pdf->SetFont('', 'B', $default_font_size - ($ticket ? 2 : 0));
		$emetteurName							= ($ticket ? $outputlangs->transnoentities("BillFrom").' : ' : '').($addresses['sender_Alias'] ? $addresses['sender_Alias'] : $emetteur->name);
		$pdf->MultiCell($dimCadres['S'] - 4, $tab_hl, $outputlangs->convToOutputCharset($emetteurName).$statusWithName, '', 'L', 0, 1, $dimCadres['xS'] + 2, $posy + 1, true, 0, 0, false, 0, 'M', false);
		$posy									= $pdf->getY();
		// Show sender information
		$pdf->SetFont('', '', $default_font_size - ($ticket ? 3 : 1));
		$pdf->MultiCell($dimCadres['S'] - 4, $tab_hl, $addresses['carac_emetteur'], '', 'L', 0, 1, $dimCadres['xS'] + 2, $posy, true, 0, 0, false, 0, 'M', false);
		$posyendsender							= $pdf->getY();
		//Show Recipient
		if (empty($ticket))						$posy	= $dimCadres['Y'];
		else									$posy	= $posyendsender;
		if (empty($ticket))						$pdf->SetTextColor($frmrTxtColor[0], $frmrTxtColor[1], $frmrTxtColor[2]);
		// Show recipient name
		$pdf->SetFont('', 'B', $default_font_size - ($ticket ? 2 : 0));
		$pdf->MultiCell($dimCadres['R'] - 4, $tab_hl, ($ticket ? $outputlangs->transnoentities("BillTo").' : ' : '').$addresses['carac_client_name'], '', 'L', 0, 1, $dimCadres['xR'] + 2, $posy + 1, true, 0, 0, false, 0, 'M', false);
		$posy									= $pdf->getY();
		// Show recipient information
		$pdf->SetFont('', '', $default_font_size - ($ticket ? 3 : 1));
		$pdf->MultiCell($dimCadres['R'] - 4, $tab_hl, $addresses['carac_client'], '', 'L', 0, 1, $dimCadres['xR'] + 2, $posy, true, 0, 0, false, 0, 'M', false);
		$posyendrecipient						= $pdf->getY();
		if ($posyendsender > $posyendrecipient)	return ($posyendsender - $dimCadres['Y']) + 1;
		else									return ($posyendrecipient - $dimCadres['Y']) + 1;
	}	// function pdf_InfraSPlus_writeFrame(&$pdf, $outputlangs, $default_font_size, $tab_hl, $dimCadres, $emetteur, $addresses, $ticket = 0)

	/************************************************
	*	Show top small header of page.
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Object		$object     	Object shown in PDF
	*	@param  int	    	$showaddress    0=no, 1=yes
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	string		$title			string title in connection with objet type
	* 	@param	Societe		$fromcompany	Object company
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	* 	@param	int			$decal_round	décalage en fonction du rayon des angles du tableau
	* 	@param	string		$logo			Objet logo to show
	* 	@param	array		$txtcolor		Text color
	************************************************/
	function pdf_InfraSPlus_pagesmallhead(&$pdf, $object, $showaddress, $outputlangs, $title, $fromcompany, $formatpage, $decal_round, $logo, $txtcolor = array(0, 0, 0))
	{
		global $conf;

		$default_font_size	= pdf_getPDFFontSize($outputlangs);
		$pdf->SetTextColor($txtcolor[0], $txtcolor[1], $txtcolor[2]);
		$pdf->SetFont('','', $default_font_size - 2);
		$posy				= $formatpage['mhaute'];
		$posx				= $formatpage['largeur'] - $formatpage['mdroite'] - 100 - $decal_round;
		// Logo
		if ($logo)			$logo	= $conf->mycompany->dir_output.'/logos/'.$logo;
		else				$logo	= $conf->mycompany->dir_output.'/logos/'.$fromcompany->logo;
		if ($logo)
		{
			if (is_file($logo) && is_readable($logo))	$pdf->Image($logo, $formatpage['mgauche'], $posy, 0, 6);	// width=0 (auto)
			else
			{
				$pdf->SetTextColor(200, 0 ,0);
				$pdf->SetFont('', 'B', $default_font_size - 2);
				$pdf->MultiCell(100, 4, $outputlangs->transnoentities("PDFInfraSPlusLogoFileNotFound", $logo), '', 'L', 0, 1, $formatpage['mgauche'], $posy, true, 0, 0, false, 0, 'M', false);
				$pdf->MultiCell(100, 4, $outputlangs->transnoentities("ErrorGoToGlobalSetup"), '', 'L', 0, 1, $formatpage['mgauche'], $posy + 8, true, 0, 0, false, 0, 'M', false);
				$pdf->SetTextColor($txtcolor[0], $txtcolor[1], $txtcolor[2]);
			}	// else	// if (is_file($logo) && is_readable($logo))
		}	// if ($logo)
		else
		{
			$text	= $fromcompany->name;
			$pdf->MultiCell(100, 4, $outputlangs->convToOutputCharset($text), '', 'L', 0, 1, $formatpage['mgauche'], $posy, true, 0, 0, false, 0, 'M', false);
		}	// else	// if ($logo)
		$posy				+= 3;
		pdf_InfraSPlus_pagesrefdate($pdf, $object, $outputlangs, $title, $posy, $posx);
	}	// 	function pdf_InfraSPlus_pagesmallhead(&$pdf, $object, $showaddress, $outputlangs, $title, $fromcompany, $formatpage, $decal_round, $logo, $txtcolor = array(0, 0, 0))

	/************************************************
	*	Show free text
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Object		$object         Object shown in PDF
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	*	@param	int			$posx			Position depart (largeur)
	*	@param	int			$posy			Position depart (hauteur)
	*	@param	Translate	$outputlangs	Objet langs
	* 	@param	Societe		$fromcompany	Object company
	* 	@param	array		$listfreetext	Root of constant names of free text
	*	@param	int			$withupline		Trace une ligne au-dessus du texte sur la largeur
	*	@param	int			$calculseul		Arrête la fonction au calcul de hauteur nécessaire
	*	@param	array		$LineStyle		PDF Line style
	* 	@return	int							Return height of free text
	************************************************/
	function pdf_InfraSPlus_free_text(&$pdf, $object, $formatpage, $posx, $posy, $outputlangs, $fromcompany, $listfreetext, $withupline = 0, $calculseul = 0, $LineStyle = null)
	{
		global $db, $conf;

		$pdf->startTransaction();
		$posy0			= $posy;
		$line			= '';
		if ($listfreetext && $listfreetext != 'None')
		{
			foreach ($listfreetext as $freeT)
			{
				$freetext	= isset($conf->global->$freeT) ? $conf->global->$freeT : '';
				// Line of free text
				if (! empty($freetext))
				{
					// Make substitution
					$substitutionarray						= pdf_getSubstitutionArray($outputlangs, null, $object);
					// More substitution keys
					$substitutionarray['__FROM_NAME__']		= $fromcompany->name;
					$substitutionarray['__FROM_EMAIL__']	= $fromcompany->email;
					complete_substitutions_array($substitutionarray, $outputlangs, $object);
					$newfreetext							= make_substitutions($freetext, $substitutionarray, $outputlangs);
					$newfreetext							= preg_replace('/(<img.*src=")[^\"]*viewimage\.php[^\"]*modulepart=medias[^\"]*file=([^\"]*)("[^\/]*\/>)/', '\1'.DOL_DATA_ROOT.'/medias/\2\3', $newfreetext);
					$line									.= $line ? '<br />'.$outputlangs->convToOutputCharset($newfreetext) : $outputlangs->convToOutputCharset($newfreetext);
				}	// if (! empty($freetext))
			}	// foreach ($listfreetext as $freeT)
		}	// if ($listfreetext && $listfreetext != 'None')
		if ($line)	// Free text
		{
			$default_font_size	= pdf_getPDFFontSize($outputlangs);
			$pdf->SetFont('', '', $default_font_size - 2);
			if ($withupline)
			{
				$posy	+= 0.5;
				$pdf->Line($posx, $posy, $formatpage['largeur'] - $formatpage['mdroite'], $posy, $LineStyle);
			}	// if ($withupline)
			$posy	+= 0.5;
			$pdf->writeHTMLCell(0, 3, $posx, $posy, dol_htmlentitiesbr($line), 0, 1);
			$posy	= $pdf->GetY() + 1;
		}	// if ($line)
		if ($calculseul)
		{
			$heightforfreetext	= ($posy - $posy0);
			$pdf->rollbackTransaction(true);
			return $heightforfreetext;
		}	// if ($calculseul)
		else
		{
			$pdf->commitTransaction();
			return $posy;
		}	// else // if ($calculseul)
	}	// function pdf_InfraSPlus_free_text(&$pdf, $object, $formatpage, $posx, $posy, $outputlangs, $fromcompany, $paramfreetext, $withupline = 0, $calculseul = 0, $LineStyle = null)

	/********************************************
	*	Show notes
	*
	*	@param		TCPDF		$pdf				The PDF factory
	*	@param		Object		$object				Object shown in PDF
	* 	@param		array		$listnotep			Root of constant names of standard public notes
	*	@param		Translate	$outputlangs		Object lang for output
	*	@param		array		$exftxtcolor		text color values (RGB)
	*	@param		int			$default_font_size	font size value
	*	@param		int			$tab_top			height for top page (header)
	*	@param		int			$larg_util_txt		note width
	*	@param		int			$tab_hl				line height
	*	@param		int			$posx_G_txt			x position for the notes
	*	@param		array		$horLineStyle		params for horizontale line style
	*	@param		int			$usedSpace			used space height
	*	@param		int			$page_hauteur		page width
	*	@param		int			$Rounded_rect		radius value for rounded corner
	*	@param		boolean		$showtblline		Show note frame
	*	@param		int			$marge_gauche		left margin
	*	@param		int			$larg_util_cadre	table width
	*	@param		array		$tblLineStyle		params for table line style
	*	@param		int			$typeNotes			type of notes to show :	0 => standard (sales representative + public notes + extrafields)
	*																		-1 => short (public notes + extrafields)
	*																		1 => extended (sales representative + public notes + extrafields + serial number for equipement)
	*	@param		boolean		$firstpageempty		If we insert an empty page first we need to change the condition to check the pitch of the notes
	*	@return		int								Return height of notes
	********************************************/
	function pdf_InfraSPlus_Notes(&$pdf, $object, $listnotep, $outputlangs, $exftxtcolor, $default_font_size, $tab_top, $larg_util_txt, $tab_hl, $posx_G_txt, $horLineStyle, $usedSpace, $page_hauteur, $Rounded_rect, $showtblline, $marge_gauche, $larg_util_cadre, $tblLineStyle, $typeNotes = 0, $firstpageempty = 0)
	{
		global $conf;

		$show_sales_rep_in_notes	= isset($conf->global->INFRASPLUS_PDF_FIRST_SALES_REP_IN_NOTE)	? $conf->global->INFRASPLUS_PDF_FIRST_SALES_REP_IN_NOTE	: 0;
		switch ($object->element)
		{
			case 'propal':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_D)	? $conf->global->INFRASPLUS_PDF_EXF_D	: 0;
			break;
			case 'commande':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_C)	? $conf->global->INFRASPLUS_PDF_EXF_C	: 0;
			break;
			case 'contrat':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_CT)	? $conf->global->INFRASPLUS_PDF_EXF_CT	: 0;
			break;
			case 'fichinter':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_FI)				? $conf->global->INFRASPLUS_PDF_EXF_FI				: 0;
				$lastNoteAsTable			= isset($conf->global->INFRASPLUS_PDF_LAST_NOTE_AS_TABLE)	? $conf->global->INFRASPLUS_PDF_LAST_NOTE_AS_TABLE	: 0;
			break;
			case 'shipping':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_E)	? $conf->global->INFRASPLUS_PDF_EXF_E	: 0;
			break;
			case 'facture':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_F)	? $conf->global->INFRASPLUS_PDF_EXF_F	: 0;
			break;
			case 'supplier_proposal':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_DF)	? $conf->global->INFRASPLUS_PDF_EXF_DF	: 0;
			break;
			case 'order_supplier':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_CF)	? $conf->global->INFRASPLUS_PDF_EXF_CF	: 0;
			break;
			case 'invoice_supplier':
				$show_ExtraFields_in_notes	= isset($conf->global->INFRASPLUS_PDF_EXF_FF)	? $conf->global->INFRASPLUS_PDF_EXF_FF	: 0;
			break;
			default:
				$show_ExtraFields_in_notes	= 0;
			break;
		}	// switch ($object->element)
		$height_note	= 0;
		$salesrep		= ! empty($show_sales_rep_in_notes) && $typeNotes > -1 ? pdf_InfraSPlus_SalesRepInNotes($object, $outputlangs) : '';
		if ($listnotep && $listnotep != 'None')
		{
			$notesptoshow	= array();
			foreach ($listnotep as $noteP)
			{
				$notePub				= isset($conf->global->$noteP) ? $conf->global->$noteP : '';
				if (! empty($notePub))	$notesptoshow[]	= pdf_InfraSPlus_formatNotes($object, $outputlangs, $notePub);
			}	// foreach ($listnotep as $noteP)
		}	// iif ($listnotep && $listnotep != 'None')
		$notetoshow		= ! empty($object->note_public) && empty($lastNoteAsTable) ? pdf_InfraSPlus_formatNotes($object, $outputlangs, $object->note_public) : '';
		$extraDet		= ! empty($show_ExtraFields_in_notes) ? pdf_InfraSPlus_ExtraFieldsInNotes($object, $exftxtcolor) : '';
		$serialEquip	= $typeNotes > 0 ? pdf_InfraSPlus_getEquipementSerialDesc($object, $outputlangs, 0, 'intervention') : '';
		if ($salesrep || !empty($notesptoshow) || $notetoshow || $extraDet || $serialEquip)
		{
			$pdf->SetFont('', '', $default_font_size - 1);
			$pdf->startTransaction();
			$nexY				= $tab_top;
			if ($firstpageempty)
			{
				$pdf->line($posx_G_txt, $nexY + 1, $posx_G_txt + $larg_util_txt, $nexY + 1, $horLineStyle);
				$nexY	+= 2;
			}	// if ($firstpageempty)
			if ($salesrep)		$nexY	= pdf_InfraSPlus_writeNotes($pdf, $larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $salesrep, $horLineStyle, (!empty($notesptoshow) || $notetoshow || $extraDet || $serialEquip ? 1 : 0));
			if (!empty($notesptoshow))
				foreach ($notesptoshow as $noteptoshow)	$nexY	= pdf_InfraSPlus_writeNotes($pdf, $larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $noteptoshow, $horLineStyle, (next($notesptoshow) !== FALSE || $notetoshow || $extraDet || $serialEquip ? 1 : 0));
			if ($notetoshow)	$nexY	= pdf_InfraSPlus_writeNotes($pdf, $larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $notetoshow, $horLineStyle, ($extraDet || $serialEquip ? 1 : 0));
			if ($extraDet)		$nexY	= pdf_InfraSPlus_writeNotes($pdf, $larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $extraDet, $horLineStyle, ($serialEquip ? 1 : 0));
			if ($serialEquip)	$nexY	= pdf_InfraSPlus_writeNotes($pdf, $larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $serialEquip, $horLineStyle, 0);
			if ($pdf->getPage() > ($firstpageempty ? 2 : 1) || $pdf->GetY() > ($page_hauteur - (($tab_hl * 4) + $usedSpace))) // Notes need pagebreak or There is no space left for footer
			{
				$pdf->rollbackTransaction(true);
				$pdf->writeHTMLCell($larg_util_txt, $tab_hl, $posx_G_txt, $tab_top, dol_htmlentitiesbr($outputlangs->transnoentities("PDFInfraSPlusNoteTooLong")), 0, 1);
			}	// if ($pdf->getPage() > ($firstpageempty ? 2 : 1) || $pdf->GetY() > ($page_hauteur - (($tab_hl * 4) + $usedSpace)))
			else				$pdf->commitTransaction();
			$nexY				= $pdf->GetY();
			$height_note		= $Rounded_rect * 2 > $nexY - $tab_top ? $Rounded_rect * 2 : $nexY - $tab_top;
			if ($showtblline)	$pdf->RoundedRect($marge_gauche, $tab_top - 1, $larg_util_cadre, $height_note + 2, $Rounded_rect, '1111', null, $tblLineStyle);
			$height_note		+= $tab_hl * 1.5;
		}	// if ($salesrep || !empty($notesptoshow) || $notetoshow || $extraDet || $serialEquip)
		return $height_note;
	}	// function pdf_InfraSPlus_Notes(&$pdf, $object, $listnotep, $outputlangs, $exftxtcolor, $default_font_size, $tab_top, $larg_util_txt, $tab_hl, $posx_G_txt, $horLineStyle, $usedSpace, $page_hauteur, $Rounded_rect, $showtblline, $marge_gauche, $larg_util_cadre, $tblLineStyle, $typeNotes = 0, $firstpageempty = 0)

	/********************************************
	*	Write notes
	*
	*	@param		TCPDF		$pdf				The PDF factory
	*	@param		int			$larg_util_txt		note width
	*	@param		int			$tab_hl				line height
	*	@param		int			$posx_G_txt			x position for the notes
	*	@param		int			$nexY				y position to start
	*	@param		string		$notes				notes to write
	*	@param		array		$horLineStyle		params for horizontale line style
	*	@param		boolean		$hasNextVal			Some more notes exist
	*	@return		int								Return Y value for the next position
	********************************************/
	function pdf_InfraSPlus_writeNotes($pdf, $larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $notes, $horLineStyle, $hasNextVal = 0)
	{
		$pdf->writeHTMLCell($larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $notes, 0, 1);
		$nexY	= $pdf->GetY();
		if ($hasNextVal)
		{
			$pdf->line($posx_G_txt + 30, $nexY + 2, $posx_G_txt + $larg_util_txt - 30 , $nexY + 2, $horLineStyle);
			$nexY	= $pdf->GetY() + 4;
		}	// if ($hasNextVal)
		return $nexY;
	}	// function pdf_InfraSPlus_writeNotes($pdf, $larg_util_txt, $tab_hl, $posx_G_txt, $nexY, $notes, $horLineStyle, $hasNextVal = 0)

	/********************************************
	*	Get sales representative
	*
	*	@param		Object		$object			Object shown in PDF
	*	@param		Translate	$outputlangs	Object lang for output
	*	@return		String						Return sales representative with details if found
	********************************************/
	function pdf_InfraSPlus_SalesRepInNotes($object, $outputlangs)
	{
		global $db;

		$salesrep		= '';
		$arrayidcontact	= $object->getIdContact('internal', 'SALESREPFOLL');
		if (count($arrayidcontact) > 0)
		{
			$tmpuser					= new User($db);
			$tmpuser->fetch($arrayidcontact[0]);
			$salesrep					.= $outputlangs->transnoentities("PDFInfraSPlusAffSuivie").' <b>'.$tmpuser->getFullName($outputlangs).'</b>';
			if ($tmpuser->email) 		$salesrep	.= ', '.$outputlangs->transnoentities("Email").' : '.$outputlangs->convToOutputCharset($tmpuser->email);
			if ($tmpuser->office_phone)	$salesrep	.= ', '.$outputlangs->transnoentities("PhoneShort").' : '.$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($tmpuser->office_phone)));
		}	// if (count($arrayidcontact) > 0)
		return $salesrep;
	}	// function pdf_InfraSPlus_SalesRepInNotes($object, $outputlangs)

	/********************************************
	*	Format notes with substitutions and right path for pictures
	*
	*	@param		Object		$object			Object shown in PDF
	*	@param		Translate	$outputlangs	Object lang for output
	*	@param		String		$notes			html string from data base
	*	@return		String						Return html string ready to print
	********************************************/
	function pdf_InfraSPlus_formatNotes($object, $outputlangs, $notes)
	{
		global $dolibarr_main_url_root;

		$substitutionarray	= pdf_getSubstitutionArray($outputlangs, null, $object);
		complete_substitutions_array($substitutionarray, $outputlangs, $object);
		$html				= make_substitutions($notes, $substitutionarray, $outputlangs);		
		// the code below came from a Dolibarr v10 native function (convertBackOfficeMediasLinksToPublicLinks()) on functions2.lib.php
		$urlwithouturlroot	= preg_replace('/'.preg_quote(DOL_URL_ROOT, '/').'$/i', '', trim($dolibarr_main_url_root));	// Define $urlwithroot
		$urlwithroot		= $urlwithouturlroot.DOL_URL_ROOT;		// This is to use external domain name found into config file
		$html				= preg_replace('/src="[a-zA-Z0-9_\/\-\.]*(viewimage\.php\?modulepart=medias[^"]*)"/', 'src="'.$urlwithroot.'/\1"', $html);
		return $html;
	}	// function pdf_InfraSPlus_formatNotes($object, $outputlangs, $notes)

	/********************************************
	*	Get document extrafields
	*
	*	@param		Object		$object			Object shown in PDF
	*	@param		array		$exftxtcolor	array with rgb color code
	*	@return		String						Return extrafields found
	********************************************/
	function pdf_InfraSPlus_ExtraFieldsInNotes($object, $exftxtcolor)
	{
		global $db, $conf;

		$extraDet		= '';
		$extrafields	= new ExtraFields($db);
		$extralabels	= $extrafields->fetch_name_optionals_label($object->table_element);
		$object->fetch_optionals();
		foreach ($extralabels as $key => $label)
		{
			// check key prefix to make a difference between extrafields we want to print and some we just need to see
			$prefix															= isset($conf->global->INFRASPLUS_PDF_EXF_PRE) ? $conf->global->INFRASPLUS_PDF_EXF_PRE : '';
			if ($prefix && substr($key, 0, strlen($prefix)) !== $prefix)	continue;
			$options_key													= $object->array_options["options_" . $key];
			$value															= $extrafields->showOutputField($key, $options_key);
			if (preg_match('#<img.*src=.*\/>#', $value))					$value	= preg_replace('#src=\"\/viewimage.*modulepart=#', 'src="'.DOL_DATA_ROOT.'/', preg_replace('#&amp;entity=[0-9]*&amp;file=#', '/', $value));
			if (! empty($value)) // check if something is writting for this extrafield else we avoid to print an empty value
			{
				$value		= '<span style = "color: rgb('.$exftxtcolor[0].', '.$exftxtcolor[1].', '.$exftxtcolor[2].')">'.$value.'</span>';
				$extraDet	.= (empty($extraDet) ? '<ul><li>' : '<li>').$label.' : <b>'.$value.'</b></li>';
			}	// if (! empty($value))
		}	// foreach ($extralabels as $key => $label)
		$extraDet	.= empty($extraDet) ? '' : '</ul>';
		return $extraDet;
	}	// function pdf_InfraSPlus_ExtraFieldsInNotes($object, $exftxtcolor)

	/********************************************
	*	Get serial Number for equipement
	*
	*	@param		Object		$object			Object shown in PDF
	*	@param		Translate	$outputlangs	Object lang for output
	*	@param		int			$i				Row number.
	*	@param		String		$typedoc		type of document asking.
	*	@return		String						Return equipement serial number
	********************************************/
	function pdf_InfraSPlus_getEquipementSerialDesc($object, $outputlangs, $i, $typedoc = '')
	{
		global $db;

		$idprod	= (!empty($object->lines[$i]->fk_product) ? $object->lines[$i]->fk_product : false);
		$space	= '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;';
		$retStr	= '';
		if ($idprod || $typedoc == 'intervention')
		{
			if ($typedoc == 'expedition')
			{
				$sql	= 'SELECT eq.ref';
				$sql	.= ' FROM '.MAIN_DB_PREFIX.'equipement AS eq,'.MAIN_DB_PREFIX.'equipementevt AS eqv';
				$sql	.= ' WHERE eq.rowid = eqv.fk_equipement';
				$sql	.= ' AND eqv.fk_expedition = "'.$object->id.'"';
				$sql	.= ' AND eq.fk_product = "'.$idprod.'"';
			}	// if ($typedoc == 'expedition')
			elseif ($typedoc == 'facture')
			{
				$sql	= 'SELECT eq.ref';
				$sql	.= ' FROM '.MAIN_DB_PREFIX.'equipement AS eq';
				$sql	.= ' WHERE eq.fk_facture = "'.$object->id.'"';
				$sql	.= ' AND eq.fk_product = "'.$idprod.'"';
			}	// elseif ($typedoc == 'facture')
			elseif ($typedoc == 'intervention')
			{
				$sql	= 'SELECT eq.ref, p.ref as refproduct';
				$sql	.= ' FROM '.MAIN_DB_PREFIX.'equipement AS eq,'.MAIN_DB_PREFIX.'equipementevt AS eqv,'.MAIN_DB_PREFIX.'product AS p';
				$sql	.= ' WHERE eq.rowid = eqv.fk_equipement';
				$sql	.= ' AND p.rowid = eq.fk_product';
				$sql	.= ' AND eqv.fk_fichinter = "'.$object->id.'"';
				$sql	.= ' ORDER BY eq.fk_product';
			}	// elseif ($typedoc == 'facture')
			else return	$retStr;
			$result	= $db->query($sql);
			if ($result)
			{
				$num = $db->num_rows($result);
				if ($num>0)
				{
					$retStr	= $outputlangs->trans("PDFInfraSPlusSerialRef").' = '.($num > 1 ? '<br/>' : '');
					for ($i = 0; $i < $num; $i++)
					{
						$objp							= $db->fetch_object($result);
						if ($typedoc == 'intervention')	$retStr	.= ($num > 1 ? ($i == 0 ? $space : '<br/>'.$space) : '&nbsp;').$outputlangs->trans("PDFInfraSPlusFicheInterSerialNum", $objp->refproduct, $objp->ref);
						else							$retStr	.= ($num > 1 ? ($i == 0 ? $space : '<br/>'.$space) : '&nbsp;').$objp->ref;
					}	// for ($i = 0; $i < $num; $i++)
				}	// if ($num>0)
			} else	$retStr	= '';
		}	// if ($idprod || $typedoc == 'intervention')
		return	$retStr;
	}	// function pdf_InfraSPlus_getEquipementSerialDesc($object, $outputlangs, $i, $typedoc)

	/********************************************
	*	Return line product ref for Intervention card
	*
	*	@param	Object	$object		Object
	*	@param	int		$i			Current line number
	* 	@return	array
	 ********************************************/
	function pdf_infrasplus_getlinefichinter($object, $i)
	{
		global $db;

		$prodfichinter	= array();
		$sql	= 'SELECT fid.total_ht, fid.subprice, fid.fk_product, fid.tva_tx, fid.localtax1_tx, fid.localtax1_type, fid.localtax2_tx, fid.localtax2_type, fid.qty,';
		$sql	.= ' fid.remise_percent, fid.remise, fid.fk_remise_except, fid.price, fid.total_tva, fid.total_localtax1, fid.total_localtax2, fid.total_ttc,';
		$sql	.= ' fid.product_type, fid.info_bits, fid.buy_price_ht, fid.fk_product_fournisseur_price, p.ref, p.label';
		$sql	.= ' FROM '.MAIN_DB_PREFIX.'fichinterdet AS fid';
		$sql	.= ' LEFT JOIN '.MAIN_DB_PREFIX.'product AS p ON fid.fk_product = p.rowid';
		$sql	.= ' WHERE fid.fk_fichinter = '.$object->id.' AND fid.rowid = '.$object->lines[$i]->id;
		$resql	= $db->query($sql);
		if ($resql)
		{
			$num	= $db->num_rows($resql);
			$j		= 0;
			while ($j < $num)
			{
				$objp											= $db->fetch_object($resql);
				$prodfichinter['total_ht']						= $objp->total_ht;
				$prodfichinter['subprice']						= $objp->subprice;
				$prodfichinter['fk_product']					= $objp->fk_product;
				$prodfichinter['tva_tx']						= $objp->tva_tx;
				$prodfichinter['localtax1_tx']					= $objp->localtax1_tx;
				$prodfichinter['localtax1_type']				= $objp->localtax1_type;
				$prodfichinter['localtax2_tx']					= $objp->localtax2_tx;
				$prodfichinter['localtax2_type']				= $objp->localtax2_type;
				$prodfichinter['qty']							= $objp->qty;
				$prodfichinter['remise_percent']				= $objp->remise_percent;
				$prodfichinter['remise']						= $objp->remise;
				$prodfichinter['fk_remise_except']				= $objp->fk_remise_except;
				$prodfichinter['price']							= $objp->price;
				$prodfichinter['total_tva']						= $objp->total_tva;
				$prodfichinter['total_localtax1']				= $objp->total_localtax1;
				$prodfichinter['total_localtax2']				= $objp->total_localtax2;
				$prodfichinter['total_ttc']						= $objp->total_ttc;
				$prodfichinter['product_type']					= $objp->product_type;
				$prodfichinter['info_bits']						= $objp->info_bits;
				$prodfichinter['buy_price_ht']					= $objp->buy_price_ht;
				$prodfichinter['fk_product_fournisseur_price']	= $objp->fk_product_fournisseur_price;
				$prodfichinter['ref']							= $objp->ref;
				$prodfichinter['label']							= $objp->label;
				$j++;
			}	// while ($j < $num)
			$db->free($resql);
		}	// if ($resql)
		return $prodfichinter;
	}	// function pdf_infrasplus_getlinefichinter($object, $i)

	/********************************************
	*	Return dimensions to use for images onto PDF,
	*	checking that width and height are not higher than maximum (20x32 by default).
	*
	*	@param	string		$realpath			Full path to photo file to use
	*	@param	int			$maxwidth			Maximum width to use
	*	@param	int			$maxheight			Maximum height to use
	*	@return	array							Height and width to use to output image (in pdf user unit, so mm)
	 ********************************************/
	function pdf_InfraSPlus_getSizeForImage($realpath, $maxwidth, $maxheight)
	{
		global $conf;

		include_once DOL_DOCUMENT_ROOT.'/core/lib/images.lib.php';
		$imglinesize	= dol_getImageSize($realpath);
		if ($imglinesize['height'])
		{
			$width	= (int) round($maxheight * $imglinesize['width'] / $imglinesize['height']);	// I try to use maxheight
			if ($width > $maxwidth)	// Pb with maxheight, so i use maxwidth
			{
				$width	= $maxwidth;
				$height	= (int) round($maxwidth * $imglinesize['height'] / $imglinesize['width']);
			}	// if ($width > $maxwidth)
			else	$height	= $maxheight;	// No pb with maxheight
		}	// if ($imglinesize['height'])
		return array('width'=>$width, 'height'=>$height);
	}	// function pdf_InfraSPlus_getSizeForImage($realpath, $maxwidth, $maxheight)

	/********************************************
	*  Return line url
	*
	*  @param  Object		$object			Object shown in PDF
	*  @param  int			$i				Current line number (0 = first line, 1 = second line, ...)
	*  @return string						url found in product/service data
	 ********************************************/
	function pdf_InfraSPlus_getlineurl(&$object, $i)
	{
		global $db;

		$idprod		= (! empty($object->lines[$i]->fk_product) ? $object->lines[$i]->fk_product : false);
		$prodser	= new Product($db);
		$prodser->fetch($idprod);

		return $prodser->url;
	}	// function pdf_InfraSPlus_getlineurl(&$object, $i)

	/********************************************
	*	Return line product ref
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string
	 ********************************************/
	function pdf_infrasplus_getlineref($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $hookmanager;

		$reshook	= 0;
		$result		= '';
		$refprod	= $prodfichinter ? $prodfichinter['ref'] : $object->lines[$i]->product_ref;
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlineref', $parameters,$object, $action);    // Note that $action and $object may have been modified by some hooks
			$result												.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
			if (empty($hidedetails) || $hidedetails > 1)	$result	.= dol_htmlentitiesbr($refprod);
		return $result;
	}	// function pdf_infrasplus_getlineref($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/************************************************
	*	Return line ref_supplier
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string
	************************************************/
	function pdf_InfraSPlus_getlineref_supplier($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $db, $hookmanager;

		if (empty($object->lines[$i]->fk_product))	$ref_supplier	= 'none';
		else										$ref_supplier	= (! empty($object->lines[$i]->ref_supplier) ? $object->lines[$i]->ref_supplier : (! empty($object->lines[$i]->ref_fourn) ? $object->lines[$i]->ref_fourn : ''));
		if (empty($ref_supplier))
		{
			$ref									= $object->lines[$i]->ref;
			$sql									= 'SELECT pfp.ref_fourn ';
			$sql									.= 'FROM '.MAIN_DB_PREFIX.'product AS p ';
			$sql									.= 'LEFT JOIN '.MAIN_DB_PREFIX.'product_fournisseur_price AS pfp ON p.rowid = pfp.fk_product ';
			$sql									.= 'WHERE p.ref = "'.$ref.'" AND pfp.fk_soc = "'.$object->thirdparty->id.'"';
			$resql									= $object->db->query($sql);
			if ($resql)
			{
				$obj								= $db->fetch_object($resql);
				if ($obj)							$ref_supplier	= $obj->ref_fourn;
			}	// if ($resql)
			$db->free($resql);
		}	// if (empty($ref_supplier))
		if ($ref_supplier == 'none')				$ref_supplier	= '';
		$reshook									= 0;
		$result										= '';
		if (is_object($hookmanager))
		{
			$special_code							= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))
				$special_code						= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters								= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code);
			$action									= '';
			$reshook								= $hookmanager->executeHooks('pdf_getlineref_supplier', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			$result									.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))						$result			.= dol_htmlentitiesbr($ref_supplier);
		return $result;
	}	// function pdf_InfraSPlus_getlineref_supplier($object, $i, $outputlangs, $hidedetails = 0)

	/********************************************
	*	Output line product bar code
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param	Object		$object			Object
	*	@param	int			$i				Current line number or -1 if we want to print the object bar code instead of line object bar code
	*	@param  Array		$bodytxtcolor	current text color
	*	@param	float		$posx			x position
	*	@param	float		$posy			y position
	*	@param	float		$width			bar code width
	*	@param	float		$height			bar code height (for 2D code such as Qr Code width = height and we change the x position to be on the middle of the width)
	* 	@return	Boolean						1 -> Ok ; <1 -> Ko
	 ********************************************/
	 function pdf_InfraSPlus_writelineBC(&$pdf, $object, $i, $bodytxtcolor, $posx, $posy, $width, $height)
	 {
		global $db;

		$pos	= $i > -2 ? 'C' : '';
		$decal	= 0;
		if ($i > -1)
		{
			$idprod			= ! empty($object->lines[$i]->fk_product) ? $object->lines[$i]->fk_product : false;
			$prodser		= new Product($db);
			if ($idprod)	$prodser->fetch($idprod);
			$pos			= '';
			$decal			= ($width - $height) / 2;
		}	// if ($i > -1)
		else	$prodser	= $object;
		if ($prodser->barcode)
		{
			// Complete object if not complete
			if (empty($prodser->barcode_type_code) || empty($prodser->barcode_type_coder))
			{
				$result	= $prodser->fetch_barcode();
				if ($result < 1)	$BC	= '-- ErrorFetchBarcode -- '.$result;	//Check if fetch_barcode() failed
			}	// if (empty($prodser->barcode_type_code) || empty($prodser->barcode_type_coder))
			if (! $BC)
			{
				if ($prodser->barcode_type <= 6)
				{
					require_once(DOL_DOCUMENT_ROOT."/core/lib/barcode.lib.php");
					if ($prodser->barcode_type == 2 && strlen($prodser->barcode) > 12)
					{
						$ean	= substr($prodser->barcode, 0, 12);
						$eansum	= barcode_gen_ean_sum($ean);
						if (substr($prodser->barcode, -1) != $eansum)
						{
							$txtErr	= '<b><FONT size="7">'.$ean.'<FONT color="red">'.substr($prodser->barcode, -1).'</FONT><FONT color="green">('.$eansum.')</FONT></FONT></b>';
							$pdf->writeHTMLCell($width, $height, $posx, $posy, $txtErr, 0, 0, false, true, 'C');
							return 1;
						}	// if (substr($prodser->barcode, -1) != $eansum)
					}	// if ($prodser->barcode_type == 2 && strlen($prodser->barcode) > 12)
					switch ($width)
					{
						case 25:
							$xres	= 0.2;
							break;
						case 35:
							$xres	= 0.3;
							break;
						case 45:
							$xres	= 0.4;
							break;
						default:
							$xres	= 0.2;
					}	// switch ($width)
					$styleBC	= array('position'		=> $pos,
										'align'			=> '',
										'stretch'		=> false,
										'fitwidth'		=> true,
										'cellfitalign'	=> 'C',
										'border'		=> false,
										'hpadding'		=> '0',
										'vpadding'		=> '0',
										'fgcolor'		=> array($bodytxtcolor[0], $bodytxtcolor[1], $bodytxtcolor[2]),
										'bgcolor'		=> false,
										'text'			=> true,
										'label'			=> $prodser->barcode,
										'font'			=> $pdf->getFontFamily(),
										'fontsize'		=> 6,
										'stretchtext'	=> 4
										);
					$pdf->write1DBarcode($prodser->barcode, $prodser->barcode_type_code, $posx, $posy, $width, $height, $xres, $styleBC, 'B');
					return 1;
				}	// if ($prodser->barcode_type <= 6)
				if ($prodser->barcode_type > 6)
				{
					$posx			+= $decal;
					$styleBC		= array('position'		=> $pos,
											'border'		=> false,
											'hpadding'		=> '0',
											'vpadding'		=> '0',
											'fgcolor'		=> array($bodytxtcolor[0], $bodytxtcolor[1], $bodytxtcolor[2]),
											'bgcolor'		=> false,
											'module_width'	=> 1,
											'module_height'	=> 1
											);
					$pdf->write2DBarcode($prodser->barcode, $prodser->barcode_type_code, $posx, $posy, $height, $height, $styleBC, 'B');
					return 2;
				}	// if if ($prodser->barcode_type > 6)
			}	// if (! $BC)
			else $pdf->writeHTMLCell(0, 0, $posx, $posy, $BC, 0, 1);
		}	// if ($prodser->barcode)
		return 0;
	 }	// function pdf_InfraSPlus_writelineBC(&$pdf, $object, $i, $bodytxtcolor, $posx, $posy, $width, $height)

	/********************************************
	*	Get lines extrafields
	*
	*	@param		Object		$line				Line shown in PDF
	*	@param		Object		$extralabelsline	Extra label line in document
	*	@param		array		$exfltxtcolor		array with rgb color code
	*	@return		String							Return extrafields found
	********************************************/
	function pdf_InfraSPlus_ExtraFieldsLines($line, $extrafieldsline, $extralabelsline, $exfltxtcolor)
	{
		global $db, $conf;

		$extraDet		= '';
		$line->fetch_optionals($line->rowid);
		foreach ($extralabelsline as $key => $label)
		{
			$prefix															= isset($conf->global->INFRASPLUS_PDF_EXFL_PRE) ? $conf->global->INFRASPLUS_PDF_EXFL_PRE : '';
			if ($prefix && substr($key, 0, strlen($prefix)) !== $prefix)	continue;
			$options_key													= $line->array_options['options_'.$key];
			$value															= $extrafieldsline->showOutputField($key, $options_key);
			if (preg_match('#<img.*src=.*\/>#', $value))					$value	= preg_replace('#src=\"\/viewimage.*modulepart=#', 'src="'.DOL_DATA_ROOT.'/', preg_replace('#&amp;entity=[0-9]*&amp;file=#', '/', $value));
			if (! empty($value))
			{
				$value		= '<span style = "color: rgb('.$exfltxtcolor[0].', '.$exfltxtcolor[1].', '.$exfltxtcolor[2].')">'.$value.'</span>';
				$extraDet	.= (empty($extraDet) ? '' : '<br/>').$label.' : <b>'.$value.'</b>';
			}	// if (! empty($value))
		}	// foreach ($extralabelsline as $key => $label)
		return $extraDet;
	}	// function pdf_InfraSPlus_ExtraFieldsLines($line, $extralabelsline, $exfltxtcolor)

	/********************************************
	*  Return line weight volume dimensions and Customs code into array
	*
	*  @param  Object		$object			Object shown in PDF
	*  @param  int			$i				Current line number (0 = first line, 1 = second line, ...)
	*  @param  Translate	$outputlangs	Object langs for output
	*  @return array						Array with elements found
	 ********************************************/
	function pdf_InfraSPlus_getlinewvdcc(&$object, $i, $outputlangs)
	{
		global $db, $conf, $langs;
		include_once DOL_DOCUMENT_ROOT.'/core/lib/product.lib.php';

		if ($i === 'P')
		{
			$idprod		= $object->id;
			$type		= $object->fk_product_type;
		}
		else
		{
			$idprod		= (! empty($object->lines[$i]->fk_product) ? $object->lines[$i]->fk_product : false);
			$type		= $object->lines[$i]->product_type;
		}	// else	// if ($i == 'P')
		$prodser	= new Product($db);
		$dimtxt		= '';
		$weighttxt	= '';
		$voltxt		= '';
		$surftxt	= '';
		$ccodetxt	= '';
		$countrytxt	= '';
		if ($idprod && $type == 0)
		{
			$prodser->fetch($idprod);
			if ($prodser->length || $prodser->width || $prodser->height)
			{
				$txtDim	= '';
				$txtDim	.= ($prodser->length ? $langs->trans("Length") : '');
				$txtDim	.= ($prodser->length && ($prodser->width || $prodser->height) ? ' x ' : '');
				$txtDim	.= ($prodser->width ? $langs->trans("Width") : '');
				$txtDim	.= (($prodser->width && $prodser->height) ? ' x ' : '');
				$txtDim	.= ($prodser->height ? $langs->trans("Height") : '');
				$dimtxt	= $txtDim.' : ';
				$txtDim	= ($prodser->length ? $prodser->length : '');
				$txtDim	.= ($prodser->length && ($prodser->width || $prodser->height) ? ' x ' : '');
				$txtDim	.= ($prodser->width ? $prodser->width : '');
				$txtDim	.= (($prodser->width && $prodser->height) ? ' x ' : '');
				$txtDim	.= ($prodser->height ? $prodser->height : '');
				$txtDim	.= ' '.measuring_units_string($prodser->length_units, 'size');
				$dimtxt	.= $txtDim;
			}	// if ($prodser->length || $prodser->width || $prodser->height)
			if ($prodser->weight)		$weighttxt	= $outputlangs->trans("Weight").' : '.$prodser->weight.' '.measuring_units_string($prodser->weight_units, "weight");
			if ($prodser->volume)		$voltxt		= $outputlangs->trans("Volume").' : '.$prodser->volume.' '.measuring_units_string($prodser->volume_units, "volume");
			if ($prodser->surface)		$surftxt	= $outputlangs->trans("Surface").' : '.$prodser->surface.' '.measuring_units_string($prodser->surface_units, "surface");
			if ($prodser->customcode)	$ccodetxt	= $outputlangs->trans("CustomCode").' : '.$prodser->customcode;
			if ($prodser->country_id)	$countrytxt	= $outputlangs->trans("CountryOrigin").' : '.getCountry($prodser->country_id, 0, $db);
		}	// if ($idprod && $object->lines[$i]->product_type == 0)
		$linewvdcc	= '';
		$linewvdcc	= $ccodetxt;
		$linewvdcc	= ($linewvdcc == '' ? $countrytxt	: ($countrytxt == ''	? '' : $linewvdcc.'<br/>'.$countrytxt));
		$linewvdcc	= ($linewvdcc == '' ? $surftxt		: ($surftxt == ''		? '' : $surftxt.'<br/>'.$linewvdcc));
		$linewvdcc	= ($linewvdcc == '' ? $voltxt		: ($voltxt == ''		? '' : $voltxt.'<br/>'.$linewvdcc));
		$linewvdcc	= ($linewvdcc == '' ? $weighttxt	: ($weighttxt == ''		? '' : $weighttxt.'<br/>'.$linewvdcc));
		$linewvdcc	= ($linewvdcc == '' ? $dimtxt		: ($dimtxt == ''		? '' : $dimtxt.'<br/>'.$linewvdcc));
		return $linewvdcc;
	}	// function pdf_InfraSPlus_getlinewvdcc(&$object, $i, $outputlangs)

	/********************************************
	*	Output line description into PDF
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param	Object		$object			Object shown in PDF
	*	@param	int			$i				Current line number
	*	@param  Translate	$outputlangs	Object lang for output
	*	@param  int			$w				Width
	*	@param  int			$h				Height
	*	@param  int			$posx			Pos x
	*	@param  int			$posy			Pos y
	*	@param  int			$hideref       	Hide reference
	*	@param  int			$hidedesc		Hide description
	*	@param	int			$issupplierline	Is it a line for a supplier object ?
	*	@param	int			$extraDet		Module equipement compatibility
	*	@return	string
	 ********************************************/
	function pdf_InfraSPlus_writelinedesc(&$pdf, $object, $i, $outputlangs, $w, $h, $posx, $posy, $hideref = 0, $hidedesc = 0, $issupplierline = 0, $extraDet = '', $prodfichinter = null)
	{
		global $db, $conf, $hookmanager;

		$reshook	= 0;
		$result		= '';
		if ($object->lines[$i]->product_type == 9)	// ligne de sous-titre ou de sous-total
		{
			if ($conf->global->MAIN_MODULE_SUBTOTAL)
			{
				$bodytxtcolor							= isset($conf->global->INFRASPLUS_PDF_BODY_TEXT_COLOR)	? $conf->global->INFRASPLUS_PDF_BODY_TEXT_COLOR		: 0;
				$bodytxtcolor							= explode(',', $bodytxtcolor);
				$bodytxtsubticolor						= isset($conf->global->INFRASPLUS_PDF_TEXT_SUBTI_COLOR)	? $conf->global->INFRASPLUS_PDF_TEXT_SUBTI_COLOR	: 0;
				$bodytxtsubticolor						= explode(',', $bodytxtsubticolor);
				$bodytxtsubtocolor						= isset($conf->global->INFRASPLUS_PDF_TEXT_SUBTO_COLOR)	? $conf->global->INFRASPLUS_PDF_TEXT_SUBTO_COLOR	: 0;
				$bodytxtsubtocolor						= explode(',', $bodytxtsubtocolor);
				if ($object->lines[$i]->qty  < 10)		$pdf->SetTextColor($bodytxtsubticolor[0], $bodytxtsubticolor[1], $bodytxtsubticolor[2]);	// Sous-titre ATM
				else if($object->lines[$i]->qty  > 90)	$pdf->SetTextColor($bodytxtsubtocolor[0], $bodytxtsubtocolor[1], $bodytxtsubtocolor[2]);	// Sous-total ATM
			}	// if ($conf->global->MAIN_MODULE_SUBTOTAL)
			if ($conf->global->MAIN_MODULE_MILESTONE)
			{
				$bodytxtcolor		= isset($conf->global->INFRASPLUS_PDF_BODY_TEXT_COLOR)	? $conf->global->INFRASPLUS_PDF_BODY_TEXT_COLOR		: 0;
				$bodytxtcolor		= explode(',', $bodytxtcolor);
				$bodytxtsubticolor	= isset($conf->global->INFRASPLUS_PDF_TEXT_SUBTI_COLOR)	? $conf->global->INFRASPLUS_PDF_TEXT_SUBTI_COLOR	: 0;
				$bodytxtsubticolor	= explode(',', $bodytxtsubticolor);
				$bodybgsubticolor	= isset($conf->global->MILESTONE_BACKGROUND_COLOR)	? $conf->global->MILESTONE_BACKGROUND_COLOR	: 'e6e6e6';
				$bodybgsubticolor	= colorStringToArray($bodybgsubticolor);
				$pdf->SetTextColor($bodytxtsubticolor[0], $bodytxtsubticolor[1], $bodytxtsubticolor[2]);	// Sous-titre Milestone/Jalon
				$frm				= implode(',', $bodybgsubticolor) == '255, 255, 255' ? '' : 'F';
				$frmstyle			= array('width'=>'0.2', 'dash'=>'0', 'cap'=>'butt', 'color'=>'255, 255, 255');
				$pdf->RoundedRect($object->marge_gauche, $posy, $object->larg_util_cadre, $h--, 0.001, '1111', $frm, $frmstyle, $bodybgsubticolor);
			}	// if ($conf->global->MAIN_MODULE_MILESTONE)
		}	// if ($object->lines[$i]->product_type == 9)
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('pdf'=>$pdf, 'i'=>$i, 'outputlangs'=>$outputlangs, 'w'=>$w, 'h'=>$h, 'posx'=>$posx, 'posy'=>$posy, 'hideref'=>$hideref, 'hidedesc'=>$hidedesc, 'issupplierline'=>$issupplierline, 'special_code'=>$special_code);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_writelinedesc', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if (!empty($hookmanager->resPrint))					$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			$labelproductservice	= pdf_InfraSPlus_getlinedesc($object, $i, $outputlangs, $hideref, $hidedesc, $issupplierline, $extraDet, $prodfichinter);
			// Fix bug of some HTML editors that replace links <img src="http://localhostgit/viewimage.php?modulepart=medias&file=image/efd.png" into <img src="http://localhostgit/viewimage.php?modulepart=medias&amp;file=image/efd.png"
			// We make the reverse, so PDF generation has the real URL.
			$labelproductservice	= preg_replace('/(<img[^>]*src=")([^"]*)(&amp;)([^"]*")/', '\1\2&\4', $labelproductservice, -1, $nbrep);
			// Description
			$pdf->writeHTMLCell($w, $h, $posx, $posy, $outputlangs->convToOutputCharset($labelproductservice), 0, 1, false, true, 'J', true);
			$result					.= $labelproductservice;
		}	// if (empty($reshook))
		if ($object->lines[$i]->product_type == 9)	$pdf->SetTextColor($bodytxtcolor[0], $bodytxtcolor[1], $bodytxtcolor[2]); // retour à la normal
		return $result;
	}	// function pdf_InfraSPlus_writelinedesc(&$pdf, $object, $i, $outputlangs, $w, $h, $posx, $posy, $hideref = 0, $hidedesc = 0, $issupplierline = 0, $extraDet = '', $prodfichinter = null)

	/********************************************
	*  Return line description translated in outputlangs and encoded into htmlentities and with <br>
	*
	*  @param  Object		$object			Object shown in PDF
	*  @param  int			$i				Current line number (0 = first line, 1 = second line, ...)
	*  @param  Translate	$outputlangs	Object langs for output
	*  @param  int			$hideref		Hide reference
	*  @param  int			$hidedesc		Hide description
	*  @param  int			$issupplierline	Is it a line for a supplier object ?
	*	@param	int			$extraDet		Module equipement compatibility
	*  @return string						String with line
	 ********************************************/
	function pdf_InfraSPlus_getlinedesc(&$object, $i, $outputlangs, $hideref = 0, $hidedesc = 0, $issupplierline = 0, $extraDet = '', $prodfichinter = null)
	{
		global $db, $conf, $langs;

		$idprod			= $prodfichinter ? $prodfichinter['fk_product'] : (! empty($object->lines[$i]->fk_product) ? $object->lines[$i]->fk_product : false);
		$label			= $prodfichinter ? $prodfichinter['label'] : (! empty($object->lines[$i]->label) ? $object->lines[$i]->label : (! empty($object->lines[$i]->product_label) ? $object->lines[$i]->product_label : ''));
		$desc			= (! empty($object->lines[$i]->desc) ? $object->lines[$i]->desc : (! empty($object->lines[$i]->description) ? $object->lines[$i]->description : ''));
		$note			= (! empty($object->lines[$i]->note) ? $object->lines[$i]->note : '');
		$dbatch			= (! empty($object->lines[$i]->detail_batch) ? $object->lines[$i]->detail_batch : false);
		$isMultilangs	= isset($conf->global->MAIN_MULTILANGS)								? $conf->global->MAIN_MULTILANGS							: 0;
		$forceTranslate	= isset($conf->global->MAIN_MULTILANG_TRANSLATE_EVEN_IF_MODIFIED)	? $conf->global->MAIN_MULTILANG_TRANSLATE_EVEN_IF_MODIFIED	: 0;
		$hidelabel		= isset($conf->global->INFRASPLUS_PDF_HIDE_LABEL)					? $conf->global->INFRASPLUS_PDF_HIDE_LABEL					: 0;
		$labelbold		= isset($conf->global->INFRASPLUS_PDF_LABEL_BOLD)					? $conf->global->INFRASPLUS_PDF_LABEL_BOLD					: 0;
		$extraDetPos2	= isset($conf->global->INFRASPLUS_PDF_EXTRADET_SECOND)				? $conf->global->INFRASPLUS_PDF_EXTRADET_SECOND				: 0;
		$depositDate	= isset($conf->global->INVOICE_ADD_DEPOSIT_DATE)					? $conf->global->INVOICE_ADD_DEPOSIT_DATE					: 0;
		$descFirst		= isset($conf->global->MAIN_DOCUMENTS_DESCRIPTION_FIRST)			? $conf->global->MAIN_DOCUMENTS_DESCRIPTION_FIRST			: 0;
		$prodAddType	= isset($conf->global->PRODUCT_ADD_TYPE_IN_DOCUMENTS)				? $conf->global->PRODUCT_ADD_TYPE_IN_DOCUMENTS				: 0;
		$prodRefSupp	= isset($conf->global->PDF_HIDE_PRODUCT_REF_IN_SUPPLIER_LINES)		? $conf->global->PDF_HIDE_PRODUCT_REF_IN_SUPPLIER_LINES		: 1;
		$HTMLinDesc		= isset($conf->global->ADD_HTML_FORMATING_INTO_DESC_DOC)			? $conf->global->ADD_HTML_FORMATING_INTO_DESC_DOC			: 0;
		$CatInDesc		= isset($conf->global->CATEGORY_ADD_DESC_INTO_DOC)					? $conf->global->CATEGORY_ADD_DESC_INTO_DOC					: 0;
		$hideServDate	= isset($conf->global->INFRASPLUS_PDF_HIDE_SERVICE_DATES)			? $conf->global->INFRASPLUS_PDF_HIDE_SERVICE_DATES			: 0;
		if ($issupplierline)
		{
			$ref_supplier		= (! empty($object->lines[$i]->ref_supplier) ? $object->lines[$i]->ref_supplier : (! empty($object->lines[$i]->ref_fourn) ? $object->lines[$i]->ref_fourn : ''));
			if (empty($ref_supplier))
			{
				$ref			= $object->lines[$i]->ref;
				$sql			= 'SELECT pfp.ref_fourn ';
				$sql			.= 'FROM '.MAIN_DB_PREFIX.'product AS p ';
				$sql			.= 'LEFT JOIN '.MAIN_DB_PREFIX.'product_fournisseur_price AS pfp ON p.rowid = pfp.fk_product ';
				$sql			.= 'WHERE p.ref = "'.$ref.'" AND pfp.fk_soc = "'.$object->thirdparty->id.'"';
				$resql			= $object->db->query($sql);
				if ($resql)
				{
					$obj		= $db->fetch_object($resql);
					if ($obj)	$ref_supplier	= $obj->ref_fourn;
				}	// if ($resql)
				$db->free($resql);
			}	// if (empty($ref_supplier))
			$prodser			= new ProductFournisseur($db);
		}	// if ($issupplierline)
		else					$prodser	= new Product($db);
		if ($idprod)
		{
			$prodser->fetch($idprod);
			// If a predefined product and multilang and on other lang, we renamed label with label translated
			if ($isMultilangs && ($outputlangs->defaultlang != $langs->defaultlang))
			{
				$translatealsoifmodified	= (! empty($forceTranslate));	// By default if value was modified manually, we keep it (no translation because we don't have it)

				// TODO Instead of making a compare to see if param was modified, check that content contains reference translation. If yes, add the added part to the new translation
				// ($textwasmodified is replaced with $textwasmodifiedorcompleted and we add completion).

				// Set label
				// If we want another language, and if label is same than default language (we did force it to a specific value), we can use translation.
				//var_dump($outputlangs->defaultlang.' - '.$langs->defaultlang.' - '.$label.' - '.$prodser->label);exit;
				$textwasmodified			= ($label == $prodser->label);
				if (! empty($prodser->multilangs[$outputlangs->defaultlang]["label"]) && ($textwasmodified || $translatealsoifmodified))	$label=$prodser->multilangs[$outputlangs->defaultlang]["label"];

				// Set desc
				// Manage HTML entities description test because $prodser->description is store with htmlentities but $desc no
				$textwasmodified																												= false;
				if (!empty($desc) && dol_textishtml($desc) && !empty($prodser->description) && dol_textishtml($prodser->description))			$textwasmodified	= (strpos(dol_html_entity_decode($desc,ENT_QUOTES | ENT_HTML401), dol_html_entity_decode($prodser->description,ENT_QUOTES | ENT_HTML401)) !== false);
				else																															$textwasmodified	= ($desc == $prodser->description);
				if (! empty($prodser->multilangs[$outputlangs->defaultlang]["description"]) && ($textwasmodified || $translatealsoifmodified))	$desc				= $prodser->multilangs[$outputlangs->defaultlang]["description"];
				// Set note
				$textwasmodified																												= ($note == $prodser->note);
				if (! empty($prodser->multilangs[$outputlangs->defaultlang]["note"]) && ($textwasmodified || $translatealsoifmodified))			$note				= $prodser->multilangs[$outputlangs->defaultlang]["note"];
			}	// if ($isMultilangs && ($outputlangs->defaultlang != $langs->defaultlang))
		}	// if ($idprod)
		$libelleproduitservice	= '';
		// Description short of product line
		if (empty($hidelabel))
		{
			if ($labelbold && $label)	$libelleproduitservice	.= '<b>'.$label.'</b>';
			else						$libelleproduitservice	.= $label;
		}	// if (empty($hidelabel))
		// Extra-details of product line
		if ($extraDet)	$libelleproduitservice	.= empty($extraDetPos2) ? $extraDet : '';
		// Description long of product line
		if (! empty($desc) && ($desc != $label))
		{
			if ($libelleproduitservice && empty($hidedesc))	$libelleproduitservice	.= '__N__';
			if ($desc == '(CREDIT_NOTE)' && $object->lines[$i]->fk_remise_except)
			{
				$discount				= new DiscountAbsolute($db);
				$discount->fetch($object->lines[$i]->fk_remise_except);
				$sourceref				= !empty($discount->discount_type) ? $discount->ref_invoive_supplier_source : $discount->ref_facture_source;
				$libelleproduitservice	= $outputlangs->transnoentitiesnoconv("DiscountFromCreditNote", $sourceref);
			}	// if ($desc == '(CREDIT_NOTE)' && $object->lines[$i]->fk_remise_except)
			elseif ($desc == '(DEPOSIT)' && $object->lines[$i]->fk_remise_except)
			{
				$discount					= new DiscountAbsolute($db);
				$discount->fetch($object->lines[$i]->fk_remise_except);
				$sourceref					= !empty($discount->discount_type) ? $discount->ref_invoive_supplier_source : $discount->ref_facture_source;
				$libelleproduitservice		= $outputlangs->transnoentitiesnoconv("DiscountFromDeposit", $sourceref);
				// Add date of deposit
				if (! empty($depositDate))	echo ' ('.dol_print_date($discount->datec, 'day', '', $outputlangs).')';
			}	// elseif ($desc == '(DEPOSIT)' && $object->lines[$i]->fk_remise_except)
			if ($desc == '(EXCESS RECEIVED)' && $object->lines[$i]->fk_remise_except)
			{
				$discount				= new DiscountAbsolute($db);
				$discount->fetch($object->lines[$i]->fk_remise_except);
				$libelleproduitservice	= $outputlangs->transnoentitiesnoconv("DiscountFromExcessReceived", $discount->ref_facture_source);
			}	// if ($desc == '(EXCESS RECEIVED)' && $object->lines[$i]->fk_remise_except)
			elseif ($desc == '(EXCESS PAID)' && $object->lines[$i]->fk_remise_except)
			{
				$discount				= new DiscountAbsolute($db);
				$discount->fetch($object->lines[$i]->fk_remise_except);
				$libelleproduitservice	= $outputlangs->transnoentitiesnoconv("DiscountFromExcessPaid", $discount->ref_invoice_supplier_source);
			}	// elseif ($desc == '(EXCESS PAID)' && $object->lines[$i]->fk_remise_except)
			else
			{
				if ($idprod)
				{
					if (empty($hidedesc))
					{
						if (!empty($descFirst))	$libelleproduitservice	= $desc.'__N__'.$libelleproduitservice;
						else					$libelleproduitservice	.= $desc;
					}	// if (empty($hidedesc))
				}	// if ($idprod)
				else	$libelleproduitservice	.= $desc;
			}	// else	// elseif ($desc == '(...)' && $object->lines[$i]->fk_remise_except)
		}	// if (! empty($desc) && ($desc != $label))
		if (! empty($extraDetPos2))	$libelleproduitservice	.= ! empty($extraDet) ? $extraDet : '';
		// We add ref of product (and supplier ref if defined)
		$prefix_prodserv	= "";
		$ref_prodserv		= "";
		if (! empty($prodAddType))   // In standard mode, we do not show this
		{
			if ($prodser->isService())	$prefix_prodserv	= $outputlangs->transnoentitiesnoconv("Service")." ";
			else						$prefix_prodserv	= $outputlangs->transnoentitiesnoconv("Product")." ";
		}	// if (! empty(prodAddType))
		if (empty($hideref))
		{
			if ($issupplierline)
			{
				if ($prodRefSupp == 1)		$ref_prodserv	= $ref_supplier;
				elseif ($prodRefSupp == 2)	$ref_prodserv	= $ref_supplier.' ('.$outputlangs->transnoentitiesnoconv("ProductRef").' '.$prodser->ref.')';
				else	// Common case
				{
					$ref_prodserv		= $prodser->ref; // Show local ref
					if ($ref_supplier)	$ref_prodserv	.= ($prodser->ref ? ' (' : '').$outputlangs->transnoentitiesnoconv("SupplierRef").' '.$ref_supplier.($prodser->ref ? ')' : '');
				}	// else	// elseif (prodRefSupp == 2)	// if (prodRefSupp == 1)
			}	// if ($issupplierline)
			else															$ref_prodserv	= $prodser->ref; // Show local ref only
			if (! empty($libelleproduitservice) && ! empty($ref_prodserv))	$ref_prodserv	.= " - ";
		}	// if (empty($hideref))
		if (!empty($ref_prodserv) && !empty($HTMLinDesc))	$ref_prodserv	= '<b>'.$ref_prodserv.'</b>';
		$libelleproduitservice								= $prefix_prodserv.$ref_prodserv.$libelleproduitservice;
		// Add an additional description for the category products
		if (! empty($CatInDesc) && $idprod && ! empty($conf->categorie->enabled))
		{
			include_once DOL_DOCUMENT_ROOT.'/categories/class/categorie.class.php';
			$categstatic	= new Categorie($db);
			$tblcateg		= $categstatic->containing($idprod, Categorie::TYPE_PRODUCT);	// recovering the list of all the categories linked to product
			foreach($tblcateg as $cate)
			{
				$desccateg		= $cate->add_description;	// Adding the descriptions if they are filled
				if ($desccateg)	$libelleproduitservice	.= '__N__'.$desccateg;
			}	// foreach($tblcateg as $cate)
		}	// if (! empty($CatInDesc && $idprod && ! empty($conf->categorie->enabled)))
		if (empty($hideServDate))
		{
			$format		= 'day';
			$period		= '';
			$period2	= '';
			// Show duration if exists
			if ($object->element == 'contrat')
			{
				if ($object->lines[$i]->date_start)		$period		.= $outputlangs->transnoentitiesnoconv('DateStartPlanned').' : '.dol_print_date($object->lines[$i]->date_start, $format, false, $outputlangs);
				if ($object->lines[$i]->date_end)		$period		.= ($period ? ' - ' : '').$outputlangs->transnoentitiesnoconv('DateEndPlanned').' : '.dol_print_date($object->lines[$i]->date_end, $format, false, $outputlangs);
				if ($object->lines[$i]->date_ouverture)	$period2	.= $outputlangs->transnoentitiesnoconv('DateStartReal').' : '.dol_print_date($object->lines[$i]->date_ouverture, $format, false, $outputlangs);
				if ($object->lines[$i]->date_cloture)	$period2	.= ($period ? ' - ' : '').$outputlangs->transnoentitiesnoconv('DateEndReal').' : '.dol_print_date($object->lines[$i]->date_cloture, $format, false, $outputlangs);
				$period	.= ($period && $period2 ? '__N__'.$period2 : $period2);
			}	// if ($object->element == 'contrat')
			else
			{
				if ($object->lines[$i]->date_start && $object->lines[$i]->date_end)		$period	= '('.$outputlangs->transnoentitiesnoconv('DateFromTo', dol_print_date($object->lines[$i]->date_start, $format, false, $outputlangs), dol_print_date($object->lines[$i]->date_end, $format, false, $outputlangs)).')';
				if ($object->lines[$i]->date_start && ! $object->lines[$i]->date_end)	$period	= '('.$outputlangs->transnoentitiesnoconv('DateFrom', dol_print_date($object->lines[$i]->date_start, $format, false, $outputlangs)).')';
				if (! $object->lines[$i]->date_start && $object->lines[$i]->date_end)	$period = '('.$outputlangs->transnoentitiesnoconv('DateUntil', dol_print_date($object->lines[$i]->date_end, $format, false, $outputlangs)).')';
			}	// else	// if ($object->element == 'contrat')
			if(!empty($HTMLinDesc))	$libelleproduitservice	.= '<b style = "color: #333666;" ><em>'."__N__</b> ".$period.'</em>';
			else					$libelleproduitservice	.= "__N__".$period;
		}	// if (empty($hideServDate))
		if ($dbatch)
		{
			$format='day';
			foreach($dbatch as $detail)
			{
				$dte=array();
				if ($detail->eatby)		$dte[]	= $outputlangs->transnoentitiesnoconv('printEatby', dol_print_date($detail->eatby, $format, false, $outputlangs));
				if ($detail->sellby)	$dte[]	= $outputlangs->transnoentitiesnoconv('printSellby', dol_print_date($detail->sellby, $format, false, $outputlangs));
				if ($detail->batch)		$dte[]	= $outputlangs->transnoentitiesnoconv('printBatch', $detail->batch);
				$dte[]					= $outputlangs->transnoentitiesnoconv('printQty', $detail->qty);
				$libelleproduitservice	.= "__N__  ".implode(" - ", $dte);
			}	// foreach($dbatch as $detail)
		}	// if ($dbatch)
		// Now we convert \n into br
		if (dol_textishtml($libelleproduitservice))	$libelleproduitservice	= preg_replace('/__N__/', '<br>', $libelleproduitservice);
		else										$libelleproduitservice	= preg_replace('/__N__/', "\n", $libelleproduitservice);
		$libelleproduitservice						= dol_htmlentitiesbr($libelleproduitservice, 1);
		return $libelleproduitservice;
	}	// function pdf_InfraSPlus_getlinedesc($object, $i, $outputlangs, $hideref = 0, $hidedesc = 0, $issupplierline = 0, $extraDet = '', $prodfichinter = null)

	/********************************************
	*	Return line quantity
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	*	@return	string
	 ********************************************/
	function pdf_InfraSPlus_getlineqty($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $hookmanager;

		$result		= '';
		$reshook	= 0;
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i,'outputlangs'=>$outputlangs,'hidedetails'=>$hidedetails,'special_code'=>$special_code);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlineqty', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint))					$result			= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
		   if ($object->lines[$i]->special_code == 3)	return '';
		   if (empty($hidedetails) || $hidedetails > 1)	$result	.= $prodfichinter ? $prodfichinter['qty'] : $object->lines[$i]->qty;
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlineqty($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return line vat rate
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string
	********************************************/
	function pdf_InfraSPlus_getlinevatrate($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $conf, $hookmanager, $mysoc;

		$result		= '';
		$reshook	= 0;
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i,'outputlangs'=>$outputlangs,'hidedetails'=>$hidedetails,'special_code'=>$special_code);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlinevatrate',$parameters,$object,$action);    // Note that $action and $object may have been modified by some hooks
			if (!empty($hookmanager->resPrint))					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if (empty($hidedetails) || $hidedetails > 1)
			{
				$tva_tx		= $prodfichinter ? $prodfichinter['tva_tx'] : $object->lines[$i]->tva_tx;
				$info_bits	= $prodfichinter ? $prodfichinter['info_bits'] : $object->lines[$i]->info_bits;
				$tmpresult	= '';
				$tmpresult	.= vatrate($tva_tx, 0, $info_bits, -1);
				if (empty($conf->global->MAIN_PDF_MAIN_HIDE_SECOND_TAX))
				{
					$total_localtax1	= $prodfichinter ? $prodfichinter['total_localtax1'] : $object->lines[$i]->total_localtax1;
					if ($total_localtax1 != 0)
					{
						$localtax1_tx								= $prodfichinter ? $prodfichinter['localtax1_tx'] : $object->lines[$i]->localtax1_tx;
						if (preg_replace('/[\s0%]/','',$tmpresult))	$tmpresult	.= '/';
						else										$tmpresult	= '';
						$tmpresult									.= vatrate(abs($localtax1_tx), 0);
					}	// if ($total_localtax1 != 0)
				}	// if (empty($conf->global->MAIN_PDF_MAIN_HIDE_SECOND_TAX))
				if (empty($conf->global->MAIN_PDF_MAIN_HIDE_THIRD_TAX))
				{
					$total_localtax2	= $prodfichinter ? $prodfichinter['total_localtax2'] : $object->lines[$i]->total_localtax2;
					if ($total_localtax2 != 0)
					{
						$localtax2_tx								= $prodfichinter ? $prodfichinter['localtax2_tx'] : $object->lines[$i]->localtax2_tx;
						if (preg_replace('/[\s0%]/','',$tmpresult))	$tmpresult	.= '/';
						else										$tmpresult	= '';
						$tmpresult									.= vatrate(abs($localtax2_tx), 0);
					}	// if ($total_localtax2 != 0)
				}	// if (empty($conf->global->MAIN_PDF_MAIN_HIDE_THIRD_TAX))
				$tmpresult	.= '%';
				$result		.= $tmpresult;
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// f (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlinevatrate($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return line remise percent
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string
	********************************************/
	function pdf_InfraSPlus_getlineremisepercent($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $hookmanager;

		include_once DOL_DOCUMENT_ROOT.'/core/lib/functions2.lib.php';

		$reshook	= 0;
		$result		= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlineremisepercent', $parameters,$object,$action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint))					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)		return '';
			$remise_percent									= $prodfichinter ? $prodfichinter['remise_percent'] : $object->lines[$i]->remise_percent;
			if (empty($hidedetails) || $hidedetails > 1)	$result	.= dol_print_reduction($remise_percent, $outputlangs);
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlineremisepercent($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return formated price
	*
	*	@param	Object		$object				Object
	*	@param	float		$price				price to format
	*	@param  Translate	$outputlangs		Object langs for output
	* 	@return	price
	 ********************************************/
	function pdf_InfraSPlus_price($object, $price, $outputlangs, $forceSymb = 0)
	{
		global $conf;

		$rounding		= min($conf->global->MAIN_MAX_DECIMALS_UNIT, $conf->global->MAIN_MAX_DECIMALS_TOT);
		$currency		= !empty($object->multicurrency_code) ? $object->multicurrency_code : $conf->currency;
		$showCurSymb	= $forceSymb ? 1 : (isset($conf->global->INFRASPLUS_PDF_SHOW_CUR_SYMB) ? $conf->global->INFRASPLUS_PDF_SHOW_CUR_SYMB : 0);
		return price($price, 0, $outputlangs, 1, -1, $rounding, ($showCurSymb ? $currency : ''));
	}	// function pdf_InfraSPlus_price($object, $price, $outputlangs, $forceSymb = 0)
	/********************************************
	*	Return line unit price excluding tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string							Line unit price excluding tax
	 ********************************************/
	function pdf_InfraSPlus_getlineupexcltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $conf, $hookmanager;

		$sign																									= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))	$sign	= -1;
		$reshook																								= 0;
		$result																									= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i,'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlineupexcltax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if (empty($hidedetails) || $hidedetails > 1)
			{
				switch ($object->element)
				{
					case 'contrat':
						$subprice	= $conf->multicurrency->enabled && $object->lines[$i]->multicurrency_subprice != 0 ? $object->lines[$i]->multicurrency_subprice : $object->lines[$i]->subprice;
					break;
					case 'fichinter':
						$subprice	= $prodfichinter ? $prodfichinter['subprice'] : 0;
					break;
					default:
						$subprice	= $conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_subprice : $object->lines[$i]->subprice;
					break;
				}	// switch ($object->element)
				$result		.= pdf_InfraSPlus_price($object, $sign * $subprice, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlineupexcltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return line unit price including tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string							Line unit price including tax
	 ********************************************/
	function pdf_InfraSPlus_getlineupincltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $conf, $hookmanager;

		$sign																									= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))	$sign	= -1;
		$reshook																								= 0;
		$result																									= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i,'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlineupwithtax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if (empty($hidedetails) || $hidedetails > 1)
			{
				switch ($object->element)
				{
					case 'contrat':
						$subprice	= $conf->multicurrency->enabled && $object->lines[$i]->multicurrency_subprice != 0 ? $object->lines[$i]->multicurrency_subprice : $object->lines[$i]->subprice;
					break;
					case 'fichinter':
						$subprice	= $prodfichinter ? $prodfichinter['subprice'] : 0;
					break;
					default:
						$subprice	= $conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_subprice : $object->lines[$i]->subprice;
					break;
				}	// switch ($object->element)
				$tva_tx		= $prodfichinter ? $prodfichinter['tva_tx'] : $object->lines[$i]->tva_tx;
				$result		.= pdf_InfraSPlus_price($object, $sign * ($subprice + ($subprice * $tva_tx / 100)), $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlineupincltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return line unit price with discount and excluding tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string							Line unit price with discount and excluding tax
	 ********************************************/
	function pdf_InfraSPlus_getlineincldiscountexcltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $conf, $hookmanager;

		$sign																											= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))	$sign	= -1;
		$reshook																										= 0;
		$result																											= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i,'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlineupexcltax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				switch ($object->element)
				{
					case 'contrat':
						$total_ht	= $conf->multicurrency->enabled && $object->lines[$i]->multicurrency_total_ht != 0 ? $object->lines[$i]->multicurrency_total_ht : $object->lines[$i]->total_ht;
					break;
					case 'fichinter':
						$total_ht	= $prodfichinter ? $prodfichinter['total_ht'] : 0;
					break;
					default:
						$total_ht	= $conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ht : $object->lines[$i]->total_ht;
					break;
				}	// switch ($object->element)
				$qty		= $prodfichinter ? $prodfichinter['qty'] : $object->lines[$i]->qty;
				$qty		= $qty == 0 ? 1 : $qty;
				$result		.= pdf_InfraSPlus_price($object, $sign * ($total_ht / $qty), $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlineincldiscountexcltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return line unit price with discount and including tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param 	Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide value (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string							Line unit price with discount and including tax
	 ********************************************/
	function pdf_InfraSPlus_getlineincldiscountincltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $conf, $hookmanager;

		$sign																									= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))	$sign	= -1;
		$reshook																								= 0;
		$result																									= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlineupwithtax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				switch ($object->element)
				{
					case 'contrat':
						$total_ttc	= $conf->multicurrency->enabled && $object->lines[$i]->multicurrency_total_ttc != 0 ? $object->lines[$i]->multicurrency_total_ttc : $object->lines[$i]->total_ttc;
					break;
					case 'fichinter':
						$total_ttc	= $prodfichinter ? $prodfichinter['total_ttc'] : 0;
					break;
					default:
						$total_ttc	= $conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ttc : $object->lines[$i]->total_ttc;
					break;
				}	// switch ($object->element)
				$qty		= $prodfichinter ? $prodfichinter['qty'] : $object->lines[$i]->qty;
				$qty		= $qty == 0 ? 1 : $qty;
				$result		.= pdf_InfraSPlus_price($object, $sign * ($total_ttc / $qty), $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlineincldiscountincltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return total of line excluding tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param 	Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide value (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string							Total of line excluding tax
	 ********************************************/
	function pdf_InfraSPlus_getlinetotalexcltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $conf, $hookmanager;

		$sign																									= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))	$sign	= -1;
		$reshook																								= 0;
		$result																									= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlinetotalexcltax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				switch ($object->element)
				{
					case 'contrat':
						$total_ht	= $conf->multicurrency->enabled && $object->lines[$i]->multicurrency_total_ht != 0 ? $object->lines[$i]->multicurrency_total_ht : $object->lines[$i]->total_ht;
					break;
					case 'fichinter':
						$total_ht	= $prodfichinter ? $prodfichinter['total_ht'] : 0;
					break;
					default:
						$total_ht	= $conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ht : $object->lines[$i]->total_ht;
					break;
				}	// switch ($object->element)
				$result		.= pdf_InfraSPlus_price($object, $sign * $total_ht, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlinetotalexcltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return total of line including tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param 	Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide value (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string							Total of line including tax
	 ********************************************/
	function pdf_InfraSPlus_getlinetotalincltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)
	{
		global $conf, $hookmanager;

		$sign																									= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))	$sign	= -1;
		$reshook																								= 0;
		$result																									= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_getlinetotalwithtax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint))					$result			.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				switch ($object->element)
				{
					case 'contrat':
						$total_ttc	= $conf->multicurrency->enabled && $object->lines[$i]->multicurrency_total_ttc != 0 ? $object->lines[$i]->multicurrency_total_ttc : $object->lines[$i]->total_ttc;
					break;
					case 'fichinter':
						$total_ttc	= $prodfichinter ? $prodfichinter['total_ttc'] : 0;
					break;
					default:
						$total_ttc	= $conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ttc : $object->lines[$i]->total_ttc;
					break;
				}	// switch ($object->element)
				$result		.= pdf_InfraSPlus_price($object, $sign * $total_ttc, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraSPlus_getlinetotalincltax($object, $i, $outputlangs, $hidedetails = 0, $prodfichinter = null)

	/********************************************
	*	Return line product ref for Intervention card
	*
	*	@param	Object	$object		Object
	* 	@return	array
	 ********************************************/
	function pdf_infrasplus_getpricefichinter($object)
	{
		global $db;

		$pricefichinter	= array();
		$sql	= 'SELECT fi.total_ht, fi.total_ttc, fi.total_tva, fi.total_localtax1, fi.total_localtax2';
		$sql	.= ' FROM '.MAIN_DB_PREFIX.'fichinter AS fi';
		$sql	.= ' WHERE fi.rowid = '.$object->id;
		$resql	= $db->query($sql);
		if ($resql)
		{
			$num	= $db->num_rows($resql);
			$j		= 0;
			while ($j < $num)
			{
				$objp								= $db->fetch_object($resql);
				$pricefichinter['total_ht']			= $objp->total_ht;
				$pricefichinter['total_ttc']		= $objp->total_ttc;
				$prodfichinter['total_tva']			= $objp->total_tva;
				$prodfichinter['total_localtax1']	= $objp->total_localtax1;
				$prodfichinter['total_localtax2']	= $objp->total_localtax2;
				$j++;
			}	// while ($j < $num)
			$db->free($resql);
		}	// if ($resql)
		return $pricefichinter;
	}	// function pdf_infrasplus_getpricefichinter($object)

	/************************************************
	*	Show CGV for PDF generation
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  string		$cgv			PDF file name
	*	@param	int			$hidepagenum	Hide page num (x/y)
	*	@param  Object		$object     	Object shown in PDF
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	************************************************/
	function pdf_InfraSPlus_CGV(&$pdf, $cgv, $hidepagenum = 0, $object, $outputlangs, $formatpage)
	{
		global $conf;

		$path			= ($conf->entity > 1 ? "/".$conf->entity : '');
		$cgv_pdf		= DOL_DATA_ROOT.$path.'/mycompany/'.$cgv;
		pdf_InfraSPlus_Merge($pdf, $cgv_pdf, $hidepagenum, $object, $outputlangs, $formatpage);
	}	// function pdf_InfraSPlus_CGV($pdf, $cgv_pdf, $hidepagenum = 0, $object, $outputlangs, $formatpage)

	/************************************************
	*	Show files for PDF generation
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  string		$files			list of rowid for PDF file from llx_ecm_files
	*	@param	int			$hidepagenum	Hide page num (x/y)
	*	@param  Object		$object     	Object shown in PDF
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	************************************************/
	function pdf_InfraSPlus_files(&$pdf, $files, $hidepagenum = 0, $object, $outputlangs, $formatpage)
	{
		global $conf, $db;

		if ($files && $files != 'None')
		{
			foreach ($files as $fileID)
			{
				$sql	= ' SELECT filename, filepath';
				$sql	.= ' FROM '.MAIN_DB_PREFIX.'ecm_files';
				$sql	.= ' WHERE rowid = '.$fileID;
				$sql	.= ' AND entity = '.$conf->entity;
				$resql	= $db->query($sql);
				if ($resql)
				{
					$objFile	= $db->fetch_object($resql);
					if ($objFile)
					{
						$filename	= $objFile->filename;
						$filepath	= $objFile->filepath;
						$file		= DOL_DATA_ROOT.'/'.$filepath.'/'.$filename;
						pdf_InfraSPlus_Merge($pdf, $file, $hidepagenum, $object, $outputlangs, $formatpage);
					}	// if ($objFile)
				}	// if ($resql)
			}	// foreach ($files as $fileID)
		}	// if ($files && $files != 'None')
	}	// function pdf_InfraSPlus_files($pdf, $files, $hidepagenum = 0, $object, $outputlangs, $formatpage)

	/************************************************
	*	Show CGV for PDF generation
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  string		$infile			PDF file full name (with path) to merge
	*	@param	int			$hidepagenum	Hide page num (x/y)
	*	@param  Object		$object     	Object shown in PDF
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	************************************************/
	function pdf_InfraSPlus_Merge(&$pdf, $infile, $hidepagenum = 0, $object, $outputlangs, $formatpage)
	{
		global $conf;

		if (file_exists($infile) && is_readable($infile))
		{
			$finfo	= finfo_open(FILEINFO_MIME_TYPE);
			if (finfo_file($finfo, $infile) == 'application/pdf')
			{
				try
				{
					$pagecount	= $pdf->setSourceFile($infile);
					for ($i = 1; $i <= $pagecount; $i ++)
					{
						$tplIdx	= $pdf->importPage($i);
						if ($tplIdx !== false)
						{
							$s	= $pdf->getTemplatesize($tplIdx);
							$pdf->AddPage($s['h'] > $s['w'] ? 'P' : 'L', array($s['w'], $s['h']));
							$pdf->useTemplate($tplIdx);
							if (! $hidepagenum)
							{
								$prevFont									= $pdf->getFontFamily();
								$pdf->SetFont('Helvetica');
								if (empty($conf->global->MAIN_USE_FPDF))	$pdf->MultiCell(20, 2, $pdf->PageNo().' / '.$pdf->getAliasNbPages(), 0, 'R', 0, 1, $conf->global->INFRASPLUS_PDF_X_PAGE_NUM, $conf->global->INFRASPLUS_PDF_Y_PAGE_NUM, true, 0, 0, false, 0, 'M', false);
								else										$pdf->MultiCell(20, 2, $pdf->PageNo().' / {nb}', 0, 'R', 0, 1, $conf->global->INFRASPLUS_PDF_X_PAGE_NUM, $conf->global->INFRASPLUS_PDF_Y_PAGE_NUM, true, 0, 0, false, 0, 'M', false);
								$pdf->SetFont($prevFont);
							}	// if (! $hidepagenum)
							if (! empty($conf->global->INFRASPLUS_PDF_REFDATE_MERGE))	pdf_InfraSPlus_pagesrefdate($pdf, $object, $outputlangs, '', $formatpage['mhaute'], $formatpage['largeur'] - $formatpage['mdroite'] - 100);
						}	// if ($tplIdx!==false)
						else	setEventMessages(null, array($outputlangs->trans("PDFInfraSPlusPdfFileError1", $infile)), 'warnings');
					}	// for ($i = 1; $i <= $pagecount; $i ++)
				}	// try
				catch (exception $e)
				{
					setEventMessages(null, array($outputlangs->trans("PDFInfraSPlusPdfFileError1", $infile).$outputlangs->trans("PDFInfraSPlusPdfFileError2", $e->getMessage())), 'warnings');
				}	// try	// catch (exception $e)
			}	// if (finfo_file($finfo, $infile) == 'application/pdf')
		}	// if (file_exists($infile) && is_readable($infile))
	}	// function pdf_InfraSPlus_Merge($pdf, $cgv_pdf, $hidepagenum = 0, $object, $outputlangs, $formatpage)

	/************************************************
	*	Show reference and date of document.
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Object		$object     	Object shown in PDF
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	string		$title			string title in connection with objet type
	*	@param	int			$posx			Position depart (largeur)
	*	@param	int			$posy			Position depart (hauteur)
	************************************************/
	function pdf_InfraSPlus_pagesrefdate(&$pdf, $object, $outputlangs, $title, $posy, $posx, $ticket = 0)
	{
		global $conf;

		$ref				= $outputlangs->transnoentities("Ref");
		$reference			= $outputlangs->convToOutputCharset($object->ref);
		$date				= dol_print_date($object->date, "day", false, $outputlangs, true);
		if (empty($date))	$date	= dol_print_date($object->date_commande, "day", false, $outputlangs, true);
		if (empty($date))	$date	= dol_print_date($object->date_contrat, "day", false, $outputlangs, true);
		if (empty($date))	$date	= dol_print_date($object->date_delivery, "day", false, $outputlangs, true);
		if (empty($date))	$date	= dol_print_date($object->datec, "day", false, $outputlangs, true);
		if (! empty($date))	$pdf->MultiCell(100, 4, ($title ? $title." " : "").$ref." ".$reference." ".$outputlangs->transnoentities("Of")." ".$date, '', $ticket ? 'L' : 'R', 0, 1, $posx, $posy, true, 0, 0, false, 0, 'M', false);
		else				$pdf->MultiCell(100, 4, ($title ? $title." " : "").$ref." ".$reference, '', $ticket ? 'L' : 'R', 0, 1, $posx, $posy, true, 0, 0, false, 0, 'M', false);
	}	// function pdf_InfraSPlus_pagesrefdate(&$pdf, $object, $outputlangs, $title, $posy, $posx, $ticket = 0)

	/********************************************
	*	Add a draft watermark on PDF files
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Translate	$outputlangs	Object lang
	*	@param  string		$text           Text to show
	*	@param  int		    $center_y       Y center of rotation
	*	@param  int		    $w		        Width of table
	*	@param  int		    $hp		        Height of page
	*	@param  string	    $unit           Unit of height (mm, pt, ...)
	*	@return	void
	 ********************************************/
	function pdf_InfraSPlus_watermark(&$pdf, $outputlangs, $text, $center_y, $w, $hp, $unit)
	{
		global $conf;

		$watermark_t_opacity	= isset($conf->global->INFRASPLUS_PDF_T_WATERMARK_OPACITY)		? $conf->global->INFRASPLUS_PDF_T_WATERMARK_OPACITY : 10;
		// Print Draft Watermark
		if ($unit=='pt')		$k = 1;
		elseif ($unit=='mm')	$k = 72/25.4;
		elseif ($unit=='cm')	$k = 72/2.54;
		elseif ($unit=='in')	$k = 72;
		$savx					= $pdf->getX();
		$savy					= $pdf->getY();
		$watermark_angle		= 20 / 180 * pi();	// angle de rotation 20° en radian
		$center_x				= $w / 2;			// x centre
		$pdf->SetFont('', 'B', 40);
		$pdf->SetTextColor(255, 0, 0);
		$pdf->SetAlpha($watermark_t_opacity / 100);
		//rotate
		$pdf->_out(sprintf('q %.5F %.5F %.5F %.5F %.2F %.2F cm 1 0 0 1 %.2F %.2F cm', cos($watermark_angle), sin($watermark_angle),
					-sin($watermark_angle), cos($watermark_angle), $center_x * $k, ($hp - $center_y) * $k, -$center_x * $k, -($hp - $center_y) * $k));
		//print watermark
		$pdf->SetXY(10, $center_y - 10);
		$pdf->Cell($w, 20, $outputlangs->convToOutputCharset($text), "", 2, "C", 0);
		//antirotate
		$pdf->_out('Q');
		$pdf->SetXY($savx, $savy);
		$pdf->SetAlpha(1);
	}	// function pdf_InfraSPlus_watermark(&$pdf, $outputlangs, $text, $center_y, $w, $hp, $unit)

	/************************************************
	*	Show customer signature
	*
	*	@param	TCPDF		$pdf     		The PDF factory
	*	@param	str			$signvalue		base64 string for png image
	*	@param  int		    $larg_signarea	Width of area
	*	@param  int		    $ht_signarea	Height of area
	*	@param  int		    $posxsignarea	X position for the up left corner of area
	*	@param  int		    $posysignarea	Y position for the up left corner of area
	* 	@return	void
	************************************************/
	function pdf_InfraSPlus_Client_Sign($pdf, $signvalue, $larg_signarea, $ht_signarea, $posxsignarea, $posysignarea)
	{
		require_once DOL_DOCUMENT_ROOT.'/core/lib/files.lib.php';

		$imgSign64			= preg_replace('#^data:image/[^;]+;base64,#', '', $signvalue);
		$fileSign			= dol_buildpath('infraspackplus', 0).'/tmp/tmp.png';
		file_put_contents($fileSign, base64_decode($imgSign64));
		if ($fileSign && is_readable($fileSign))
		{
			$imgsize	= array();
			$imgsize	= pdf_InfraSPlus_getSizeForImage($fileSign, $larg_signarea, $ht_signarea);
			if (isset($imgsize['width']) && isset($imgsize['height']))
			{
				$posxSign	= ($larg_signarea - $imgsize['width']) / 2;	// centre l'image dans la zone
				$posySign	= ($ht_signarea - $imgsize['height']) / 2;	// centre l'image dans la zone
				$pdf->Image($fileSign, $posxsignarea + $posxSign, $posysignarea + $posySign, $imgsize['width'], $imgsize['height'], '', '', '', false, 300, '', false, false, 0);	// set sgnature image
			}	// if (isset($imgsize['width']) && isset($imgsize['height']))
		}	// if ($fileSign && is_readable($fileSign))
		dol_delete_file($fileSign);
	}	// function pdf_InfraSPlus_Client_Sign($pdf, $signvalue, $larg_signarea, $ht_signarea, $posxsignarea, $posysignarea)

	/************************************************
	*	Show footer of page for PDF generation
	*
	*	@param	TCPDF		$pdf     		The PDF factory
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	Societe		$fromcompany	Object company
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	* 	@param	int			$showdetails	Show company details into footer. This param seems to not be used by standard version.
											00000 = vide
											1xx0x = address								=> line 1
											1xx1x = address need 2 lines				=> line 1 + 1bis
											2xxxx = contacts (phone, fax, url, mail)	=> line 2
											3xxxx = 1xxx + 2xxx
											x1xxx = manager								=> line 3
											xx1xx = type soc. + capital					=> line 3
											xx2xx = prof. ids							=> line 4
											xx3xx = xx1x + xx2x
											xxxx1 = footer image						=> line 5
	*	@param	int			$hidesupline	Completly hide the line up to footer (for some edition with only table)
	*	@param	int			$calculseul		Arrête la fonction au calcul de hauteur nécessaire
	*	@param	str			$image_foot		File Name of the image to show
	* 	@param	array		$maxsizeimgfoot	Maximum size for image foot => 'largeur', 'hauteur'
	*	@param	int			$hidepagenum	Hide page num (x/y)
	* 	@param	array		$txtcolor		Text color
	*	@param	array		$LineStyle		PDF Line style
	* 	@return	int							Return height of bottom margin including footer text
	************************************************/
	function pdf_InfraSPlus_pagefoot(&$pdf, $outputlangs, $fromcompany, $formatpage, $showdetails, $hidesupline, $calculseul, $image_foot = '', $maxsizeimgfoot, $hidepagenum = 0, $txtcolor = array(0, 0, 0), $LineStyle = null)
	{
		global $conf, $user;

		$pdf->SetTextColor($txtcolor[0], $txtcolor[1], $txtcolor[2]);
		$footer_bold	= isset($conf->global->INFRASPLUS_PDF_FOOTER_BOLD) ? $conf->global->INFRASPLUS_PDF_FOOTER_BOLD : 0;

		// First line of company infos
		$line1	= ""; $line2 = ""; $line3 = ""; $line4 = ""; $line5 = 0;
		if (substr($showdetails, 0, 1) == 1 || substr($showdetails, 0, 1) == 3)
		{
			if ($fromcompany->name)			$line1	.= ($line1 ? " - " : "").$outputlangs->transnoentities("RegisteredOffice")." : ".$fromcompany->name; // Company name
			if ($fromcompany->address)		$line1	.= ($line1 ? " - " : "").str_replace(CHR(13).CHR(10)," - ",$fromcompany->address); // Address
			if ($fromcompany->zip)			$line1	.= ($line1 ? " - " : "").$fromcompany->zip; // Zip code
			if ($fromcompany->town)			$line1	.= ($line1 ? " " : "").$fromcompany->town; // Town
			if ($fromcompany->country_code)	$line1	.= ($line1 ? " - " : "").$outputlangs->transnoentitiesnoconv("Country".$fromcompany->country_code); // Country
		}	// if (substr($showdetails, 0, 1) == 1 || substr($showdetails, 0, 1) == 3)
		if (substr($showdetails, 0, 1) == 2 || substr($showdetails, 0, 1) == 3)
		{
			if ($fromcompany->phone)	$line2	.= ($line2 ? " - " : "").$outputlangs->transnoentities("PhoneShort")." : ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($fromcompany->phone))); // Phone
			if ($fromcompany->fax)		$line2	.= ($line2 ? " - " : "").$outputlangs->transnoentities("Fax")." : ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($fromcompany->fax))); // Fax
			if ($fromcompany->url)		$line2	.= ($line2 ? " - " : "").$fromcompany->url; // URL
			if ($fromcompany->email)	$line2	.= ($line2 ? " - " : "").$fromcompany->email; // Email
		}	// if (substr($showdetails, 0, 1) == 2 || substr($showdetails, 0, 1) == 3)
		if (substr($showdetails, 1, 1) == 1 || ($fromcompany->country_code == 'DE'))
			if ($fromcompany->managers)	$line3 .= ($line3 ? " - " : "").$outputlangs->transnoentities("PDFInfraSPlusManagement")." : ".$fromcompany->managers; // Managers
		if (substr($showdetails, 2, 1) == 1 || substr($showdetails, 2, 1) == 3)
		{
			if ($fromcompany->forme_juridique_code)	$line3 .= ($line3 ? " - " : "").$outputlangs->convToOutputCharset(getFormeJuridiqueLabel($fromcompany->forme_juridique_code)); // Juridical status
			if ($fromcompany->capital) // Capital
			{
				$tmpamounttoshow											= price2num($fromcompany->capital); // This field is a free string
				if (is_numeric($tmpamounttoshow) && $tmpamounttoshow > 0)	$line3	.= ($line3 ? " - " : "").$outputlangs->transnoentities("CapitalOf",price($tmpamounttoshow, 0, $outputlangs, 0, 0, 0, $conf->currency));
				else														$line3	.= ($line3 ? " - " : "").$outputlangs->transnoentities("CapitalOf",$tmpamounttoshow,$outputlangs);
			}	// if ($fromcompany->capital)
		}	// if (substr($showdetails, 2, 1) == 1 || substr($showdetails, 2, 1) == 3)
		if (substr($showdetails, 2, 1) == 2 || substr($showdetails, 2, 1) == 3)
		{
			if ($fromcompany->idprof1 && ($fromcompany->country_code != 'FR' || ! $fromcompany->idprof2)) // Prof Id 1
			{
				$field											= $outputlangs->transcountrynoentities("ProfId1", $fromcompany->country_code);
				if (preg_match('/\((.*)\)/i', $field, $reg))	$field	= $reg[1];
				$line4											.= ($line4 ? " - " : "").$field." : ".$outputlangs->convToOutputCharset($fromcompany->idprof1);
			}	// if ($fromcompany->idprof1 && ($fromcompany->country_code != 'FR' || ! $fromcompany->idprof2))
			if ($fromcompany->idprof2) // Prof Id 2
			{
				$field											= $outputlangs->transcountrynoentities("ProfId2", $fromcompany->country_code);
				if (preg_match('/\((.*)\)/i', $field, $reg))	$field	= $reg[1];
				$line4											.= ($line4 ? " - " : "").$field." : ".$outputlangs->convToOutputCharset($fromcompany->idprof2);
			}	// if ($fromcompany->idprof2)
			if ($fromcompany->idprof3) // Prof Id 3
			{
				$field											= $outputlangs->transcountrynoentities("ProfId3", $fromcompany->country_code);
				if (preg_match('/\((.*)\)/i', $field, $reg))	$field	= $reg[1];
				$line4											.= ($line4 ? " - " : "").$field." : ".$outputlangs->convToOutputCharset($fromcompany->idprof3);
			}	// if ($fromcompany->idprof3)
			if ($fromcompany->idprof4) // Prof Id 4
			{
				$field											= $outputlangs->transcountrynoentities("ProfId4", $fromcompany->country_code);
				if (preg_match('/\((.*)\)/i', $field, $reg))	$field	= $reg[1];
				$line4											.= ($line4 ? " - " : "").$field." : ".$outputlangs->convToOutputCharset($fromcompany->idprof4);
			}	// if ($fromcompany->idprof4)
			if ($fromcompany->idprof5) // Prof Id 5
			{
				$field											= $outputlangs->transcountrynoentities("ProfId5", $fromcompany->country_code);
				if (preg_match('/\((.*)\)/i', $field, $reg))	$field	= $reg[1];
				$line4											.= ($line4 ? " - " : "").$field." : ".$outputlangs->convToOutputCharset($fromcompany->idprof5);
			}	// if ($fromcompany->idprof5)
			if ($fromcompany->idprof6) // Prof Id 6
			{
				$field											= $outputlangs->transcountrynoentities("ProfId6", $fromcompany->country_code);
				if (preg_match('/\((.*)\)/i', $field, $reg))	$field	= $reg[1];
				$line4											.= ($line4 ? " - " : "").$field." : ".$outputlangs->convToOutputCharset($fromcompany->idprof6);
			}	// if ($fromcompany->idprof6)
			if ($fromcompany->tva_intra != '')	$line4	.= ($line4 ? " - " : "").$outputlangs->transnoentities("VATIntraShort")." : ".$outputlangs->convToOutputCharset($fromcompany->tva_intra); // IntraCommunautary VAT
		}	// if (substr($showdetails, 2, 1) == 2 || substr($showdetails, 2, 1) == 3)
		if (substr($showdetails, 4, 1) == 1)
		{
			$logospied	= $conf->mycompany->dir_output.'/logos/'.$image_foot;	// Logos partenaires en ligne 5
			if (is_readable($logospied))
			{
				include_once DOL_DOCUMENT_ROOT.'/core/lib/images.lib.php';
				$imglinesize	= pdf_InfraSPlus_getSizeForImage($logospied, $maxsizeimgfoot['largeur'], $maxsizeimgfoot['hauteur']);
				if ($imglinesize['height'])	$line5	= $imglinesize['height'];
			}	// if (is_readable($logospied))
		}	// if (substr($showdetails, 4, 1) == 1)
		$pdf->SetFont('', $footer_bold ? 'B' : '', 7);
		// The start of the bottom of this page footer is positioned according to # of lines
		$nopage					= $pdf->PageNo();
		$nbpage					= $pdf->getNumPages();
		$marginwithfooter		= (($nopage == $nbpage) && empty($hidesupline) ? 1 : 0) + (! empty($line1) ? 3 : 0) + (! empty($line2) ? 3 : 0) + (! empty($line3) ? 3 : 0) + (! empty($line4) ? 3 : 0) + $line5 + $formatpage['mbasse'];
		if ($calculseul == 1)	return $marginwithfooter;
		$posy					= $formatpage['hauteur'] - $marginwithfooter;
		$pdf->SetY($posy);
		if (($nopage == $nbpage) && empty($hidesupline))
		{
			$pdf->line($formatpage['mgauche'], $posy, $formatpage['largeur']-$formatpage['mdroite'], $posy, $LineStyle);
			$posy++;
		}	// if (($nopage == $nbpage) && empty($hidesupline))
		if (! empty($line1))
		{
			$pdf->SetXY($formatpage['mgauche'], $posy);
			$pdf->MultiCell($formatpage['largeur'] - ($formatpage['mgauche'] + $formatpage['mdroite']), 2, $line1, 0, 'C', 0);
			$posy				+= substr($showdetails, 3, 1) == 1 ? 6 : 3;
		}	// if (! empty($line1))
		if (! empty($line2))
		{
			$pdf->SetXY($formatpage['mgauche'], $posy);
			$pdf->MultiCell($formatpage['largeur'] - ($formatpage['mgauche'] + $formatpage['mdroite']), 2, $line2, 0, 'C', 0);
			$posy				+= 3;
		}	// if (! empty($line2))
		if (! empty($line3))
		{
			$pdf->SetXY($formatpage['mgauche'], $posy);
			$pdf->MultiCell($formatpage['largeur'] - ($formatpage['mgauche'] + $formatpage['mdroite']), 2, $line3, 0, 'C', 0);
			$posy				+= 3;
		}	// if (! empty($line3))
		if (! empty($line4))
		{
			$pdf->SetXY($formatpage['mgauche'], $posy);
			$pdf->MultiCell($formatpage['largeur'] - ($formatpage['mgauche'] + $formatpage['mdroite']), 2, $line4, 0, 'C', 0);
		}	// if (! empty($line4))
		if (is_readable($logospied))
		{
			$posy				+= 3;
			$posxpicture		= $formatpage['mgauche'] + (($formatpage['largeur'] - $formatpage['mgauche'] - $formatpage['mdroite'] - $imglinesize['width']) / 2);	// centre l'image dans la colonne
			$pdf->Image($logospied, $posxpicture, $posy, $imglinesize['width'], $line5);	// width=0 or height=0 (auto)
		}	// if (is_readable($logospied))
		$pdf->SetFont('', '', 7);
		if (! $hidepagenum) // Show page nb only on iso languages (so default Helvetica font)
		{
			$prevFont						= $pdf->getFontFamily();
			$pdf->SetFont('Helvetica');
			$pdf->SetXY($formatpage['largeur'] - ($formatpage['mgauche'] + $formatpage['mdroite'] + 3), $posy);
			if (version_compare(DOL_VERSION, '8.0', '>='))	$pdf->MultiCell(20, 2, $pdf->PageNo().' / '.$pdf->getAliasNbPages(), 0, 'R', 0);
			else
			{
				if (empty($conf->global->MAIN_USE_FPDF))	$pdf->MultiCell(20, 2, $pdf->PageNo().' / '.$pdf->getAliasNbPages(), 0, 'R', 0);
				else										$pdf->MultiCell(20, 2, $pdf->PageNo().' / {nb}', 0, 'R', 0);
			}	// else	// if (version_compare(DOL_VERSION, '8.0', '>='))
			$pdf->SetFont($prevFont);
		}	// if (! $hidepagenum)
		return $marginwithfooter;
	}	// function pdf_InfraSPlus_pagefoot(&$pdf, $outputlangs, $fromcompany, $formatpage, $showdetails, $hidesupline, $calculseul, $image_foot = '', $maxsizeimgfoot, $hidepagenum = 0, $txtcolor = array(0, 0, 0), $LineStyle = null)

	/************************************************
	*	Convert RGBA color to RGB value
	*
	*	@param	string		$bgcolor     		RGB value for background color
	*	@param	string		$color	     		RGB color to convert
	*	@param	float		$alpha	     		alpha value => 0.0 to 1
	*	@return	string							new RGB color
	************************************************/
	function pdf_InfraSPlus_rgba_to_rgb(&$color, $bgcolor = '255, 255, 255', $alpha = 1)
	{
		global $conf;

		$tmpcol				= explode(',', $color);
		$tmpcol[0]			= (! empty($tmpcol[0]) ? $tmpcol[0] : 0);
		$tmpcol[1]			= (! empty($tmpcol[1]) ? $tmpcol[1] : 0);
		$tmpcol[2]			= (! empty($tmpcol[2]) ? $tmpcol[2] : 0);
		$tmpbg				= explode(',', $bgcolor);
		$tmpbg[0]			= (! empty($tmpbg[0]) ? $tmpbg[0] : 0);
		$tmpbg[1]			= (! empty($tmpbg[1]) ? $tmpbg[1] : 0);
		$tmpbg[2]			= (! empty($tmpbg[2]) ? $tmpbg[2] : 0);
		$alpha				= (! empty($alpha) && 0 < $alpha && $alpha < 1 ? $alpha : 1);
		$tmpvalr			= ((1 - $alpha) * $tmpbg[0]) + ($alpha * $tmpcol[0]);
		$tmpvalg			= ((1 - $alpha) * $tmpbg[1]) + ($alpha * $tmpcol[1]);
		$tmpvalb			= ((1 - $alpha) * $tmpbg[2]) + ($alpha * $tmpcol[2]);
		$tmpval				= $tmpvalr.', '.$tmpvalg.', '.$tmpvalb;
		return $tmpval;
	}	// function pdf_InfraSPlus_rgba_to_rgb(&$color, $bgcolor = '255, 255, 255', $alpha = 1)

?>
