<?php
/* Copyright (C) Yassine Belkaid y.belkaid@nextconcept.ma
 *
 */

/**
 * \file    marches/marches.class.php
 * \ingroup marches
 * \brief   This file is marches CRUD class file
 *
 */

require_once DOL_DOCUMENT_ROOT . '/btp/nc_under_modules/marches/db/nxdb.class.php';
require_once DOL_DOCUMENT_ROOT.'/btp/nc_under_modules/marches/executions/class/execution.class.php';
require_once DOL_DOCUMENT_ROOT.'/btp/nc_under_modules/marches/lib/tools.lib.php';

/**
 * Class marches
 *
 * Contract Management class
 */
class Marches extends NXdb
{
	/**
	 * @var string Error code (or message)
	 * @deprecated
	 * @see marches::errors
	 */
	public $error;
	/**
	 * @var string[] Error codes (or messages)
	 */
	public $errors = array();
	/**
	 * @var string Id to identify managed objects
	 */
	public $element = 'marches';
	/**
	 * @var string Name of table without pref_marcheix where object is stored
	 */
	public $table_element = 'marches';

	/**
	 * @var int ID
	 */
	public $id;
	public $ref_marche;
	public $fk_created_by;
	public $fk_appel_offre;
	public $label;
	public $budget;
	public $sig_date;
	public $finished_date;
	public $statut; /* 0 = active, 1 = archived from OA, 3 => archived from Recap */
	public $created_at;
	public $updated_at;

	public $now;
	public $lines = array();
	public $appel_offre = array();
	public $approval = array();
	public $winner = array();
 
	/**
	 * Constructor
	 *
	 * @param DoliDb $db Database handler
	 */
	public function __construct(DoliDBMysqli $db)
	{
		$this->db 		 = $db;
		$this->now 		 = new \DateTime("now");
		$this->now 		 = $this->now->format('Y-m-d H:i:s');
		$this->updated_at= $this->now;
		return 1;
	}

	/**
	 * Create object into database
	 *
	 * @return int   id of last inserted id, otherwise -1 if error arised 
	 */
	public function create()
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		// Clean parameters
		$this->ref_marche 			= $this->ref_marche ? $this->db->escape($this->ref_marche): null;
		$this->fk_created_by 		= $this->fk_created_by ?: null;
		$this->fk_appel_offre 		= $this->fk_appel_offre ?: null;
		$this->created_at 			= $this->now;
		$this->label				= $this->label ? $this->db->escape($this->label) : null;
		$this->budget				= $this->budget ? $this->db->escape($this->budget) : 0;
		$this->sig_date				= $this->sig_date ? $this->db->escape($this->sig_date) : null;
		
		// Insert request
		$sql = 'INSERT INTO ' . MAIN_DB_PREFIX . $this->table_element . ' (ref_marche, fk_created_by, fk_appel_offre, created_at, updated_at, label, budget, sig_date) VALUES (';		
		$sql .= '"'.$this->ref_marche.'", '.$this->fk_created_by.', '.$this->fk_appel_offre.', "'.$this->created_at.'", "'.$this->updated_at .'", "'.$this->label.'", '.$this->budget.', "'.$this->sig_date.'")';

		$this->db->begin();
		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error Marches ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return $this->getLasInsrtedId("");
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param int    $id  Id object
	 * @param string $ref_marche Ref
	 *
	 * @return int <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id, $ref_marche = null, $fk_appel_offre = null)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);
		$extras = 'm.*,a.label as alabel,a.ref as aref,a.fk_third_party,a.subject as asubject,a.start_date as astart_date,a.provisional_amount as aprovisional_amount,a.provisional_date as aprovisional_date,p.amount_ht,c.amount as estimated_amout';
		$fields = (null === $fk_appel_offre ? $extras : '*');

		$sql = 'SELECT '. $fields .' FROM ' . MAIN_DB_PREFIX . $this->table_element.' m';
		
		if (null !== $ref_marche) {
			$sql .= ' INNER JOIN '.  MAIN_DB_PREFIX .'appel_offre a on (m.rowid = a.rowid) WHERE m.ref_marche = ' . '\'' . $ref_marche . '\'';
		} elseif (null !== $fk_appel_offre) {
			$sql .= ' WHERE m.fk_appel_offre = ' . $fk_appel_offre;
		} else {
			$sql .= ' INNER JOIN '.  MAIN_DB_PREFIX .'appel_offre a INNER JOIN '.  MAIN_DB_PREFIX .'m_approval p INNER JOIN '.  MAIN_DB_PREFIX .'m_competitors c WHERE m.rowid = ' . $id .' AND (m.fk_appel_offre = a.rowid AND p.fk_appel_offre = m.fk_appel_offre AND c.fk_appel_offre = m.fk_appel_offre AND c.winner = 1)';
		}

		$resql = $this->db->query($sql);
		if ($resql) {
			$numrows = $this->db->num_rows($resql);

			if ($numrows) {
				$obj 					 = $this->db->fetch_object($resql);
				$this->id 				 = $obj->rowid;
				$this->ref_marche 		 = $obj->ref_marche;
				$this->fk_created_by 	 = $obj->created_by;
				$this->fk_appel_offre 	 = $obj->fk_appel_offre;
				$this->label 			 = $obj->label;
				$this->budget 			 = (float) $obj->budget;
				$this->statut  		 	 = $obj->statut;
				$this->sig_date  		 = $this->db->jdate($obj->sig_date);
				$this->finished_date	 = $this->db->jdate($obj->finished_date);
				$this->updated_at 		 = $this->db->jdate($obj->updated_at);
				$this->created_at 		 = $this->db->jdate($obj->created_at);

				if (null === $fk_appel_offre) {
					$this->appel_offre = array(
						'label' 			 => $obj->alabel,
						'ref' 				 => $obj->aref,
						'tier' 				 => $obj->fk_third_party,
						'subject' 			 => nl2br($obj->asubject),
						'start_date' 		 => $obj->astart_date,
						'provisional_amount' => $obj->aprovisional_amount,
						'provisional_date' 	 => $this->db->jdate($obj->aprovisional_date),
					);
					$amounts 		 = Tools::getTva($obj->amount_ht);
					$estimated_amout = Tools::getTva($obj->estimated_amout);
					$this->approval = array(
						'amount_ht'  => $obj->amount_ht,
						'tva' 		 => $amounts['tva'],
						'amount_ttc' => $amounts['amount_ttc']
					);
					$this->winner = array(
						'amount_ht'  => $obj->estimated_amout,
						'tva' 		 => $estimated_amout['tva'],
						'amount_ttc' => $estimated_amout['amount_ttc']
					);
				}

			}

			$this->db->free($resql);

			if ($numrows) {
				return 1;
			} else {
				return 0;
			}
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param string $sortorder Sort Order
	 * @param string $sortfield Sort field
	 * @param int    $limit     offset limit
	 * @param int    $offset    offset limit
	 * @param array  $filter    filter array
	 * @param string $filtermode filter mode (AND or OR)
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function fetchAll($sortorder = '', $sortfield = '', $limit = 0, $offset = 0, $filter = '', $status = false, $type = '', $id = '')
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT m.rowid as mid, m.ref_marche, m.statut as mstatut, m.fk_created_by as mfk_created_by, m.label as mlabel, m.budget as mbudget, m.sig_date as msig_date, m.updated_at as mupdated_at, m.created_at as mcreated_at, m.fk_appel_offre, a.type as marche_type FROM ' . MAIN_DB_PREFIX . $this->table_element .' as m INNER JOIN '. MAIN_DB_PREFIX.'appel_offre as a';

		if ($status && empty($filter)) {
			$sql .= ' WHERE '. $status;
		} else if (!empty($filter)) {
			$sql .= ' WHERE '. substr($filter, 4);
			$sql .= ($status ? ' AND'.$status: ' AND m.statut = 0');
		} else {
			$sql .= ' WHERE m.statut = 0';
		}

		$sql .= ' AND a.rowid = m.fk_appel_offre';

		if (isset($type) && !empty($type)) {
			$sql .=($type != -1 ? " AND a.type = ". (int)$type : '');
		}

		if (isset($id) && !empty($id))
			$sql .= ' AND m.rowid = '. $id;
		else 
			$sql .= " group by m.fk_appel_offre";

		if (!empty($sortfield)) {
			$sql .= $this->db->order($sortfield, $sortorder);
		}

		$this->lines = array();
		$resql 		 = $this->db->query($sql);

		if ($resql) {
			$num = $this->db->num_rows($resql);

			while ($obj = $this->db->fetch_object($resql)) {
				$line = new stdClass;

				$line->id 			 = $obj->mid;
				$line->ref_marche 	 = $obj->ref_marche;
				$line->statut 		 = $obj->mstatut;
				$line->fk_created_by = $obj->mfk_created_by;
				$line->fk_appel_offre= $obj->fk_appel_offre;
				$line->label 		 = $obj->mlabel;
				$line->marche_type	 = $obj->marche_type;
				$line->budget 		 = (float) $obj->mbudget;
				$line->finished_date = $this->db->jdate($obj->finished_date);
				$line->sig_date  	 = $this->db->jdate($obj->msig_date);
				$line->updated_at 	 = $this->db->jdate($obj->mupdated_at);
				$line->created_at 	 = $this->db->jdate($obj->mcreated_at);
				$this->lines[] 		 = $line;
			}
			$this->db->free($resql);

			return $num;
		} else {
			$this->error 	= 'Error ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	public function countMarches($id)
	{
		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;

		$sql .= ' WHERE fk_appel_offre = ' . $this->getAoId($id);

		$resql = $this->db->query($sql);

		if ($resql) {
			$numrows = $this->db->num_rows($resql);

			if ($numrows) {
				$ids = array();

				while ($obj = $this->db->fetch_object($resql))
					$ids[] = $obj/*array(
						'id' 		=> $obj->rowid,
						'name' 		=> strtolower($obj->ref_marche),
						'sig_date' 	=> dol_print_date($obj->sig_date, 'day'),
						'budget' 	=> number_format($obj->budget, 2)
					)*/;
				
				return $ids;
			}

			$this->db->free($resql);

			return 0;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	public function getAoId($id)
	{
		$sql = 'SELECT fk_appel_offre FROM ' . MAIN_DB_PREFIX . $this->table_element;

		$sql .= ' WHERE rowid = ' . $id;

		$resql = $this->db->query($sql);

		if ($resql) {
			$numrows = $this->db->num_rows($resql);

			if ($numrows) {
				$obj = $this->db->fetch_object($resql);
				
				return $obj->fk_appel_offre;
			}

			$this->db->free($resql);

			return 0;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	public function isMarcheCadre($id)
	{
		$sql = 'SELECT a.type as atype FROM ' . MAIN_DB_PREFIX . $this->table_element .' as m INNER JOIN '. MAIN_DB_PREFIX .'appel_offre as a ON (a.rowid = m.fk_appel_offre) WHERE m.rowid = '. $id .' AND a.type = 2';

		$resql = $this->db->query($sql);

		if ($resql) {
			$numrows = $this->db->num_rows($resql);

			if ($numrows) {
				$obj = $this->db->fetch_object($resql);
				
				return true;
			}

			$this->db->free($resql);

			return false;
		} else {
			return false;
		}
	}

	/**
	 * Update object into database
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function update($id, array $data)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		if (!$id || $id <= 0)
			return false;

		if (count($data) && is_array($data)) {
			// Update request
			$sql = 'UPDATE ' . MAIN_DB_PREFIX . $this->table_element . ' SET';

			foreach ($data as $key => $val) {
				$val = is_numeric($val) ? $val : '"'. $val .'"';
				$sql .= ' `'. $key. '` = '. $val .',';
			}

			$sql  = substr($sql, 0, -1);
			$sql .= ' WHERE rowid = ' . $id;

			$this->db->begin();

			$resql = $this->db->query($sql);

			if (!$resql) {
				$this->db->rollback();
				
				$this->error = 'Error ' . $this->db->lasterror();
				$this->errors[] = $this->error;
				dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

				return -1;
			} else {
				$this->db->commit();

				return 1;
			}
		}
	}

	/**
	 * Delete object in database
	 *
	 * @param User $user      User that deletes
	 * @param bool $notrigger false=launch triggers after, true=disable triggers
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function delete()
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$this->db->begin();

		$sql 	= 'DELETE FROM ' . MAIN_DB_PREFIX . $this->table_element .' WHERE rowid = ' . $this->id;
		$resql 	= $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}

	/**
     * return contract by salary id with filters if available
     *
     * @author Yassine Belkaid <y.belkaid@nextconcept.ma>
     * @return integer
     */
	public function fetchByUser($user_id, $order = '', $filter = '')
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element .' WHERE fk_created_by = '. (int) $user_id.' '. $order;

		// Manage filter
		if (!empty($filter)) {
			$sql .= $filter;
		}

		$this->lines = array();

		$resql = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);

			while ($obj = $this->db->fetch_object($resql)) {
				$line = new stdClass;

				$line->id 				 = $obj->rowid;
				$line->ref_marche	 	 = $obj->ref_marche;
				$line->statut 			 = $obj->statut;
				$line->fk_created_by 	 = $obj->fk_created_by;
				$line->fk_appel_offre	 = $obj->fk_appel_offre;
				$line->label 			 = $obj->label;
				$line->budget 			 = (float) $obj->budget;
				$line->updated_at 		 = $this->db->jdate($obj->updated_at);
				$line->created_at 		 = $this->db->jdate($obj->created_at);
				$this->lines[] 			 = $line;
			}
			$this->db->free($resql);

			return $num;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
     *	Return clicable name (with picto eventually)
     *
     *  @author Yassine Belkaid <y.belkaid@nextconcept.ma>
     *	@param		int			$withpicto		0=_No picto, 1=Includes the picto in the linkn, 2=Picto only
     *	@return		string						String with URL
     */
    public function getNomUrl($withpicto=0,  $id = null, $ref_marche = null)
    {
        global $langs;

        $result	= '';
        $setRef	= '';

        $setRef = (null !== $ref_marche) ? $ref_marche : '';
        $id  	= ($id  ?: '');
        $label  = $langs->trans("Show").': '. $setRef;

        $link 	 = '<a href="'.DOL_URL_ROOT.'/btp/nc_under_modules/marches/marche/index.php?id='. $id .'" title="'.dol_escape_htmltag($label, 1).'" class="classfortooltip">';
        $linkend ='</a>';
        $picto   = $this->defaultPicto;

        if ($withpicto) $result.=($link.img_object($label, $picto, 'class="classfortooltip"').$linkend);
        if ($withpicto && $withpicto != 2) $result.=' ';
        if ($withpicto != 2) $result.=$link.$setRef.$linkend;
        return $result;
    }

    public function getMarketHeader($mid = null, $start_date_market = false)
    {
    	global $langs;

		echo $this->marchesDropDown($this->id);

    	$html = '<table class="border" width="100%">';
		$html .= '<tr><td width="25%">'.$langs->trans("MarketRef").'</td>';
		$html .= '<td colspan="2">'. $this->ref_marche .'</td></tr>';
		$html .= '<tr><td width="25%">'.$langs->trans("MarketLabel").'</td>';
		$html .= '<td colspan="2">'. $this->label .'</td></tr>';
		$html .= '<tr><td width="25%">'.$langs->trans("MarketDate").'</td>';
		$html .= '<td colspan="2">'. dol_print_date($this->sig_date, 'day') .'</td></tr>';
		$html .= '<tr><td width="25%" class="fieldrequired">'.$langs->trans("MarketAmount").'</td>';
		$html .= '<td colspan="2">'. number_format($this->budget, 2).' '.$langs->getCurrencySymbol($conf->currency).'</td></tr>';

		if (false !== $start_date_market && null !== $mid) {
			$execution = new Executions($this->db);
			$execution->fetch(0, $mid);
			if ($execution->id) {
				$start_date	= $execution->start_date ? dol_print_date($execution->start_date, 'day') : '';
				$execution_end_date = Tools::getEndDatFromDays(dol_print_date($execution->start_date, 'dayrfc'), $execution->months, $execution->days);
				$html .= '<tr><td width="25%" class="fieldrequired">'. $langs->trans("CommenceDate") .'</td>';
				$html .= '<td colspan="2">'. $start_date .'</td></tr>';
				if ($execution_end_date) {
					$html .= '<tr><td width="25%" class="fieldrequired">'. $langs->trans("EndDate") .'</td>';
					$html .= '<td colspan="2">'. $execution_end_date->format('d/m/Y') .'</td></tr>';
				}
			}
		}

		$html .= '</table><br />';

		return $html;
    }

    public function marchesDropDown($id)
    {
    	global $langs;

    	if (!$id)
    		return false;

    	$html = '';

    	if (false !== $this->isMarcheCadre($id)) {
    		$id 	 = (int)$id;
    		$marches = $this->countMarches($id);

    		if (count($marches)) {
    			$html .= '<div class="pull-right" style="float:right;margin-bottom:10px;"><div>';
    			$html .= "<span>". $langs->trans('SelectMarketRange') .":</span> ";
    			  $html .= '<select id="marches_cadres">';

	    			foreach ($marches as $key => $marche) {
	    				$selected = ($marche->rowid == $id ? ' selected="selected"' : '');
	    				$label 	  = Tools::trunc_words($marche->label);
	    				$ref_marche= (!empty($marche->ref_marche) ? (strlen($marche->ref_marche) > 20 ? substr(trim(ucfirst($marche->ref_marche)), 0, 20) .'...' : trim(ucfirst($marche->ref_marche))) : '');
	    		  		$html .= '<option value="'. $marche->rowid .'"'. $selected .'>'. ucfirst($ref_marche) .' - '. $label .'</option>';
	    			}

	    		  $html .= '</select></div>';
	    		$html .= "<div style='float: right;margin-right: 0px;margin-top: 10px;'><a class='button' href='". DOL_URL_ROOT ."/marches/marche/card.php?mid=". $id ."&action=clone_marche'>". $langs->trans('CreateMarketRange') ."</a></div>";
	    		$html .= '</div>';
	    		$html .= '<script type="text/javascript">
	    		  var marchObj = document.getElementById("marches_cadres");
	    		  marchObj.addEventListener("change", function() {
	    		  	var marcheId = this.options[marchObj.selectedIndex].value;
	    		  	document.location.href = "'. DOL_URL_ROOT .'/btp/nc_under_modules/marches/marche/index.php?id="+marcheId;
	    		  });
	    		</script>';
    		}
    	}

    	return $html;
    }

    public function cloneMarche($id, $user)
	{
		if (!$id)
			return false;

		$sql = 'SELECT a.ref as cloned_ref, a.label as cloned_label, m.fk_appel_offre as cloned_ao, p.signature_date as cloned_signature_date, c.amount as cloned_amount_ht FROM '. MAIN_DB_PREFIX .'marches as m INNER JOIN '. MAIN_DB_PREFIX .'appel_offre as a INNER JOIN '. MAIN_DB_PREFIX .'m_approval as p INNER JOIN '. MAIN_DB_PREFIX .'m_competitors as c WHERE m.rowid = 1 AND (a.rowid = m.fk_appel_offre AND a.rowid = p.fk_appel_offre AND c.fk_appel_offre = a.rowid) AND c.winner = 1';

		$resql = $this->db->query($sql);

		if ($resql) {
			$numrows = $this->db->num_rows($resql);

			if ($numrows) {
				$obj = $this->db->fetch_object($resql);
				
				return $obj;
			}

			$this->db->free($resql);

			return false;
		} else {
			return false;
		}
	}

    public function statusLevel($statut)
	{
		global $langs;
		$status = '';
		
		switch ($statut) {
			case 0:
				return $langs->trans('Still').' '.img_picto($langs->trans('Still'), 'statut0');
				break;
			case 1:
			case 3:
				return $langs->trans('Archived').' '.img_picto($langs->trans('Archived'), 'statut5');
				break;
		}

		return false;
    }
}
