<?php
	/************************************************
	* Copyright (C) 2016	Sylvain Legrand - <contact@infras.fr>	InfraS - <http://www.infras.fr>
	*
	* This program is free software: you can redistribute it and/or modify
	* it under the terms of the GNU General Public License as published by
	* the Free Software Foundation, either version 3 of the License, or
	* (at your option) any later version.
	*
	* This program is distributed in the hope that it will be useful,
	* but WITHOUT ANY WARRANTY; without even the implied warranty of
	* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	* GNU General Public License for more details.
	*
	* You should have received a copy of the GNU General Public License
	* along with this program.  If not, see <http://www.gnu.org/licenses/>.
	************************************************/

	/************************************************
	* 	\file		../infraspack/core/lib/infras.pdf.lib.php
	* 	\ingroup	InfraS
	* 	\brief		Set of functions used for InfraS PDF generation
	************************************************/
	
	/************************************************
	*	Show top small header of page.
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Object		$object     	Object shown in PDF
	*	@param  int	    	$showaddress    0=no, 1=yes
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	string		$title			string title in connection with objet type
	* 	@param	Societe		$fromcompany	Object company
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	* 	@param	int			$Rounded_rect	rayon des angles du tableau
	* 	@param	string		$logo			Objet logo to show
	************************************************/
	function pdf_InfraS_pagesmallhead(&$pdf, $object, $showaddress, $outputlangs, $title, $fromcompany, $formatpage, $Rounded_rect, $logo)
	{
		global $conf;
		
		$default_font_size	= pdf_getPDFFontSize($outputlangs);
		$pdf->SetTextColor(0, 0, 0);
		$pdf->SetFont('','', $default_font_size - 2);
		$posy				= $formatpage['mhaute'];
		$posx				= $formatpage['largeur'] - $formatpage['mdroite'] - 100 - $Rounded_rect;
		$pdf->SetXY($formatpage['mgauche'],$posy);
		// Logo
		if ($logo)
		{
			if (is_file($logo) && is_readable($logo))	$pdf->Image($logo, $formatpage['mgauche'], $posy, 0, 6);	// width=0 (auto)
			else
			{
				$pdf->SetTextColor(200, 0 ,0);
				$pdf->SetFont('', 'B', $default_font_size - 2);
				$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ErrorLogoFileNotFound", $logo), 0, 'L');
				$pdf->MultiCell(100, 3, $outputlangs->transnoentities("ErrorGoToGlobalSetup"), 0, 'L');
				$pdf->SetTextColor(0, 0, 0);
			}	// else	// if (is_file($logo) && is_readable($logo))
		}	// if ($logo)
		else
		{
			$text	= $fromcompany->name;
			$pdf->MultiCell(100, 4, $outputlangs->convToOutputCharset($text), 0, 'L');
		}	// else	// if ($logo)
		$posy				+= 3;
		$pdf->SetXY($posx,$posy);
		$ref				= $outputlangs->transnoentities("Ref");
		$reference			= $outputlangs->convToOutputCharset($object->ref);
		$date				= dol_print_date($object->date, "day", false, $outputlangs, true);
		if (! empty($date))	$pdf->MultiCell(100, 3, $title." ".$ref." ".$reference." du ".$date, '', 'R');
		else				$pdf->MultiCell(100, 3, $title." ".$ref." ".$reference, '', 'R');
	}	// 	function pdf_InfraS_pagesmallhead(&$pdf, $object, $showaddress, $outputlangs, $title, $fromcompany, $formatpage, $Rounded_rect, $logo)

	/************************************************
	*	Show free text
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Object		$object         Object shown in PDF
	*	@param	int			$posx			Position depart (largeur)
	*	@param	int			$posy			Position depart (hauteur)
	*	@param	Translate	$outputlangs	Objet langs
	* 	@param	Societe		$fromcompany	Object company
	* 	@param	string		$paramfreetext	Constant name of free text
	*	@param	int			$calculseul		Arrête la fonction au calcul de hauteur nécessaire
	* 	@return	int							Return height of free text
	************************************************/
	function pdf_InfraS_free_text(&$pdf, $object, $posx, $posy, $outputlangs, $fromcompany, $paramfreetext, $calculseul = 0)
	{
		global $conf;

		$pdf->startTransaction();
		$posy0	= $posy;
		$line	= '';
		// Line of free text
		if (! empty($conf->global->$paramfreetext))
		{
			// Make substitution
			$substitutionarray						= pdf_getSubstitutionArray($outputlangs, null, $object);
			// More substitution keys
			$substitutionarray['__FROM_NAME__']		= $fromcompany->name;
			$substitutionarray['__FROM_EMAIL__']	= $fromcompany->email;
			complete_substitutions_array($substitutionarray, $outputlangs, $object);
			$newfreetext							= make_substitutions($conf->global->$paramfreetext, $substitutionarray, $outputlangs);
			$newfreetext							= preg_replace('/(<img.*src=")[^\"]*viewimage\.php[^\"]*modulepart=medias[^\"]*file=([^\"]*)("[^\/]*\/>)/', '\1'.DOL_DATA_ROOT.'/medias/\2\3', $newfreetext);
			$line									.= $outputlangs->convToOutputCharset($newfreetext);
		}	// if (! empty($conf->global->$paramfreetext))
		if ($line)	// Free text
		{
			$default_font_size	= pdf_getPDFFontSize($outputlangs);
			$pdf->SetFont('','', $default_font_size - 3);
			$pdf->writeHTMLCell(0, 3, $posx, $posy, dol_htmlentitiesbr($line), 0, 1);
			$posy				= $pdf->GetY();
		}	// if ($line)
		if ($calculseul)
		{
			$heightforfreetext	= ($posy - $posy0) + 3;
			$pdf->rollbackTransaction(true);
			return $heightforfreetext;
		}	// if ($calculseul)
		else
		{
			$pdf->commitTransaction();
			return $posy;
		}	// else // if ($calculseul)
	}	// function pdf_InfraS_free_text(&$pdf, $object, $posx, $posy, $outputlangs, $fromcompany, $paramfreetext, $calculseul = 0)
 
	/************************************************
	*	Show footer of page for PDF generation
	*
	*	@param	TCPDF		$pdf     		The PDF factory
	*	@param  Translate	$outputlangs	Object lang for output
	* 	@param	Societe		$fromcompany	Object company
	* 	@param	array		$formatpage		Page Format => 'largeur', 'hauteur', 'mgauche', 'mdroite', 'mhaute', 'mbasse'
	* 	@param	int			$showdetails	Show company details into footer. This param seems to not be used by standard version.
											(10=minimum, 11 = 1 + Logos de pied, 20 = Show address and managers, 21 = Show 2 + Logos de pied)
	*	@param	int			$hidesupline	Completly hide the line up to footer (for some edition with only table)
	*	@param	int			$calculseul		Arrête la fonction au calcul de hauteur nécessaire
	*	@param	str			$image_foot		File Name of the image to show 
	* 	@param	array		$maxsizeimgfoot	Maximum size for image foot => 'largeur', 'hauteur'
	*	@param	int			$hidepagenum	Hide page num (x/y)
	* 	@return	int							Return height of bottom margin including footer text
	************************************************/
	function pdf_InfraS_pagefoot(&$pdf, $outputlangs, $fromcompany, $formatpage, $showdetails, $hidesupline, $calculseul, $image_foot = '', $maxsizeimgfoot, $hidepagenum = 0)
	{
		global $conf, $user;

		$outputlangs->load("dict");
		$dims	= $pdf->getPageDimensions();
		$pdf->SetTextColor(0, 0, 0);
		
		// First line of company infos
		$line1	= ""; $line2 = ""; $line3 = ""; $line4 = ""; $line5 = 0;
		if (substr($showdetails, 0, 1) == 2)
		{
			if ($fromcompany->name)		$line1	.= ($line1 ? " - " : "").$outputlangs->transnoentities("RegisteredOffice")." : ".$fromcompany->name; // Company name
			if ($fromcompany->address)	$line1	.= ($line1 ? " - " : "").str_replace(CHR(13).CHR(10)," - ",$fromcompany->address); // Address
			if ($fromcompany->zip)		$line1	.= ($line1 ? " - " : "").$fromcompany->zip; // Zip code
			if ($fromcompany->town)		$line1	.= ($line1 ? " " : "").$fromcompany->town; // Town
			if ($fromcompany->phone)	$line2	.= ($line2 ? " - " : "").$outputlangs->transnoentities("Phone")." : ".$fromcompany->phone; // Phone
			if ($fromcompany->fax)		$line2	.= ($line2 ? " - " : "").$outputlangs->transnoentities("Fax")." : ".$fromcompany->fax; // Fax
			if ($fromcompany->url)		$line2	.= ($line2 ? " - " : "").$fromcompany->url; // URL
			if ($fromcompany->email)	$line2	.= ($line2 ? " - " : "").$fromcompany->email; // Email
		}	// if (substr($showdetails, 0, 1) == 2)
		if (substr($showdetails, 0, 1) == 2 || ($fromcompany->country_code == 'DE'))
			if ($fromcompany->managers)	$line2 .= ($line2?" - ":"").$fromcompany->managers; // Managers
		if ($fromcompany->forme_juridique_code)	$line3 .= ($line3?" - ":"").$outputlangs->convToOutputCharset(getFormeJuridiqueLabel($fromcompany->forme_juridique_code)); // Juridical status
		if ($fromcompany->capital) // Capital
		{
			$tmpamounttoshow											= price2num($fromcompany->capital); // This field is a free string
			if (is_numeric($tmpamounttoshow) && $tmpamounttoshow > 0)	$line3	.= ($line3?" - ":"").$outputlangs->transnoentities("CapitalOf",price($tmpamounttoshow, 0, $outputlangs, 0, 0, 0, $conf->currency));
			else														$line3	.= ($line3?" - ":"").$outputlangs->transnoentities("CapitalOf",$tmpamounttoshow,$outputlangs);
		}	// if ($fromcompany->capital)
		if ($fromcompany->idprof1 && ($fromcompany->country_code != 'FR' || ! $fromcompany->idprof2)) // Prof Id 1
		{
			$field										= $outputlangs->transcountrynoentities("ProfId1", $fromcompany->country_code);
			if (preg_match('/\((.*)\)/i',$field,$reg))	$field	= $reg[1];
			$line3										.= ($line3?" - ":"").$field.": ".$outputlangs->convToOutputCharset($fromcompany->idprof1);
		}	// if ($fromcompany->idprof1 && ($fromcompany->country_code != 'FR' || ! $fromcompany->idprof2))
		if ($fromcompany->idprof2) // Prof Id 2
		{
			$field										= $outputlangs->transcountrynoentities("ProfId2", $fromcompany->country_code);
			if (preg_match('/\((.*)\)/i',$field,$reg))	$field	= $reg[1];
			$line3										.= ($line3?" - ":"").$field.": ".$outputlangs->convToOutputCharset($fromcompany->idprof2);
		}	// if ($fromcompany->idprof2)
		if ($fromcompany->idprof3) // Prof Id 3
		{
			$field										= $outputlangs->transcountrynoentities("ProfId3", $fromcompany->country_code);
			if (preg_match('/\((.*)\)/i',$field,$reg))	$field	= $reg[1];
			$line4										.= ($line4?" - ":"").$field.": ".$outputlangs->convToOutputCharset($fromcompany->idprof3);
		}	// if ($fromcompany->idprof3)
		if ($fromcompany->idprof4) // Prof Id 4
		{
			$field										= $outputlangs->transcountrynoentities("ProfId4", $fromcompany->country_code);
			if (preg_match('/\((.*)\)/i',$field,$reg))	$field	= $reg[1];
			$line4										.= ($line4?" - ":"").$field.": ".$outputlangs->convToOutputCharset($fromcompany->idprof4);
		}	// if ($fromcompany->idprof4)
		if ($fromcompany->idprof5) // Prof Id 5
		{
			$field										= $outputlangs->transcountrynoentities("ProfId5", $fromcompany->country_code);
			if (preg_match('/\((.*)\)/i',$field,$reg))	$field	= $reg[1];
			$line4										.= ($line4?" - ":"").$field.": ".$outputlangs->convToOutputCharset($fromcompany->idprof5);
		}	// if ($fromcompany->idprof5)
		if ($fromcompany->idprof6) // Prof Id 6
		{
			$field										= $outputlangs->transcountrynoentities("ProfId6", $fromcompany->country_code);
			if (preg_match('/\((.*)\)/i',$field,$reg))	$field	= $reg[1];
			$line4										.= ($line4?" - ":"").$field.": ".$outputlangs->convToOutputCharset($fromcompany->idprof6);
		}	// if ($fromcompany->idprof6)
			if ($fromcompany->tva_intra != '')	$line4	.= ($line4?" - ":"").$outputlangs->transnoentities("VATIntraShort").": ".$outputlangs->convToOutputCharset($fromcompany->tva_intra); // IntraCommunautary VAT
		if (substr($showdetails, 1, 1) == 1)
		{
			$logospied	= $conf->mycompany->dir_output.'/logos/'.$image_foot;	// Logos partenaires en ligne 5
			if (is_readable($logospied))
			{
				include_once DOL_DOCUMENT_ROOT.'/core/lib/images.lib.php';
				$imglinesize				= pdf_InfraS_getSizeForImage($logospied, $maxsizeimgfoot['largeur'], $maxsizeimgfoot['hauteur']);
				if ($imglinesize['height'])	$line5	= $imglinesize['height'];
			}	// if (is_readable($logospied))
		}	// if (substr($showdetails, 1, 1) == 1)
		$pdf->SetFont('', '', 7);
		$pdf->SetDrawColor(224, 224, 224);
		// The start of the bottom of this page footer is positioned according to # of lines
		$marginwithfooter		= $formatpage['mbasse'] + (! empty($line1) ? 3 : 0) + (! empty($line2) ? 3 : 0) + (! empty($line3) ? 3 : 0) + (! empty($line4) ? 3 : 0) + $line5;
		if ($calculseul == 1)	return $marginwithfooter;
		$posy=$marginwithfooter + 0;
		$pdf->SetY(-$posy);
		$nopage					= $pdf->PageNo();
		$nbpage					= $pdf->getNumPages();
		if (($nopage == $nbpage) && empty($hidesupline))
		{
			$pdf->line($dims['lm'], $dims['hk']-$posy, $dims['wk']-$dims['rm'], $dims['hk']-$posy);
			$posy--;
		}	// if (($nopage == $nbpage) && empty($hidesupline))
		if (! empty($line1))
		{
			$pdf->SetFont('', 'B', 7);
			$pdf->SetXY($dims['lm'], -$posy);
			$pdf->MultiCell($dims['wk'] - $dims['rm'], 2, $line1, 0, 'C', 0);
			$posy				-= 3;
			$pdf->SetFont('', '', 7);
		}	// if (! empty($line1))
		if (! empty($line2))
		{
			$pdf->SetFont('', 'B', 7);
			$pdf->SetXY($dims['lm'], -$posy);
			$pdf->MultiCell($dims['wk'] - $dims['rm'], 2, $line2, 0, 'C', 0);
			$posy				-= 3;
			$pdf->SetFont('', '', 7);
		}	// if (! empty($line2))
		if (! empty($line3))
		{
			$pdf->SetXY($dims['lm'], -$posy);
			$pdf->MultiCell($dims['wk'] - $dims['rm'], 2, $line3, 0, 'C', 0);
			$posy				-= 3;
		}	// if (! empty($line3))
		if (! empty($line4))
		{
			$pdf->SetXY($dims['lm'],-$posy);
			$pdf->MultiCell($dims['wk'] - $dims['rm'], 2, $line4, 0, 'C', 0);
		}	// if (! empty($line4))
		if (is_readable($logospied))
		{
			$posy				-= 3;
			$posxpicture		= $formatpage['mgauche'] + (($formatpage['largeur'] - $formatpage['mgauche'] - $formatpage['mdroite'] - $imglinesize['width']) / 2);	// centre l'image dans la colonne
			$pdf->Image($logospied, $posxpicture, $formatpage['hauteur'] - $posy, $imglinesize['width'], $line5);	// width=0 or height=0 (auto)
		}	// if (is_readable($logospied))
		if (! $hidepagenum) // Show page nb only on iso languages (so default Helvetica font)
		{
			$prevFont									= $pdf->getFontFamily();
			$pdf->SetFont('Helvetica');
			$pdf->SetXY(-27, -$posy);
			if (empty($conf->global->MAIN_USE_FPDF))	$pdf->MultiCell(20, 2, $pdf->PageNo().' / '.$pdf->getAliasNbPages(), 0, 'R', 0);
			else										$pdf->MultiCell(20, 2, $pdf->PageNo().' / {nb}', 0, 'R', 0);
			$pdf->SetFont($prevFont);
		}	// if (! $hidepagenum)
		return $marginwithfooter;
	}	// function pdf_InfraS_pagefoot(&$pdf, $outputlangs, $fromcompany, $formatpage, $showdetails, $hidesupline, $calculseul, $image_foot = '', $maxsizeimgfoot, $hidepagenum = 0)

	/************************************************
	*	Returns the name of the thirdparty
	*
	*	@param	Societe|Contact     $thirdparty     Contact or thirdparty
	*	@param	Translate           $outputlangs    Output language
	*	@param	int                 $includealias   1 = Include alias name before name
	*	@return	string				String with name of thirdparty (+ alias if requested)
	************************************************/
	function pdf_InfraS_Build_Third_party_Name($thirdparty, $outputlangs, $includealias = 0)
	{
		global $conf;
		// Recipient name
		$socname = '';

		if ($thirdparty instanceof Societe)
		{
			if ($includealias && !empty($thirdparty->name_alias))	$socname	.= $thirdparty->name_alias."\n";
			else													$socname	.= $thirdparty->name;
		}	// if ($thirdparty instanceof Societe)
		elseif ($thirdparty instanceof Contact)	$socname	= $thirdparty->socname;
		else												throw new InvalidArgumentException('Parameter 1=$thirdparty is not a Societe nor Contact');

		return $outputlangs->convToOutputCharset($socname);
	}	// function pdf_InfraS_Build_Third_party_Name($thirdparty, Translate $outputlangs, $includealias = 0)

	/********************************************
	*	Return line unit price excluding tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string							Line unit price excluding tax
	 ********************************************/
	function pdf_InfraS_getlineupexcltax($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $conf, $hookmanager;

		$sign											= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))
			$sign										= -1;
		$reshook										= 0;
		$result											= '';
		if (is_object($hookmanager))
		{
			$special_code								= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line)) $special_code = $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters									= array('i'=>$i,'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action='';
			$reshook									= $hookmanager->executeHooks('pdf_getlineupexcltax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 			$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if (empty($hidedetails) || $hidedetails > 1)
			{
				$subprice								= ($conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_subprice : $object->lines[$i]->subprice);
				$result									.= price($sign * $subprice, 0, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraS_getlineupexcltax($object, $i, $outputlangs, $hidedetails = 0)

	/********************************************
	*	Return line unit price including tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string							Line unit price including tax
	 ********************************************/
	function pdf_InfraS_getlineupincltax($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $conf, $hookmanager;

		$sign											= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))
			$sign										= -1;
		$reshook										= 0;
		$result											= '';
		if (is_object($hookmanager))
		{
			$special_code								= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line)) $special_code = $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters									= array('i'=>$i,'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action='';
			$reshook									= $hookmanager->executeHooks('pdf_getlineupwithtax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 			$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if (empty($hidedetails) || $hidedetails > 1)
			{
				$subprice								= ($conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_subprice : $object->lines[$i]->subprice);
				$result									.= price($sign * ($subprice + ($subprice * ($object->lines[$i]->tva_tx) / 100)), 0, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraS_getlineupincltax($object, $i, $outputlangs, $hidedetails = 0)

	/********************************************
	*	Return line unit price with discount and excluding tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0=no, 1=yes, 2=just special lines)
	* 	@return	string							Line unit price with discount and excluding tax
	 ********************************************/
	function pdf_InfraS_getlineincldiscountexcltax($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $conf, $hookmanager;

		$sign											= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))
			$sign										= -1;
		$reshook										= 0;
		$result											= '';
		if (is_object($hookmanager))
		{
			$special_code								= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line)) $special_code = $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters									= array('i'=>$i,'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action='';
			$reshook									= $hookmanager->executeHooks('pdf_getlineupexcltax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 			$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				$total_ht								= ($conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ht : $object->lines[$i]->total_ht);
				$qty									= $object->lines[$i]->qty;
				$result									.= price($sign * ($total_ht / $qty), 0, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraS_getlineincldiscountexcltax($object, $i, $outputlangs, $hidedetails = 0)

	/********************************************
	*	Return line unit price with discount and including tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param 	Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide value (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string							Line unit price with discount and including tax
	 ********************************************/
	function pdf_InfraS_getlineincldiscountincltax($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $conf, $hookmanager;

		$sign											= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))
			$sign										= -1;
		$reshook										= 0;
		$result											= '';
		if (is_object($hookmanager))
		{
			$special_code								= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line)) $special_code = $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters									= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action										= '';
			$reshook									= $hookmanager->executeHooks('pdf_getlineupwithtax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 			$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				$total_ttc								= ($conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ttc : $object->lines[$i]->total_ttc);
				$qty									= $object->lines[$i]->qty;
				$result									.= price($sign * ($total_ttc / $qty), 0, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraS_getlineincldiscountincltax($object, $i, $outputlangs, $hidedetails = 0)

	/********************************************
	*	Return total of line excluding tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param 	Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide value (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string							Total of line excluding tax
	 ********************************************/
	function pdf_InfraS_getlinetotalexcltax($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $conf, $hookmanager;

		$sign											= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))
			$sign										= -1;
		$reshook										= 0;
		$result											= '';
		if (is_object($hookmanager))
		{
			$special_code								= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line)) $special_code = $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters									= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action										= '';
			$reshook									= $hookmanager->executeHooks('pdf_getlinetotalexcltax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint)) 			$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				$total_ht								= ($conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ht : $object->lines[$i]->total_ht);
				if ($object->lines[$i]->situation_percent > 0)
				{
					$prev_progress						= 0;
					$progress							= 1;
					if (method_exists($object, 'get_prev_progress'))
					{
						$prev_progress					= $object->lines[$i]->get_prev_progress($object->id);
						$progress						= ($object->lines[$i]->situation_percent - $prev_progress) / 100;
					}	// if (method_exists($object, 'get_prev_progress'))
					$result								.= price($sign * ($total_ht / ($object->lines[$i]->situation_percent / 100)) * $progress, 0, $outputlangs);
				}	// if ($object->lines[$i]->situation_percent > 0)
				else									$result	.= price($sign * $total_ht, 0, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraS_getlinetotalexcltax($object, $i, $outputlangs, $hidedetails = 0)

	/********************************************
	*	Return total of line including tax
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param 	Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide value (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string							Total of line including tax
	 ********************************************/
	function pdf_InfraS_getlinetotalincltax($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $conf, $hookmanager;

		$sign											= 1;
		if (isset($object->type) && $object->type == 2 && ! empty($conf->global->INVOICE_POSITIVE_CREDIT_NOTE))
			$sign										= -1;
		$reshook										= 0;
		$result											= '';
		if (is_object($hookmanager))
		{
			$special_code								= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line)) $special_code = $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters									= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code, 'sign'=>$sign);
			$action										= '';
			$reshook									= $hookmanager->executeHooks('pdf_getlinetotalwithtax', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if(!empty($hookmanager->resPrint))			$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($object->lines[$i]->special_code == 3)	return $outputlangs->transnoentities("Option");
			if (empty($hidedetails) || $hidedetails > 1)
			{
				$total_ttc								= ($conf->multicurrency->enabled && $object->multicurrency_tx != 1 ? $object->lines[$i]->multicurrency_total_ttc : $object->lines[$i]->total_ttc);
				if ($object->lines[$i]->situation_percent > 0)
				{
					$prev_progress						= 0;
					$progress							= 1;
					if (method_exists($object, 'get_prev_progress'))
					{
						$prev_progress					= $object->lines[$i]->get_prev_progress($object->id);
						$progress						= ($object->lines[$i]->situation_percent - $prev_progress) / 100;
					}	// if (method_exists($object, 'get_prev_progress'))
					$result								.= price($sign * ($total_ttc / ($object->lines[$i]->situation_percent / 100)) * $progress, 0, $outputlangs);
				}	// if ($object->lines[$i]->situation_percent > 0)
				else									$result	.= price($sign * $total_ttc, 0, $outputlangs);
			}	// if (empty($hidedetails) || $hidedetails > 1)
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraS_getlinetotalincltax($object, $i, $outputlangs, $hidedetails = 0)

	/********************************************
	*	Return dimensions to use for images onto PDF,
	*	checking that width and height are not higher than maximum (20x32 by default).
	*
	*	@param	string		$realpath			Full path to photo file to use
	*	@param	int			$maxwidth			Maximum width to use
	*	@param	int			$maxheight			Maximum height to use
	*	@return	array							Height and width to use to output image (in pdf user unit, so mm)
	 ********************************************/
	function pdf_InfraS_getSizeForImage($realpath, $maxwidth, $maxheight)
	{
		global $conf;

		include_once DOL_DOCUMENT_ROOT.'/core/lib/images.lib.php';
		$imglinesize	= dol_getImageSize($realpath);
		if ($imglinesize['height'])
		{
			$width	= (int) round($maxheight * $imglinesize['width'] / $imglinesize['height']);	// I try to use maxheight
			if ($width > $maxwidth)	// Pb with maxheight, so i use maxwidth
			{
				$width	= $maxwidth;
				$height	= (int) round($maxwidth * $imglinesize['height'] / $imglinesize['width']);
			}	// if ($width > $maxwidth)
			else	$height	= $maxheight;	// No pb with maxheight
		}	// if ($imglinesize['height'])
		return array('width'=>$width, 'height'=>$height);
	}	// function pdf_InfraS_getSizeForImage($realpath, $maxwidth, $maxheight)

	/********************************************
	*	Add a draft watermark on PDF files
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Translate	$outputlangs	Object lang
	*	@param  string		$text           Text to show
	*	@param  int		    $center_y       Y center of rotation
	*	@param  int		    $w		        Width of table
	*	@param  int		    $hp		        Height of page
	*	@param  string	    $unit           Unit of height (mm, pt, ...)
	*	@return	void
	 ********************************************/
	function pdf_InfraS_watermark(&$pdf, $outputlangs, $text, $center_y, $w, $hp, $unit)
	{
		// Print Draft Watermark
		if ($unit=='pt')		$k = 1;
		elseif ($unit=='mm')	$k = 72/25.4;
		elseif ($unit=='cm')	$k = 72/2.54;
		elseif ($unit=='in')	$k = 72;
		$savx					= $pdf->getX();
		$savy					= $pdf->getY();
		$pdf->SetFont('', 'B', 40);
		$pdf->SetTextColor(255, 192, 203);
		$watermark_angle		= 20 / 180 * pi();	// angle de rotation 20° en radian
		$center_x				= $w / 2;			// x centre
		$pdf->SetFont('', 'B', 40);
		$pdf->SetTextColor(255, 192, 203);
		//rotate
		$pdf->_out(sprintf('q %.5F %.5F %.5F %.5F %.2F %.2F cm 1 0 0 1 %.2F %.2F cm', cos($watermark_angle), sin($watermark_angle), 
					-sin($watermark_angle), cos($watermark_angle), $center_x * $k, ($hp - $center_y) * $k, -$center_x * $k, -($hp - $center_y) * $k));
		//print watermark
		$pdf->SetXY(10, $center_y - 10);
		$pdf->Cell($w, 20, $outputlangs->convToOutputCharset($text), "", 2, "C", 0);
		//antirotate
		$pdf->_out('Q');
		$pdf->SetXY($savx, $savy);
		$pdf->SetTextColor(0, 0, 0);
	}	// function pdf_InfraS_watermark(&$pdf, $outputlangs, $text, $center_y, $w, $hp, $unit)

	/********************************************
	*	Output line description into PDF
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param	Object		$object			Object shown in PDF
	*	@param	int			$i				Current line number
	*	@param  Translate	$outputlangs	Object lang for output
	*	@param  int			$w				Width
	*	@param  int			$h				Height
	*	@param  int			$posx			Pos x
	*	@param  int			$posy			Pos y
	*	@param  int			$hideref       	Hide reference
	*	@param  int			$hidedesc		Hide description
	*	@param	int			$issupplierline	Is it a line for a supplier object ?
	*	@return	string
	 ********************************************/
	function pdf_InfraS_writelinedesc(&$pdf, $object, $i, $outputlangs, $w, $h, $posx, $posy, $hideref = 0, $hidedesc = 0, $issupplierline = 0)
	{
		global $db, $conf, $langs, $hookmanager;

		$reshook	= 0;
		$result		= '';
		if (is_object($hookmanager))
		{
			$special_code										= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))	$special_code	= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters											= array('pdf'=>$pdf, 'i'=>$i, 'outputlangs'=>$outputlangs, 'w'=>$w, 'h'=>$h, 'posx'=>$posx, 'posy'=>$posy, 'hideref'=>$hideref, 'hidedesc'=>$hidedesc, 'issupplierline'=>$issupplierline, 'special_code'=>$special_code);
			$action												= '';
			$reshook											= $hookmanager->executeHooks('pdf_writelinedesc', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			if (!empty($hookmanager->resPrint))					$result	.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			$labelproductservice	= pdf_InfraS_getlinedesc($object, $i, $outputlangs, $hideref, $hidedesc, $issupplierline);
			// Description
			$pdf->writeHTMLCell($w, $h, $posx, $posy, $outputlangs->convToOutputCharset($labelproductservice), 0, 1, false, true, 'J', true);
			$result					.= $labelproductservice;
		}	// if (empty($reshook))
		return $result;
	}	// function pdf_InfraS_writelinedesc(&$pdf, $object, $i, $outputlangs, $w, $h, $posx, $posy, $hideref = 0, $hidedesc = 0, $issupplierline = 0)

	/********************************************
	*  Return line description translated in outputlangs and encoded into htmlentities and with <br>
	*
	*  @param  Object		$object			Object shown in PDF
	*  @param  int			$i				Current line number (0 = first line, 1 = second line, ...)
	*  @param  Translate	$outputlangs	Object langs for output
	*  @param  int			$hideref		Hide reference
	*  @param  int			$hidedesc		Hide description
	*  @param  int			$issupplierline	Is it a line for a supplier object ?
	*  @return string						String with line
	 ********************************************/
	function pdf_InfraS_getlinedesc(&$object, $i, $outputlangs, $hideref = 0, $hidedesc = 0, $issupplierline = 0)
	{
		global $db, $conf, $langs;

		$idprod								= (! empty($object->lines[$i]->fk_product) ? $object->lines[$i]->fk_product : false);
		$label								= (! empty($object->lines[$i]->label) ? $object->lines[$i]->label : (! empty($object->lines[$i]->product_label) ? $object->lines[$i]->product_label : ''));
		$desc								= (! empty($object->lines[$i]->desc) ? $object->lines[$i]->desc : (! empty($object->lines[$i]->description) ? $object->lines[$i]->description : ''));
		$note								= (! empty($object->lines[$i]->note) ? $object->lines[$i]->note : '');
		$dbatch								= (! empty($object->lines[$i]->detail_batch) ? $object->lines[$i]->detail_batch : false);

		if ($issupplierline)
		{
			$ref_supplier		= (! empty($object->lines[$i]->ref_supplier) ? $object->lines[$i]->ref_supplier : (! empty($object->lines[$i]->ref_fourn) ? $object->lines[$i]->ref_fourn : ''));
			if (empty($ref_supplier))
			{
				$ref			= $object->lines[$i]->ref;
				$sql			= 'SELECT pfp.ref_fourn ';
				$sql			.= 'FROM '.MAIN_DB_PREFIX.'product AS p ';
				$sql			.= 'LEFT JOIN '.MAIN_DB_PREFIX.'product_fournisseur_price AS pfp ON p.rowid = pfp.fk_product ';
				$sql			.= 'WHERE p.ref = "'.$ref.'" AND pfp.fk_soc = "'.$object->thirdparty->id.'"';
				$resql			= $object->db->query($sql);
				if ($resql)
				{
					$obj		= $db->fetch_object($resql);
					if ($obj)	$ref_supplier	= $obj->ref_fourn;
				}	// if ($resql)
				$db->free($resql);
			}	// if (empty($ref_supplier))
			$prodser			= new ProductFournisseur($db);
		}	// if ($issupplierline)
		else					$prodser	= new Product($db);

		if ($idprod)
		{
			$prodser->fetch($idprod);
			// If a predefined product and multilang and on other lang, we renamed label with label translated
			if (! empty($conf->global->MAIN_MULTILANGS) && ($outputlangs->defaultlang != $langs->defaultlang))
			{
				$translatealsoifmodified	= (! empty($conf->global->MAIN_MULTILANG_TRANSLATE_EVEN_IF_MODIFIED));	// By default if value was modified manually, we keep it (no translation because we don't have it)

				// TODO Instead of making a compare to see if param was modified, check that content contains reference translation. If yes, add the added part to the new translation
				// ($textwasmodified is replaced with $textwasmodifiedorcompleted and we add completion).

				// Set label
				// If we want another language, and if label is same than default language (we did force it to a specific value), we can use translation.
				//var_dump($outputlangs->defaultlang.' - '.$langs->defaultlang.' - '.$label.' - '.$prodser->label);exit;
				$textwasmodified			= ($label == $prodser->label);
				if (! empty($prodser->multilangs[$outputlangs->defaultlang]["label"]) && ($textwasmodified || $translatealsoifmodified))	$label=$prodser->multilangs[$outputlangs->defaultlang]["label"];

				// Set desc
				// Manage HTML entities description test because $prodser->description is store with htmlentities but $desc no
				$textwasmodified																												= false;
				if (!empty($desc) && dol_textishtml($desc) && !empty($prodser->description) && dol_textishtml($prodser->description))			$textwasmodified	= (strpos(dol_html_entity_decode($desc,ENT_QUOTES | ENT_HTML401), dol_html_entity_decode($prodser->description,ENT_QUOTES | ENT_HTML401)) !== false);
				else																															$textwasmodified	= ($desc == $prodser->description);
				if (! empty($prodser->multilangs[$outputlangs->defaultlang]["description"]) && ($textwasmodified || $translatealsoifmodified))	$desc				= $prodser->multilangs[$outputlangs->defaultlang]["description"];
				// Set note
				$textwasmodified																												= ($note == $prodser->note);
				if (! empty($prodser->multilangs[$outputlangs->defaultlang]["note"]) && ($textwasmodified || $translatealsoifmodified))			$note				= $prodser->multilangs[$outputlangs->defaultlang]["note"];
			}	// if (! empty($conf->global->MAIN_MULTILANGS) && ($outputlangs->defaultlang != $langs->defaultlang))
		}	// if ($idprod)
		// Description short of product line
		$labelbold						= isset($conf->global->INFRAS_PDF_LABEL_BOLD) ? $conf->global->INFRAS_PDF_LABEL_BOLD : 0;
		if ($labelbold && $label)		$libelleproduitservice	= "<b>".$label."</b>";
		else							$libelleproduitservice	= $label;
		// Description long of product line
		if (! empty($desc) && ($desc != $label))
		{
			if ($libelleproduitservice && empty($hidedesc))				$libelleproduitservice	.= '__N__';
			if ($desc == '(CREDIT_NOTE)' && $object->lines[$i]->fk_remise_except)
			{
				$discount												= new DiscountAbsolute($db);
				$discount->fetch($object->lines[$i]->fk_remise_except);
				$libelleproduitservice									= $outputlangs->transnoentitiesnoconv("DiscountFromCreditNote", $discount->ref_facture_source);
			}	// if ($desc == '(CREDIT_NOTE)' && $object->lines[$i]->fk_remise_except)
			elseif ($desc == '(DEPOSIT)' && $object->lines[$i]->fk_remise_except)
			{
				$discount												= new DiscountAbsolute($db);
				$discount->fetch($object->lines[$i]->fk_remise_except);
				$libelleproduitservice									= $outputlangs->transnoentitiesnoconv("DiscountFromDeposit", $discount->ref_facture_source);
				// Add date of deposit
				if (! empty($conf->global->INVOICE_ADD_DEPOSIT_DATE))	echo ' ('.dol_print_date($discount->datec, 'day', '', $outputlangs).')';
			}	// elseif ($desc == '(DEPOSIT)' && $object->lines[$i]->fk_remise_except)
			if ($desc == '(EXCESS RECEIVED)' && $object->lines[$i]->fk_remise_except)
			{
				$discount												= new DiscountAbsolute($db);
				$discount->fetch($object->lines[$i]->fk_remise_except);
				$libelleproduitservice									= $outputlangs->transnoentitiesnoconv("DiscountFromExcessReceived", $discount->ref_facture_source);
			}	// if ($desc == '(EXCESS RECEIVED)' && $object->lines[$i]->fk_remise_except)
			else
			{
				if ($idprod)
				{
					if (empty($hidedesc))								$libelleproduitservice	.= $desc;
				}
				else													$libelleproduitservice	.= $desc;
			}	// else	// elseif ($desc == '(DEPOSIT)' && $object->lines[$i]->fk_remise_except)
		}	// if (! empty($desc) && ($desc != $label))
		// We add ref of product (and supplier ref if defined)
		$prefix_prodserv	= "";
		$ref_prodserv		= "";
		if (! empty($conf->global->PRODUCT_ADD_TYPE_IN_DOCUMENTS))   // In standard mode, we do not show this
		{
			if ($prodser->isService())	$prefix_prodserv	= $outputlangs->transnoentitiesnoconv("Service")." ";
			else						$prefix_prodserv	= $outputlangs->transnoentitiesnoconv("Product")." ";
		}	// if (! empty($conf->global->PRODUCT_ADD_TYPE_IN_DOCUMENTS))
		if (empty($hideref))
		{
			if ($issupplierline)
			{
				if ($conf->global->PDF_HIDE_PRODUCT_REF_IN_SUPPLIER_LINES == 1)		$ref_prodserv	= $ref_supplier;
				elseif ($conf->global->PDF_HIDE_PRODUCT_REF_IN_SUPPLIER_LINES == 2)	$ref_prodserv	= $ref_supplier.' ('.$outputlangs->transnoentitiesnoconv("InternalRef").' '.$prodser->ref.')';
				else	// Common case
				{
					$ref_prodserv		= $prodser->ref; // Show local ref
					if ($ref_supplier)	$ref_prodserv	.= ($prodser->ref ? ' (' : '').$outputlangs->transnoentitiesnoconv("SupplierRef").' '.$ref_supplier.($prodser->ref ? ')' : '');
				}	// else	// elseif ($conf->global->PDF_HIDE_PRODUCT_REF_IN_SUPPLIER_LINES == 2)	// if ($conf->global->PDF_HIDE_PRODUCT_REF_IN_SUPPLIER_LINES == 1)
			}	// if ($issupplierline)
			else															$ref_prodserv	= $prodser->ref; // Show local ref only
			if (! empty($libelleproduitservice) && ! empty($ref_prodserv))	$ref_prodserv	.= " - ";
		}	// if (empty($hideref))
		$libelleproduitservice	= $prefix_prodserv.$ref_prodserv.$libelleproduitservice;
		// Add an additional description for the category products
		if (! empty($conf->global->CATEGORY_ADD_DESC_INTO_DOC) && $idprod && ! empty($conf->categorie->enabled))
		{
			include_once DOL_DOCUMENT_ROOT.'/categories/class/categorie.class.php';
			$categstatic	= new Categorie($db);
			$tblcateg		= $categstatic->containing($idprod, Categorie::TYPE_PRODUCT);	// recovering the list of all the categories linked to product
			foreach($tblcateg as $cate)
			{
				$desccateg		= $cate->add_description;	// Adding the descriptions if they are filled
				if ($desccateg)	$libelleproduitservice	.= '__N__'.$desccateg;
			}	// foreach($tblcateg as $cate)
		}	// if (! empty($conf->global->CATEGORY_ADD_DESC_INTO_DOC) && $idprod && ! empty($conf->categorie->enabled))
		if (! empty($object->lines[$i]->date_start) || ! empty($object->lines[$i]->date_end))
		{
			$format='day';
			// Show duration if exists
			if ($object->lines[$i]->date_start && $object->lines[$i]->date_end)		$period	= '('.$outputlangs->transnoentitiesnoconv('DateFromTo', dol_print_date($object->lines[$i]->date_start, $format, false, $outputlangs), dol_print_date($object->lines[$i]->date_end, $format, false, $outputlangs)).')';
			if ($object->lines[$i]->date_start && ! $object->lines[$i]->date_end)	$period	= '('.$outputlangs->transnoentitiesnoconv('DateFrom', dol_print_date($object->lines[$i]->date_start, $format, false, $outputlangs)).')';
			if (! $object->lines[$i]->date_start && $object->lines[$i]->date_end)	$period = '('.$outputlangs->transnoentitiesnoconv('DateUntil', dol_print_date($object->lines[$i]->date_end, $format, false, $outputlangs)).')';
			$libelleproduitservice													.= "__N__".$period;
		}	// if (! empty($object->lines[$i]->date_start) || ! empty($object->lines[$i]->date_end))
		if ($dbatch)
		{
			$format='day';
			foreach($dbatch as $detail)
			{
				$dte=array();
				if ($detail->eatby)		$dte[]	= $outputlangs->transnoentitiesnoconv('printEatby', dol_print_date($detail->eatby, $format, false, $outputlangs));
				if ($detail->sellby)	$dte[]	= $outputlangs->transnoentitiesnoconv('printSellby', dol_print_date($detail->sellby, $format, false, $outputlangs));
				if ($detail->batch)		$dte[]	= $outputlangs->transnoentitiesnoconv('printBatch', $detail->batch);
				$dte[]					= $outputlangs->transnoentitiesnoconv('printQty', $detail->dluo_qty);
				$libelleproduitservice	.= "__N__  ".implode(" - ", $dte);
			}	// foreach($dbatch as $detail)
		}	// if ($dbatch)
		// Now we convert \n into br
		if (dol_textishtml($libelleproduitservice))	$libelleproduitservice	= preg_replace('/__N__/', '<br>', $libelleproduitservice);
		else										$libelleproduitservice	= preg_replace('/__N__/', "\n", $libelleproduitservice);
		$libelleproduitservice						= dol_htmlentitiesbr($libelleproduitservice, 1);
		return $libelleproduitservice;
	}	// function pdf_InfraS_getlinedesc($object, $i, $outputlangs, $hideref = 0, $hidedesc = 0, $issupplierline = 0)

	/************************************************
	*	Show CGV for PDF generation
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  string		$cgv			PDF file name
	* 	@param	Object		$tplidx			Object template
	*	@param	int			$hidepagenum	Hide page num (x/y)
	************************************************/
	function pdf_InfraS_CGV(&$pdf, $cgv, $tplidx, $hidepagenum = 0)
	{
		global $conf;
		
		$path			= ($conf->entity > 1 ? "/".$conf->entity : '');
		$cgv_pdf		= DOL_DATA_ROOT.$path.'/mycompany/'.$cgv;
		pdf_InfraS_Merge($pdf, $cgv_pdf, $tplidx, $hidepagenum);
	}	// function pdf_InfraS_CGV($pdf, $cgv_pdf, $tplidx, $hidepagenum = 0)
		
	/************************************************
	*	Show CGV for PDF generation
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  string		$infile			PDF file full name (with path) to merge
	* 	@param	Object		$tplidx			Object template
	*	@param	int			$hidepagenum	Hide page num (x/y)
	************************************************/
	function pdf_InfraS_Merge(&$pdf, $infile, $tplidx, $hidepagenum = 0)
	{
		global $conf;
		
		if (file_exists($infile) && is_readable($infile))
		{
			$pagecount												= $pdf->setSourceFile($infile);
			for ($i = 1; $i <= $pagecount; $i ++)
			{
				$tplIdx	= $pdf->importPage($i);
				if ($tplIdx !== false)
				{
					$s												= $pdf->getTemplatesize($tplIdx);
					$pdf->AddPage($s['h'] > $s['w'] ? 'P' : 'L');
					$pdf->useTemplate($tplIdx);
					if (! $hidepagenum)
					{
						$prevFont									= $pdf->getFontFamily();
						$pdf->SetFont('Helvetica');
						$pdf->SetXY($conf->global->INFRAS_PDF_X_PAGE_NUM, $conf->global->INFRAS_PDF_Y_PAGE_NUM);
						if (empty($conf->global->MAIN_USE_FPDF))	$pdf->MultiCell(20, 2, $pdf->PageNo().' / '.$pdf->getAliasNbPages(), 0, 'R', 0);
						else										$pdf->MultiCell(20, 2, $pdf->PageNo().' / {nb}', 0, 'R', 0);
						$pdf->SetFont($prevFont);
					}	// if (! $hidepagenum)
				}	// if ($tplIdx!==false)
				else												setEventMessages(null, array($infile.' cannot be added, probably protected PDF'), 'warnings');
			}	// for ($i = 1; $i <= $pagecount; $i ++)
		}	// if (file_exists($infile) && is_readable($infile))
	}	// function pdf_InfraS_Merge($pdf, $cgv_pdf, $tplidx, $hidepagenum = 0)

	/************************************************
	*	Show linked objects for PDF generation
	*
	*	@param	TCPDF		$pdf            The PDF factory
	*	@param  Object		$object			Object shown in PDF
	*	@param	Translate	$outputlangs	Output langs object
	*	@param  int			$posx			Pos x
	*	@param  int			$posy			Pos y
	*	@param  int			$w				Width of cells. If 0, they extend up to the right margin of the page
	*	@param  int			$tab_hl			Cell minimum height. The cell extends automatically if needed.
	*	@param	int			$align			Align
	*	@return	float						The Y PDF position
	************************************************/
	function pdf_InfraS_writeLinkedObjects(&$pdf, $object, $outputlangs, $posx, $posy, $w, $tab_hl, $align)
	{
		$linkedobjects										= pdf_InfraS_getLinkedObjects($object, $outputlangs);
		if (! empty($linkedobjects))
		{
			foreach($linkedobjects as $linkedobject)
			{
				$reftoshow									= $linkedobject["ref_title"].' : '.$linkedobject["ref_value"];
				if (! empty($linkedobject["date_value"]))	$reftoshow	.= ' / '.$linkedobject["date_value"];
				$posy										+= $tab_hl - 0.5;
				$pdf->SetXY($posx, $posy);
				$pdf->MultiCell($w, $tab_hl, $reftoshow, '', $align);
			}	// foreach($linkedobjects as $linkedobject)
		}	// if (! empty($linkedobjects))
		return $pdf->getY();
	}	// function pdf_InfraS_writeLinkedObjects(&$pdf, $object, $outputlangs, $posx, $posy, $w, $tab_hl, $align)

	/************************************************
	*	Return linked objects to use for document generation.
	*	Warning: To save space, this function returns only one link per link type (all links are concated on same record string).
	*	This function is used by pdf_InfraS_writeLinkedObjects
	*
	*	@param  Object		$object			Object shown in PDF
	*	@param	Translate	$outputlangs	Output langs object
	* 	@return	array                       Linked objects
	************************************************/
	function pdf_InfraS_getLinkedObjects($object, $outputlangs)
	{
		global $hookmanager;

		$linkedobjects	= array();
		$object->fetchObjectLinked();
		foreach($object->linkedObjects as $objecttype => $objects)
		{
			if ($objecttype == 'facture')
			{
			   // For invoice, we don't want to have a reference line on document. Image we are using recuring invoice, we will have a line longer than document width.
			}	// if ($objecttype == 'facture')
			elseif ($objecttype == 'propal' || $objecttype == 'supplier_proposal')
			{
				$outputlangs->load('propal');
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("RefProposal");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref);
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("DatePropal");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->date, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif ($objecttype == 'propal' || $objecttype == 'supplier_proposal')
			elseif ($objecttype == 'commande' || $objecttype == 'supplier_order')
			{
				$outputlangs->load('orders');
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("RefOrder");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref).($elementobject->ref_client ? ' ('.$elementobject->ref_client.')' : '') . ($elementobject->ref_supplier ? ' ('.$elementobject->ref_supplier.')' : '');
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("OrderDate");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->date, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif ($objecttype == 'commande' || $objecttype == 'supplier_order')
			elseif ($objecttype == 'contrat')
			{
				$outputlangs->load('contracts');
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("RefContract");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref);
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("DateContract");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->date_contrat, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif ($objecttype == 'contrat')
			elseif ($objecttype == 'fichinter')
			{
				$outputlangs->load('interventions');
				foreach($objects as $elementobject)
				{
					$linkedobjects[$objecttype]['ref_title']	= $outputlangs->transnoentities("PDFInfraSRefInter");
					$linkedobjects[$objecttype]['ref_value']	= $outputlangs->transnoentities($elementobject->ref);
					$linkedobjects[$objecttype]['date_title']	= $outputlangs->transnoentities("Date");
					$linkedobjects[$objecttype]['date_value']	= dol_print_date($elementobject->datec, 'day', '', $outputlangs);
				}	// foreach($objects as $elementobject)
			}	// elseif ($objecttype == 'fichinter')
			elseif ($objecttype == 'shipping')
			{
				$outputlangs->load('orders');
				$outputlangs->load('sendings');
				foreach($objects as $x => $elementobject)
				{
					$order														= null;
					// We concat this record info into fields xxx_value. title is overwrote.
					if (empty($object->linkedObjects['commande']) && $object->element != 'commande')	// There is not already a link to order and object is not the order, so we show also info with order
					{
						$elementobject->fetchObjectLinked();
						if (! empty($elementobject->linkedObjects['commande']))	$order	= reset($elementobject->linkedObjects['commande']);
					}	// if (empty($object->linkedObjects['commande']) && $object->element != 'commande')
					if (! is_object($order))
					{
						$linkedobjects[$objecttype]['ref_title']				= $outputlangs->transnoentities("RefSending");
						if (! empty($linkedobjects[$objecttype]['ref_value']))	$linkedobjects[$objecttype]['ref_value']	.=' / ';
						$linkedobjects[$objecttype]['ref_value']				.= $outputlangs->transnoentities($elementobject->ref);
					}	// if (! is_object($order))
					else
					{
						$linkedobjects[$objecttype]['ref_title']				= $outputlangs->transnoentities("RefOrder").' / '.$outputlangs->transnoentities("RefSending");
						if (empty($linkedobjects[$objecttype]['ref_value']))	$linkedobjects[$objecttype]['ref_value']	= $outputlangs->convToOutputCharset($order->ref) . ($order->ref_client ? ' ('.$order->ref_client.')' : '');
						$linkedobjects[$objecttype]['ref_value']				.= ' / '.$outputlangs->transnoentities($elementobject->ref);
					}	// else	// if (! is_object($order))
				}	// foreach($objects as $x => $elementobject)
			}	// elseif ($objecttype == 'shipping')
		}	// foreach($object->linkedObjects as $objecttype => $objects)
		// For add external linked objects
		if (is_object($hookmanager))
		{
			$parameters								= array('linkedobjects' => $linkedobjects, 'outputlangs' => $outputlangs);
			$action									= '';
			$hookmanager->executeHooks('pdf_getLinkedObjects', $parameters, $object, $action);
			if (! empty($hookmanager->resArray))	$linkedobjects	= $hookmanager->resArray;
		}	// if (is_object($hookmanager))
		return $linkedobjects;
	}	// function pdf_InfraS_getLinkedObjects($object, $outputlangs)

	/************************************************
	*	Return a string with full address formated for output on documents
	*
	*	@param	Translate	$outputlangs		Output langs object
	*	@param  Societe		$sourcecompany		Source company object
	*	@param  Societe		$targetcompany		Target company object
	*	@param  Contact		$targetcontact		Target contact object
	*	@param	int			$usecontact			Use contact instead of company
	*	@param	int			$mode				Address type ('source', 'target', 'targetwithdetails', 'targetwithdetails_xxx': target but include also phone/fax/email/url)
	*	@param  Object      $object				Object we want to build document for
	*	@return	string							String with full address
	************************************************/
	function pdf_InfraS_build_address($outputlangs, $sourcecompany, $targetcompany = '', $targetcontact = '', $usecontact = 0, $mode = 'source', $object = null, $profids = 0)
	{
		global $conf, $hookmanager;

		if ($mode == 'source' && ! is_object($sourcecompany))						return -1;
		if ($mode == 'target' && ! is_object($targetcompany))						return -1;
		if (! empty($sourcecompany->state_id) && empty($sourcecompany->departement))		$sourcecompany->departement		= getState($sourcecompany->state_id); //TODO deprecated
		if (! empty($sourcecompany->state_id) && empty($sourcecompany->state))				$sourcecompany->state			= getState($sourcecompany->state_id);
		if (! empty($sourcecompany->state_id) && !isset($sourcecompany->departement_id))	$sourcecompany->departement_id	= getState($sourcecompany->state_id, '2');
		if (! empty($targetcompany->state_id) && empty($targetcompany->departement))		$targetcompany->departement		= getState($targetcompany->state_id); //TODO deprecated
		if (! empty($targetcompany->state_id) && empty($targetcompany->state))				$targetcompany->state			= getState($targetcompany->state_id);
		if (! empty($targetcompany->state_id) && !isset($targetcompany->departement_id))	$targetcompany->departement_id	= getState($targetcompany->state_id, '2');
		$reshook											= 0;
		$stringaddress										= '';
		if (is_object($hookmanager))
		{
			$parameters										= array('sourcecompany'=>&$sourcecompany, 'targetcompany'=>&$targetcompany, 'targetcontact'=>$targetcontact, 'outputlangs'=>$outputlangs, 'mode'=>$mode, 'usecontact'=>$usecontact);
			$action											= '';
			$reshook										= $hookmanager->executeHooks('pdf_build_address', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			$stringaddress									.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))
		{
			if ($mode == 'source' || $mode == 'sourcewithnodetails')
			{
				$withCountry								= 0;
				if (!empty($sourcecompany->country_code) && ($targetcompany->country_code != $sourcecompany->country_code))	$withCountry	= 1;
				$stringaddress								.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($sourcecompany, $withCountry, "\n", $outputlangs))."\n";
				if ( $mode != 'sourcewithnodetails')
				{
					if (empty($conf->global->MAIN_PDF_DISABLESOURCEDETAILS))
					{
						// Phone
						if ($sourcecompany->phone)				$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("PhoneShort").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($sourcecompany->phone)));
						// Fax
						if ($sourcecompany->fax)				$stringaddress	.= ($stringaddress ? ($sourcecompany->phone ? " - " : "\n") : '' ).$outputlangs->transnoentities("Fax").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($sourcecompany->fax)));
						// EMail
						if ($sourcecompany->email)				$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Email").": ".$outputlangs->convToOutputCharset($sourcecompany->email);
						// Web
						if ($sourcecompany->url)				$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Web").": ".$outputlangs->convToOutputCharset($sourcecompany->url);
					}	// if (empty($conf->global->MAIN_PDF_DISABLESOURCEDETAILS))
				}	// if ( $mode != 'sourcewithnodetails')
			}	// if ($mode == 'source')
			if ($mode == 'target' || preg_match('/targetwithdetails/',$mode))
			{
				if ($usecontact)
				{
					$stringaddress							.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset($targetcontact->getFullName($outputlangs, 1, -1));
					if (!empty($targetcontact->address))	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($targetcontact));
					else									$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($targetcompany));
					// Country
					if (!empty($targetcontact->country_code) && $targetcontact->country_code != $sourcecompany->country_code)
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset($outputlangs->transnoentitiesnoconv("Country".$targetcontact->country_code));
					else if (empty($targetcontact->country_code) && !empty($targetcompany->country_code) && ($targetcompany->country_code != $sourcecompany->country_code))
						$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset($outputlangs->transnoentitiesnoconv("Country".$targetcompany->country_code));
					if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
					{
						// Phone
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_phone/',$mode))
						{
							if (! empty($targetcontact->phone_pro) || ! empty($targetcontact->phone_mobile))	$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Phone").": ";
							if (! empty($targetcontact->phone_pro))												$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcontact->phone_pro)));
							if (! empty($targetcontact->phone_pro) && ! empty($targetcontact->phone_mobile))	$stringaddress	.= " / ";
							if (! empty($targetcontact->phone_mobile))											$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcontact->phone_mobile)));
						}	//  if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_phone/',$mode))
						// Fax
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_fax/',$mode))
							if ($targetcontact->fax)		$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Fax").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcontact->fax)));
						// EMail
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_email/',$mode))
							if ($targetcontact->email)		$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Email").": ".$outputlangs->convToOutputCharset($targetcontact->email);
						// Web
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_url/',$mode))
							if ($targetcontact->url)		$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Web").": ".$outputlangs->convToOutputCharset($targetcontact->url);
					}	// if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
				}	// if ($usecontact)
				else
				{
					$stringaddress							.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset(dol_format_address($targetcompany));
					// Country
					if (!empty($targetcompany->country_code) && $targetcompany->country_code != $sourcecompany->country_code)
						$stringaddress						.= ($stringaddress ? "\n" : '' ).$outputlangs->convToOutputCharset($outputlangs->transnoentitiesnoconv("Country".$targetcompany->country_code));
					if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
					{
						// Phone
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_phone/',$mode))
						{
							if (! empty($targetcompany->phone) || ! empty($targetcompany->phone_mobile))		$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Phone").": ";
							if (! empty($targetcompany->phone))													$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcompany->phone)));
							if (! empty($targetcompany->phone) && ! empty($targetcompany->phone_mobile))		$stringaddress	.= " / ";
							if (! empty($targetcompany->phone_mobile))											$stringaddress	.= $outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcompany->phone_mobile)));
						}
						// Fax
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_fax/',$mode))
							if ($targetcompany->fax)		$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Fax").": ".$outputlangs->convToOutputCharset(dol_string_nohtmltag(dol_print_phone($targetcompany->fax)));
						// EMail
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_email/',$mode))
							if ($targetcompany->email)		$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Email").": ".$outputlangs->convToOutputCharset($targetcompany->email);
						// Web
						if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || $mode == 'targetwithdetails' || preg_match('/targetwithdetails_url/',$mode))
							if ($targetcompany->url)		$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("Web").": ".$outputlangs->convToOutputCharset($targetcompany->url);
					}	// if (! empty($conf->global->MAIN_PDF_ADDALSOTARGETDETAILS) || preg_match('/targetwithdetails/',$mode))
				}	// else	// if ($usecontact)

				// Intra VAT
				if (empty($conf->global->MAIN_TVAINTRA_NOT_IN_ADDRESS))
					if ($targetcompany->tva_intra)			$stringaddress	.= ($stringaddress ? "\n" : '' ).$outputlangs->transnoentities("VATIntraShort").': '.$outputlangs->convToOutputCharset($targetcompany->tva_intra);
				// Professionnal Ids
				if ($profids)
				{
					if (! empty($conf->global->MAIN_PROFID1_IN_ADDRESS) && ! empty($targetcompany->idprof1))
					{
						$tmp									= $outputlangs->transcountrynoentities("ProfId1",$targetcompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress							.= ($stringaddress ? "\n" : '' ).$tmp.': '.$outputlangs->convToOutputCharset($targetcompany->idprof1);
					}	// if (! empty($conf->global->MAIN_PROFID1_IN_ADDRESS) && ! empty($targetcompany->idprof1))
					if (! empty($conf->global->MAIN_PROFID2_IN_ADDRESS) && ! empty($targetcompany->idprof2))
					{
						$tmp									= $outputlangs->transcountrynoentities("ProfId2",$targetcompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress							.= ($stringaddress ? "\n" : '' ).$tmp.': '.$outputlangs->convToOutputCharset($targetcompany->idprof2);
					}	// if (! empty($conf->global->MAIN_PROFID2_IN_ADDRESS) && ! empty($targetcompany->idprof2))
					if (! empty($conf->global->MAIN_PROFID3_IN_ADDRESS) && ! empty($targetcompany->idprof3))
					{
						$tmp									= $outputlangs->transcountrynoentities("ProfId3",$targetcompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress							.= ($stringaddress ? "\n" : '' ).$tmp.': '.$outputlangs->convToOutputCharset($targetcompany->idprof3);
					}	// if (! empty($conf->global->MAIN_PROFID3_IN_ADDRESS) && ! empty($targetcompany->idprof3))
					if (! empty($conf->global->MAIN_PROFID4_IN_ADDRESS) && ! empty($targetcompany->idprof4))
					{
						$tmp									= $outputlangs->transcountrynoentities("ProfId4",$targetcompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress							.= ($stringaddress ? "\n" : '' ).$tmp.': '.$outputlangs->convToOutputCharset($targetcompany->idprof4);
					}	// if (! empty($conf->global->MAIN_PROFID4_IN_ADDRESS) && ! empty($targetcompany->idprof4))
					if (! empty($conf->global->MAIN_PROFID5_IN_ADDRESS) && ! empty($targetcompany->idprof5))
					{
						$tmp									= $outputlangs->transcountrynoentities("ProfId5",$targetcompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress							.= ($stringaddress ? "\n" : '' ).$tmp.': '.$outputlangs->convToOutputCharset($targetcompany->idprof5);
					}	// if (! empty($conf->global->MAIN_PROFID5_IN_ADDRESS) && ! empty($targetcompany->idprof5))
					if (! empty($conf->global->MAIN_PROFID6_IN_ADDRESS) && ! empty($targetcompany->idprof6))
					{
						$tmp									= $outputlangs->transcountrynoentities("ProfId6",$targetcompany->country_code);
						if (preg_match('/\((.+)\)/',$tmp,$reg))	$tmp			= $reg[1];
						$stringaddress							.= ($stringaddress ? "\n" : '' ).$tmp.': '.$outputlangs->convToOutputCharset($targetcompany->idprof6);
					}	// if (! empty($conf->global->MAIN_PROFID6_IN_ADDRESS) && ! empty($targetcompany->idprof6))
				}	// if ($profids)
				// Public note
				if (! empty($conf->global->MAIN_PUBLIC_NOTE_IN_ADDRESS))
				{
					if ($mode == 'source' && ! empty($sourcecompany->note_public))
						$stringaddress						.= ($stringaddress ? "\n" : '' ).dol_string_nohtmltag($sourcecompany->note_public);
					if (($mode == 'target' || preg_match('/targetwithdetails/',$mode)) && ! empty($targetcompany->note_public))
						$stringaddress						.= ($stringaddress ? "\n" : '' ).dol_string_nohtmltag($targetcompany->note_public);
				}	// if (! empty($conf->global->MAIN_PUBLIC_NOTE_IN_ADDRESS))
			}	// if ($mode == 'target' || preg_match('/targetwithdetails/',$mode))
		}	// if (empty($reshook))
		return $stringaddress;
	}	// function pdf_InfraS_build_address($outputlangs, $sourcecompany, $targetcompany = '', $targetcontact = '', $usecontact = 0, $mode = 'source', $object = null, $profids = 0)

	/************************************************
	*	Convert RGBA color to RGB value
	*
	*	@param	string		$bgcolor     		RGB value for background color
	*	@param	string		$color	     		RGB color to convert
	*	@param	float		$alpha	     		alpha value => 0.0 to 1
	*	@return	string							new RGB color
	************************************************/
	function pdf_InfraS_rgba_to_rgb(&$color, $bgcolor = '255, 255, 255', $alpha = 1)
	{
		global $conf;
		
		$tmpcol				= explode(',', $color);
		$tmpcol[0]			= (! empty($tmpcol[0]) ? $tmpcol[0] : 0);
		$tmpcol[1]			= (! empty($tmpcol[1]) ? $tmpcol[1] : 0);
		$tmpcol[2]			= (! empty($tmpcol[2]) ? $tmpcol[2] : 0);
		$tmpbg				= explode(',', $bgcolor);
		$tmpbg[0]			= (! empty($tmpbg[0]) ? $tmpbg[0] : 0);
		$tmpbg[1]			= (! empty($tmpbg[1]) ? $tmpbg[1] : 0);
		$tmpbg[2]			= (! empty($tmpbg[2]) ? $tmpbg[2] : 0);
		$alpha				= (! empty($alpha) && 0 < $alpha && $alpha < 1 ? $alpha : 1);
		$tmpvalr			= ((1 - $alpha) * $tmpbg[0]) + ($alpha * $tmpcol[0]);
		$tmpvalg			= ((1 - $alpha) * $tmpbg[1]) + ($alpha * $tmpcol[1]);
		$tmpvalb			= ((1 - $alpha) * $tmpbg[2]) + ($alpha * $tmpcol[2]);
		$tmpval				= $tmpvalr.', '.$tmpvalg.', '.$tmpvalb;
		return $tmpval;
	}	// function pdf_InfraS_rgba_to_rgb(&$color, $bgcolor = '255, 255, 255', $alpha = 1)

	/************************************************
	*	Select text color from background values
	*
	*	@param	string		$bgcolor     		RGB value for background color
	*	@return	int								'255' or '0' for white (255, 255 ,255) or black (0, 0, 0)
	************************************************/
	function pdf_InfraS_txt_color(&$bgcolor)
	{
		require_once DOL_DOCUMENT_ROOT.'/core/lib/functions2.lib.php';
		global $conf;
		
		$tmppart			= explode(',', $bgcolor);
		$tmpvalr			= (! empty($tmppart[0]) ? $tmppart[0] : 0) * 0.3;
		$tmpvalg			= (! empty($tmppart[1]) ? $tmppart[1] : 0) * 0.59;
		$tmpvalb			= (! empty($tmppart[2]) ? $tmppart[2] : 0) * 0.11;
		$tmpval				= $tmpvalr + $tmpvalg + $tmpvalb;
		if ($tmpval <= 128)	$colorauto	= '255, 255, 255';
		else				$colorauto	= '0, 0, 0';
		$textcolorauto		= isset($conf->global->INFRAS_PDF_TEXT_COLOR_AUTO)	? $conf->global->INFRAS_PDF_TEXT_COLOR_AUTO : 1;
		$colorman			= isset($conf->global->INFRAS_PDF_TEXT_COLOR)		? $conf->global->INFRAS_PDF_TEXT_COLOR : $colorauto;
		if ($textcolorauto)	return $colorauto;
		else				return $colorman;
	}	// function pdf_InfraS_txt_color(&$bgcolor)

	/************************************************
	*	Return line ref_supplier
	*
	*	@param	Object		$object				Object
	*	@param	int			$i					Current line number
	*	@param  Translate	$outputlangs		Object langs for output
	*	@param	int			$hidedetails		Hide details (0 = no, 1 = yes, 2 = just special lines)
	*	@return	string
	************************************************/
	function pdf_InfraS_getlineref_supplier($object, $i, $outputlangs, $hidedetails = 0)
	{
		global $db, $hookmanager;

		if (empty($object->lines[$i]->fk_product))	$ref_supplier	= 'none';
		else										$ref_supplier	= (! empty($object->lines[$i]->ref_supplier) ? $object->lines[$i]->ref_supplier : (! empty($object->lines[$i]->ref_fourn) ? $object->lines[$i]->ref_fourn : ''));
		if (empty($ref_supplier))
		{
			$ref									= $object->lines[$i]->ref;
			$sql									= 'SELECT pfp.ref_fourn ';
			$sql									.= 'FROM '.MAIN_DB_PREFIX.'product AS p ';
			$sql									.= 'LEFT JOIN '.MAIN_DB_PREFIX.'product_fournisseur_price AS pfp ON p.rowid = pfp.fk_product ';
			$sql									.= 'WHERE p.ref = "'.$ref.'" AND pfp.fk_soc = "'.$object->thirdparty->id.'"';
			$resql									= $object->db->query($sql);
			if ($resql)
			{
				$obj								= $db->fetch_object($resql);
				if ($obj)							$ref_supplier	= $obj->ref_fourn;
			}	// if ($resql)
			$db->free($resql);
		}	// if (empty($ref_supplier))
		if ($ref_supplier == 'none')				$ref_supplier	= '';
		$reshook									= 0;
		$result										= '';
		if (is_object($hookmanager))
		{
			$special_code							= $object->lines[$i]->special_code;
			if (! empty($object->lines[$i]->fk_parent_line))
				$special_code						= $object->getSpecialCode($object->lines[$i]->fk_parent_line);
			$parameters								= array('i'=>$i, 'outputlangs'=>$outputlangs, 'hidedetails'=>$hidedetails, 'special_code'=>$special_code);
			$action									= '';
			$reshook								= $hookmanager->executeHooks('pdf_getlineref_supplier', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
			$result									.= $hookmanager->resPrint;
		}	// if (is_object($hookmanager))
		if (empty($reshook))						$result			.= dol_htmlentitiesbr($ref_supplier);
		return $result;
	}	// function pdf_InfraS_getlineref_supplier($object, $i, $outputlangs, $hidedetails = 0)

	/************************************************
	*	Return array with format properties
	*
	*	@param	Translate	$outputlangs	Output lang to use to autodetect output format if setup not done
	*	@param	string		$format			specific format to use
	*	@return	array						Array('width'=>w,'height'=>h,'unit'=>u);
	************************************************/
	function pdf_InfraS_getFormat($format = '', Translate $outputlangs = null)
	{
		global $conf, $db;

		// Default value if setup was not done and/or entry into c_paper_format not defined
		$width				= 210;
		$height				= 297;
		$unit				= 'mm';
		if ($format)
		{
			$sql			= "SELECT code, label, width, height, unit FROM ".MAIN_DB_PREFIX."c_paper_format";
			$sql			.= " WHERE code = '".$format."'";
			$resql			= $db->query($sql);
			if ($resql)
			{
				$obj		= $db->fetch_object($resql);
				if ($obj)
				{
					$width	= (int) $obj->width;
					$height	= (int) $obj->height;
					$unit	= $obj->unit;
					return array('width'=>$width, 'height'=>$height, 'unit'=>$unit);
				}	// if ($obj)
			}	// if ($resql)
			$db->free($resql);
		}	// if ($format)
		if (empty($conf->global->MAIN_PDF_FORMAT))
		{
			include_once DOL_DOCUMENT_ROOT.'/core/lib/functions2.lib.php';
			$pdfformat		= dol_getDefaultFormat($outputlangs);
		}
		else				$pdfformat	= $conf->global->MAIN_PDF_FORMAT;
		$sql				= "SELECT code, label, width, height, unit FROM ".MAIN_DB_PREFIX."c_paper_format";
		$sql				.= " WHERE code = '".$pdfformat."'";
		$resql				= $db->query($sql);
		if ($resql)
		{
			$obj			= $db->fetch_object($resql);
			if ($obj)
			{
				$width		= (int) $obj->width;
				$height		= (int) $obj->height;
				$unit		= $obj->unit;
			}	// if ($obj)
		}	// if ($resql)
		$db->free($resql);
		return array('width'=>$width, 'height'=>$height, 'unit'=>$unit);
	}	// function pdf_InfraS_getFormat($format = '', Translate $outputlangs = null)
?>