<?php
	/************************************************
	* Copyright (C) 2016	Sylvain Legrand - <contact@infras.fr>	InfraS - <http://www.infras.fr>
	*
	* This program is free software: you can redistribute it and/or modify
	* it under the terms of the GNU General Public License as published by
	* the Free Software Foundation, either version 3 of the License, or
	* (at your option) any later version.
	*
	* This program is distributed in the hope that it will be useful,
	* but WITHOUT ANY WARRANTY; without even the implied warranty of
	* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	* GNU General Public License for more details.
	*
	* You should have received a copy of the GNU General Public License
	* along with this program.  If not, see <http://www.gnu.org/licenses/>.
	************************************************/

	/************************************************
	* 	\file		htdocs/custom/infras/core/modules/expedition/doc/pdf_InfraS_expedition.modules.php
	* 	\ingroup	InfraS
	* 	\brief		Class file for InfraS PDF expedition
	************************************************/

	// Libraries ************************************
	require_once DOL_DOCUMENT_ROOT.'/core/modules/expedition/modules_expedition.php';
	require_once DOL_DOCUMENT_ROOT.'/contact/class/contact.class.php';
	require_once DOL_DOCUMENT_ROOT.'/core/lib/company.lib.php';
	require_once DOL_DOCUMENT_ROOT.'/core/lib/pdf.lib.php';
	dol_include_once('/infraspack/core/lib/infras.pdf.lib.php');

	/************************************************
	*	Class to generate PDF proposal InfraS
	************************************************/
	class pdf_InfraS_expedition extends ModelePdfExpedition
	{
		public $db;
		public $name;
		public $description;
		public $type;
		public $phpmin	= array(5, 4);
		public $version	= 'dolibarr';
		public $page_largeur;
		public $page_hauteur;
		public $format;
		public $marge_gauche;
		public $marge_droite;
		public $marge_haute;
		public $marge_basse;
		public $emetteur;

		/********************************************
		*	Constructor
		*
		*	@param		DoliDB		$db      Database handler
		********************************************/
		function __construct($db)
		{
			global $conf, $langs, $mysoc;

			if((float)DOL_VERSION >= 8.0)	$langs->loadLangs(array("main", "dict", "bills", "products", "deliveries", "sendings", "productbatch", "companies", "propal", "infraspack@infraspack"));
			else
			{
				$langs->load("main");
				$langs->load("dict");
				$langs->load("bills");
				$langs->load("products");
				$langs->load("deliveries");
				$langs->load("sendings");
				$langs->load("productbatch");
				$langs->load("companies");
				$langs->load("propal");
				$langs->load("infraspack@infraspack");
			}	// else	// if((float)DOL_VERSION >= 8.0)
			
			$this->db									= $db;
			$this->name									= $langs->trans('PDFInfraSExpeditionName');
			$this->description							= $langs->trans('PDFInfraSExpeditionDescription');
			$this->type									= 'pdf';
			$formatarray								= pdf_InfraS_getFormat();
			$this->page_largeur							= $formatarray['width'];
			$this->page_hauteur							= $formatarray['height'];
			$this->format								= array($this->page_largeur, $this->page_hauteur);
			$this->marge_gauche							= isset($conf->global->MAIN_PDF_MARGIN_LEFT)					? $conf->global->MAIN_PDF_MARGIN_LEFT : 10;
			$this->marge_droite							= isset($conf->global->MAIN_PDF_MARGIN_RIGHT)					? $conf->global->MAIN_PDF_MARGIN_RIGHT : 10;
			$this->marge_haute							= isset($conf->global->MAIN_PDF_MARGIN_TOP)						? $conf->global->MAIN_PDF_MARGIN_TOP : 10;
			$this->marge_basse							= isset($conf->global->MAIN_PDF_MARGIN_BOTTOM)					? $conf->global->MAIN_PDF_MARGIN_BOTTOM : 10;
			$this->font									= isset($conf->global->INFRAS_PDF_FONT)							? $conf->global->INFRAS_PDF_FONT : 'Helvetica';
			$this->use_fpdf								= isset($conf->global->MAIN_USE_FPDF)							? $conf->global->MAIN_USE_FPDF : 0;
			$this->disable_fpdi							= isset($conf->global->MAIN_DISABLE_FPDI)						? $conf->global->MAIN_DISABLE_FPDI : 0;
			$this->add_pdf_bg							= isset($conf->global->MAIN_ADD_PDF_BACKGROUND)					? $conf->global->MAIN_ADD_PDF_BACKGROUND : 0;
			$this->Rounded_rect							= isset($conf->global->INFRAS_PDF_ROUNDED_REC)					? $conf->global->INFRAS_PDF_ROUNDED_REC : 0;
			$this->with_picture							= isset($conf->global->INFRAS_PDF_WITH_PICTURE)					? $conf->global->INFRAS_PDF_WITH_PICTURE : 0;
			$this->old_path_photo						= isset($conf->global->PRODUCT_USE_OLD_PATH_FOR_PHOTO)			? $conf->global->PRODUCT_USE_OLD_PATH_FOR_PHOTO : 0;
			$this->cat_hq_image							= isset($conf->global->CAT_HIGH_QUALITY_IMAGES)					? $conf->global->CAT_HIGH_QUALITY_IMAGES : 0;
			$this->product_use_unit						= isset($conf->global->PRODUCT_USE_UNITS)						? $conf->global->PRODUCT_USE_UNITS : 0;
			$this->show_ref_col							= isset($conf->global->INFRAS_PDF_WITH_REF_COLUMN)				? $conf->global->INFRAS_PDF_WITH_REF_COLUMN : 0;
			$this->small_head2							= isset($conf->global->INFRAS_PDF_SMALL_HEAD_2)					? $conf->global->INFRAS_PDF_SMALL_HEAD_2 : 0;
			$this->emetteur								= $mysoc;
			if (empty($this->emetteur->country_code))	$this->emetteur->country_code									= substr($langs->defaultlang, -2);
			$this->formatpage							= array('largeur'=>$this->page_largeur, 'hauteur'=>$this->page_hauteur, 'mgauche'=>$this->marge_gauche, 
																'mdroite'=>$this->marge_droite, 'mhaute'=>$this->marge_haute, 'mbasse'=>$this->marge_basse);
			$this->wpicture								= isset($conf->global->INFRAS_PDF_PICTURE_WIDTH)				? $conf->global->INFRAS_PDF_PICTURE_WIDTH : 20;
			$this->hpicture								= isset($conf->global->INFRAS_PDF_PICTURE_HEIGHT)				? $conf->global->INFRAS_PDF_PICTURE_HEIGHT : 32;
			$this->dash_between_line					= isset($conf->global->MAIN_PDF_DASH_BETWEEN_LINES)				? $conf->global->MAIN_PDF_DASH_BETWEEN_LINES : 1;
			$this->hide_top_table						= isset($conf->global->INFRAS_PDF_HIDE_TOP_TABLE)				? $conf->global->INFRAS_PDF_HIDE_TOP_TABLE : 0;
			$this->multilangs							= isset($conf->global->MAIN_MULTILANGS)							? $conf->global->MAIN_MULTILANGS : 0;
			$this->main_umask							= isset($conf->global->MAIN_UMASK)								? $conf->global->MAIN_UMASK : '0755';
			$this->use_comp_name_contact				= isset($conf->global->MAIN_USE_COMPANY_NAME_OF_CONTACT)		? $conf->global->MAIN_USE_COMPANY_NAME_OF_CONTACT : 0;
			$this->includealias							= isset($conf->global->PDF_INCLUDE_ALIAS_IN_THIRDPARTY_NAME)	? $conf->global->PDF_INCLUDE_ALIAS_IN_THIRDPARTY_NAME : 0;
			$this->use_iso_location						= isset($conf->global->MAIN_PDF_USE_ISO_LOCATION)				? $conf->global->MAIN_PDF_USE_ISO_LOCATION : 0;
			$this->show_recep_details					= isset($conf->global->INFRAS_PDF_SHOW_RECEP_DETAILS)			? $conf->global->INFRAS_PDF_SHOW_RECEP_DETAILS : 0;
			$this->invert_sender_recipient				= isset($conf->global->MAIN_INVERT_SENDER_RECIPIENT)			? $conf->global->MAIN_INVERT_SENDER_RECIPIENT : 0;
			$this->hide_recep_frame						= isset($conf->global->INFRAS_PDF_HIDE_RECEP_FRAME)				? $conf->global->INFRAS_PDF_HIDE_RECEP_FRAME : 0;
			$this->bg_color								= isset($conf->global->INFRAS_PDF_BACKGROUND_COLOR)				? $conf->global->INFRAS_PDF_BACKGROUND_COLOR : '';
			$this->title_bg								= isset($conf->global->INFRAS_PDF_TITLE_BG)						? $conf->global->INFRAS_PDF_TITLE_BG : 0;
			$this->watermark_i_opacity					= isset($conf->global->INFRAS_PDF_I_WATERMARK_OPACITY)			? $conf->global->INFRAS_PDF_I_WATERMARK_OPACITY : 1;
			$this->image_watermark						= isset($conf->global->INFRAS_PDF_IMAGE_WATERMARK)				? $conf->global->INFRAS_PDF_IMAGE_WATERMARK : '';
			$this->watermark_t_opacity					= isset($conf->global->INFRAS_PDF_T_WATERMARK_OPACITY)			? $conf->global->INFRAS_PDF_T_WATERMARK_OPACITY : 10;
			$this->draft_watermark						= isset($conf->global->PROPALE_DRAFT_WATERMARK)					? $conf->global->PROPALE_DRAFT_WATERMARK : '';
			$this->fold_mark							= isset($conf->global->INFRAS_PDF_FOLD_MARK)					? $conf->global->INFRAS_PDF_FOLD_MARK : 0;
			$this->txtcolor								= explode(',', pdf_InfraS_txt_color($this->bg_color));
			$this->alpha								= 0.2;
			$this->hide_weight_vol						= isset($conf->global->SHIPPING_PDF_HIDE_WEIGHT_AND_VOLUME)		? $conf->global->SHIPPING_PDF_HIDE_WEIGHT_AND_VOLUME : 0;
			$this->hide_pay_term_cond					= isset($conf->global->PROPALE_PDF_HIDE_PAYMENTTERMCOND)		? $conf->global->PROPALE_PDF_HIDE_PAYMENTTERMCOND : 0;
			$this->chq_num								= isset($conf->global->FACTURE_CHQ_NUMBER)						? $conf->global->FACTURE_CHQ_NUMBER : 0;
			$this->diffsize_title						= isset($conf->global->PDF_DIFFSIZE_TITLE)						? $conf->global->PDF_DIFFSIZE_TITLE : 0;
			$this->hidechq_address						= isset($conf->global->MAIN_PDF_HIDE_CHQ_ADDRESS)				? $conf->global->MAIN_PDF_HIDE_CHQ_ADDRESS : 0;
			$this->rib_num								= isset($conf->global->FACTURE_RIB_NUMBER)						? $conf->global->FACTURE_RIB_NUMBER : 0;
			$this->ht_signarea							= isset($conf->global->INFRAS_PDF_HT_SIGN_AREA)					? $conf->global->INFRAS_PDF_HT_SIGN_AREA : 24;
			$this->type_foot							= isset($conf->global->INFRAS_PDF_TYPE_FOOT)					? $conf->global->INFRAS_PDF_TYPE_FOOT : 0;
			$this->wpicturefoot							= isset($conf->global->INFRAS_PDF_PICTURE_FOOT_WIDTH)			? $conf->global->INFRAS_PDF_PICTURE_FOOT_WIDTH : 188;
			$this->hpicturefoot							= isset($conf->global->INFRAS_PDF_PICTURE_FOOT_HEIGHT)			? $conf->global->INFRAS_PDF_PICTURE_FOOT_HEIGHT : 12;
			$this->maxsizeimgfoot						= array('largeur'=>$this->wpicturefoot, 'hauteur'=>$this->hpicturefoot);
			$this->hidepagenum							= isset($conf->global->INFRAS_PDF_HIDE_PAGE_NUM)				? $conf->global->INFRAS_PDF_HIDE_PAGE_NUM : 0;
			$this->hide_ordered							= isset($conf->global->INFRAS_PDF_HIDE_ORDERED)					? $conf->global->INFRAS_PDF_HIDE_ORDERED : 0;
			$this->showadrlivr							= isset($conf->global->INFRAS_PDF_SHOW_ADRESSE_RECEPTION)		? $conf->global->INFRAS_PDF_SHOW_ADRESSE_RECEPTION : 0;
			$this->option_logo							= 1;	// Affiche logo
			$this->option_freetext						= 1;	// Support add of a personalised text
			$this->option_draft_watermark				= 1;	// Support add of a watermark on drafts
		}	// function __construct($db)

		/********************************************
		*	Function to build pdf onto disk
		*
		*	@param		Object		$object				Object to generate
		*	@param		Translate	$outputlangs		Lang output object
		*	@param		string		$srctemplatepath	Full path of source filename for generator using a template file
		*	@param		int			$hidedetails		Do not show line details (inutilisée ! laissé pour la compatibilité)
		*	@param		int			$hidedesc			Do not show desc
		*	@param		int			$hideref			Do not show ref
		*	@return     int             				1=OK, 0=KO
		********************************************/
		function write_file($object, $outputlangs, $srctemplatepath = '', $hidedetails = 0, $hidedesc = 0, $hideref = 0)
		{
			global $user, $langs, $conf, $db, $hookmanager;

			if (! is_object($outputlangs)) $outputlangs	= $langs;
			// For backward compatibility with FPDF, force output charset to ISO, because FPDF expect text to be encoded in ISO
			if (! empty($this->use_fpdf)) $outputlangs->charset_output	= 'ISO-8859-1';
			
			if((float)DOL_VERSION >= 8.0)	$outputlangs->loadLangs(array("main", "dict", "bills", "products", "deliveries", "sendings", "productbatch", "companies", "propal", "infraspack@infraspack"));
			else
			{
				$outputlangs->load("main");
				$outputlangs->load("dict");
				$outputlangs->load("bills");
				$outputlangs->load("products");
				$outputlangs->load("deliveries");
				$outputlangs->load("sendings");
				$outputlangs->load("productbatch");
				$outputlangs->load("companies");
				$outputlangs->load("propal");
				$outputlangs->load("infraspack@infraspack");
			}	// else	// if((float)DOL_VERSION >= 8.0)

			if ($conf->expedition->dir_output)
			{
				// Definition of $dir and $file
				if ($object->specimen)
				{
					$dir								= $conf->expedition->dir_output."/sending";
					$file								= $dir."/SPECIMEN.pdf";
				}	// if ($object->specimen)
				else
				{
					$objectref							= dol_sanitizeFileName($object->ref);
					$dir								= $conf->expedition->dir_output."/sending/".$objectref;
					$file								= $dir."/".$objectref.".pdf";
				}	// else	// if ($object->specimen)
				if (! file_exists($dir))
				{
					if (dol_mkdir($dir) < 0)
					{
						$this->error=$langs->transnoentities("ErrorCanNotCreateDir", $dir);
						return 0;
					}	// if (dol_mkdir($dir) < 0)
				}	// if (! file_exists($dir))
				if (file_exists($dir))
				{
					if (! is_object($hookmanager))	// Add pdfgeneration hook
					{
						include_once DOL_DOCUMENT_ROOT.'/core/class/hookmanager.class.php';
						$hookmanager					= new HookManager($this->db);
					}	// if (! is_object($hookmanager))
					$hookmanager->initHooks(array('pdfgeneration'));
					$parameters							= array('file'=>$file, 'object'=>$object, 'outputlangs'=>$outputlangs);
					global $action;
					$reshook							= $hookmanager->executeHooks('beforePDFCreation', $parameters, $object, $action);    // Note that $action and $object may have been modified by some hooks
					$this->adrlivr						= $hookmanager->resArray['adrlivr'];
					$this->pied							= $hookmanager->resArray['pied'];
					$nblignes							= count($object->lines);	// Set nblignes with the new facture lines content after hook
					if (! empty($this->show_ref_col))	$hideref = 1;	// Comme on affiche une colonne 'Référence' on s'assure de ne pas répéter l'information
					// Create pdf instance
					$pdf								= pdf_getInstance($this->format);
					$default_font_size					= pdf_getPDFFontSize($outputlangs);																								// Must be after pdf_getInstance
					$pdf->SetAutoPageBreak(1, 0);
					if (class_exists('TCPDF'))
					{
						$pdf->setPrintHeader(false);
						$pdf->setPrintFooter(false);
					}	// if (class_exists('TCPDF'))
					$pdf->SetFont($this->font);
					// Set path to the background PDF File
					if ((empty($this->disable_fpdi) || (float)DOL_VERSION >= 8.0) && ! empty($this->add_pdf_bg))
					{
						$pagecount	= $pdf->setSourceFile($conf->mycompany->dir_output.'/'.$this->add_pdf_bg);
						$tplidx		= $pdf->importPage(1);
					}	// if ((empty($this->disable_fpdi) || (float)DOL_VERSION >= 8.0) && ! empty($this->add_pdf_bg))
					$pdf->Open();
					$pdf->SetTitle($outputlangs->convToOutputCharset($object->ref));
					$pdf->SetSubject($outputlangs->transnoentities("Shipment"));
					$pdf->SetCreator("Dolibarr ".DOL_VERSION);
					$pdf->SetAuthor($outputlangs->convToOutputCharset($user->getFullName($outputlangs)));
					$pdf->SetKeyWords($outputlangs->convToOutputCharset($object->ref)." ".$outputlangs->transnoentities("Shipment"));
					$pdf->SetMargins($this->marge_gauche, $this->marge_haute, $this->marge_droite);   // Left, Top, Right
					// First loop on each lines to prepare calculs and variables
					$realpatharray						= array();
					$objphoto							= new Product($this->db);
					for ($i = 0 ; $i < $nblignes ; $i++)
					{
						// detect if there is at least one image to show
						if (! empty($this->with_picture))
						{
							if (empty($object->lines[$i]->fk_product)) continue;
							$objphoto->fetch($object->lines[$i]->fk_product);
							if (! empty($this->old_path_photo))
							{
								$pdir[0] = get_exdir($objphoto->id, 2, 0, 0, $objphoto, 'product').$objphoto->id ."/photos/";
								$pdir[1] = get_exdir(0, 0, 0, 0, $objphoto, 'product').dol_sanitizeFileName($objphoto->ref).'/';
							}	// if (! empty($this->old_path_photo))
							else
							{
								$pdir[0] = get_exdir(0, 0, 0, 0, $objphoto, 'product').dol_sanitizeFileName($objphoto->ref).'/';	// default
								$pdir[1] = get_exdir($objphoto->id, 2, 0, 0, $objphoto, 'product').$objphoto->id ."/photos/";		// alternative
							}	// else	// if (! empty($this->old_path_photo))
							$arephoto = false;
							foreach ($pdir as $midir)
							{
								if (! $arephoto)
								{
									$dir = $conf->product->dir_output.'/'.$midir;
									foreach ($objphoto->liste_photos($dir, 1) as $key => $obj)
									{
										if (empty($this->cat_hq_image))		// If CAT_HIGH_QUALITY_IMAGES not defined, we use thumb if defined and then original photo
										{
											if ($obj['photo_vignette'])	$filename	= $obj['photo_vignette'];
											else						$filename	= $obj['photo'];
										}	// if (empty($this->cat_hq_image))
										else		$filename	= $obj['photo'];
										$realpath	= $dir.$filename;
										$arephoto	= true;
									}	// foreach ($objphoto->liste_photos($dir,1) as $key => $obj)
								}	// if (! $arephoto)
							}	// foreach ($pdir as $midir)
							if ($realpath && $arephoto) $realpatharray[$i]	= $realpath;
						}	// if (! empty($this->with_picture))
					}	// for ($i = 0 ; $i < $nblignes ; $i++)
					// New page
					$pdf->AddPage();
					if (! empty($tplidx))			$pdf->useTemplate($tplidx);
					$pagenb							= 1;
					$pdf->SetFont('', '', $default_font_size - 1);
					$pdf->MultiCell(0, 3, '');		// Set interline to 3
					// Define width and position of notes frames
					$this->larg_util_txt			= $this->page_largeur - ($this->marge_gauche + $this->marge_droite + ($this->Rounded_rect * 2) + 2);
					$this->larg_util_cadre			= $this->page_largeur - ($this->marge_gauche + $this->marge_droite);
					$this->posx_G_txt				= $this->marge_gauche + $this->Rounded_rect + 1;
					// Define width and position of main table columns
					$this->larg_qtytoship			= 21;
					$this->larg_qtyordered			= $this->hide_ordered ? 0 : 21;
					$this->larg_weightvol			= $this->hide_weight_vol ? 0 : 30;
					$this->larg_weightvol			= 30;
					$this->larg_ref					= empty($this->show_ref_col) ? 0 : 24;
					$this->posxqtytoship			= $this->page_largeur - $this->marge_droite - $this->larg_qtytoship;
					$this->posxqtyordered			= $this->posxqtytoship - $this->larg_qtyordered;
					$this->posxweightvol			= $this->posxqtyordered - $this->larg_weightvol;
					$this->posxref					= $this->marge_gauche + 1;
					$this->posxdesc					= $this->marge_gauche + $this->larg_ref + 1;
					$this->larg_desc				= $this->posxweightvol - $this->posxdesc; // Largeur variable suivant la place restante
					// Define width and position of secondary tables columns
					$this->larg_tabtotal			= $this->larg_weightvol + $this->larg_qtyordered + $this->larg_qtytoship + 15;
					$this->larg_tabinfo				= $this->page_largeur - $this->marge_gauche - $this->marge_droite - $this->larg_tabtotal;
					$this->posxtabtotal				= $this->page_largeur - $this->marge_droite - $this->larg_tabtotal;
					// Calculs de positions
					$this->tab_hl					= 4;
					$head							= $this->_pagehead($pdf, $object, 1, $outputlangs);
					$hauteurhead					= $head["totalhead"];
					$hauteurcadre					= $head["hauteurcadre"];
					$tab_top						= $hauteurhead + 5;
					$tab_top_newpage				= (empty($this->small_head2) ? $hauteurhead - $hauteurcadre : 17);
					$this->ht_top_table				= $this->Rounded_rect * 2 > $this->tab_hl ? ($this->Rounded_rect * 2) + 1 : $this->tab_hl;
					$this->ht_decal_round			= $this->Rounded_rect > 0 ? $this->Rounded_rect - 1 : 0;
					$ht_colinfo						= $this->_tableau_info($pdf, $object, $bottomlasttab, $outputlangs, 1);
					$ht_coltotal					= $this->_tableau_tot($pdf, $object, $bottomlasttab, $outputlangs, 1);
					$ht2_coltotal					= pdf_InfraS_free_text($pdf, $object, $this->posxtabtotal, $ht_coltotal + 3, $outputlangs, $this->emetteur, 'SHIPPING_FREE_TEXT', 1);
					if ($ht2_coltotal > 4)			$ht_coltotal			+= ($this->tab_hl * 2) + $this->ht_signarea + 1;
					else							$ht_coltotal			+= ($this->tab_hl * 2) + $this->ht_signarea;
					$ht_coltotal					+= $ht2_coltotal;
					$heightforinfotot				= $ht_colinfo > $ht_coltotal ? $ht_colinfo : $ht_coltotal;
					$heightforfooter				= $this->_pagefoot($pdf, $object, $outputlangs, 1);
					$pdf->SetDrawColor(128, 128, 128);
					$pdf->SetTextColor(0, 0, 0);
					// Incoterm
					$height_incoterms				= 0;
					if ($conf->incoterm->enabled)
					{
						$desc_incoterms				= $object->getIncotermsForPDF();
						if ($desc_incoterms)
						{
							$pdf->SetFont('', '', $default_font_size - 1);
							$pdf->writeHTMLCell($this->larg_util_txt, $this->tab_hl, $this->posx_G_txt, $tab_top, dol_htmlentitiesbr($desc_incoterms), 0, 1);
							$nexY					= $pdf->GetY();
							$height_incoterms		= $this->Rounded_rect * 2 > $nexY - $tab_top ? $this->Rounded_rect * 2 : $nexY - $tab_top;
							$pdf->RoundedRect($this->marge_gauche, $tab_top - 1, $this->larg_util_cadre, $height_incoterms + 1, $this->Rounded_rect, '1111');
							$height_incoterms		+= $this->tab_hl;
						}	// if ($desc_incoterms)
					}	// if ($conf->incoterm->enabled)
					$tab_top						+= $height_incoterms;
					// Livraison
					$height_livr					= 0;
					if ($this->showadrlivr && $this->adrlivr)
					{
						$recepdetails						= $this->show_recep_details ? 'targetwithdetails' : 'target';
						if ($this->adrlivr == 'Default')	$livrshow	= pdf_InfraS_build_address($outputlangs, $this->emetteur, $object->thirdparty, '', 0, $recepdetails, $object, 0);
						else								$livrshow	= pdf_InfraS_build_address($outputlangs, $this->emetteur, $this->adrlivr, '', 0, $recepdetails, $object, 0);
						if ($livrshow)
						{
							$pdf->SetFont('', 'B', $default_font_size + 2);
							$pdf->writeHTMLCell($this->larg_util_txt, $this->tab_hl, $this->posx_G_txt, $tab_top, dol_htmlentitiesbr($outputlangs->transnoentities("PDFInfraSLivr")), 0, 1);
							$xlivr					= $pdf->GetX() + $pdf->GetStringWidth($outputlangs->transnoentities("PDFInfraSLivr"), '', 'B', $default_font_size + 2) + 5;
							if ($this->adrlivr->name != '')
							{
								$pdf->SetFont('', 'B', $default_font_size);
								$pdf->writeHTMLCell($this->larg_util_txt - $xlivr - 3, $this->tab_hl, $xlivr, $tab_top + 0.6, dol_htmlentitiesbr($this->adrlivr->name), 0, 1);
								$nexY				= $pdf->GetY();
							}	// if ($this->adrlivr->name)
							else					$nexY	= $tab_top + 0.6;
							$pdf->SetFont('', '', $default_font_size - 1);
							$pdf->writeHTMLCell($this->larg_util_txt - $xlivr - 3, $this->tab_hl, $xlivr, $nexY, dol_htmlentitiesbr($livrshow), 0, 1);
							$nexY					= $pdf->GetY();
							$height_livr			= $this->Rounded_rect * 2 > $nexY - $tab_top ? $this->Rounded_rect * 2 : $nexY - $tab_top;
							$pdf->RoundedRect($this->marge_gauche, $tab_top - 1, $this->larg_util_cadre, $height_livr + 2, $this->Rounded_rect, '1111');
							$height_livr			+= $this->tab_hl;
						}	// if ($livrshow)
					}	// if ($this->showadrlivr && $this->adrlivr)
					$tab_top						+= $height_livr;
					// Affiche notes
					$height_note					= 0;
					$notetoshow						= empty($object->note_public) ? '' : $object->note_public;
					if ($notetoshow)
					{
						$pdf->SetFont('', '', $default_font_size - 1);
						$pdf->startTransaction();
						$pdf->writeHTMLCell($this->larg_util_txt, $this->tab_hl, $this->posx_G_txt, $tab_top, dol_htmlentitiesbr($notetoshow), 0, 1);
						if ($pdf->getPage() > 1 || $pdf->GetY() > ($this->page_hauteur - $this->ht_top_table - $heightforfooter - 30)) // Notes need pagebreak or There is no space left for footer
						{
							$pdf->rollbackTransaction(true);
							$pdf->writeHTMLCell($this->larg_util_txt, $this->tab_hl, $this->posx_G_txt, $tab_top, dol_htmlentitiesbr($outputlangs->transnoentities("PDFInfrasNoteTooLong")), 0, 1);
						}	// if ($pdf->getPage() > 1 || $pdf->GetY() > ($this->page_hauteur - $heightforfooter))
						else						$pdf->commitTransaction();
						
						$nexY						= $pdf->GetY();
						$height_note				= $this->Rounded_rect * 2 > $nexY - $tab_top ? $this->Rounded_rect * 2 : $nexY - $tab_top;
						$pdf->RoundedRect($this->marge_gauche, $tab_top - 1, $this->larg_util_cadre, $height_note + 2, $this->Rounded_rect, '1111');
						$height_note				+= $this->tab_hl;
					}	// if ($notetoshow)
					$tab_top						+= $height_note;
					$nexY							= $tab_top + $this->ht_top_table + $this->ht_decal_round;
					// Loop on each lines
					for ($i = 0 ; $i < $nblignes ; $i++)
					{
						$curY											= $nexY;
						$pdf->SetFont('', '', $default_font_size - 1);   // Into loop to work with multipage
						$pdf->SetTextColor(0, 0, 0);
						// Define size of image if we need it
						if ($this->larg_desc - 2 < $this->wpicture)	$this->wpicture	= $this->larg_desc - 2;	// corrige la largeur maximal de l'image pour être au plus égale à la largeur colonne
						$imglinesize									= array();
						if (! empty($realpatharray[$i]))				$imglinesize	= pdf_InfraS_getSizeForImage($realpatharray[$i], $this->wpicture, $this->hpicture);
						if (empty($this->hide_top_table))				$pdf->setTopMargin($tab_top_newpage + $this->ht_top_table + $this->ht_decal_round);
						else											$pdf->setTopMargin($tab_top_newpage);
						$pdf->setPageOrientation('', 1, $heightforfooter);	// The only function to edit the bottom margin of current page to set it.
						$pageposbefore									= $pdf->getPage();
						$showpricebeforepagebreak						= 1;
						// We start with Photo of product line
						if (isset($imglinesize['width']) && isset($imglinesize['height']))
						{
							if (($curY + $imglinesize['height']) > ($this->page_hauteur - ($heightforfooter)))	// If photo too high, we moved completely on new page
							{
								$pdf->AddPage('', '', true);
								if (! empty($tplidx))			$pdf->useTemplate($tplidx);
								$pdf->setPage($pageposbefore + 1);
								$curY							= $tab_top_newpage + ($this->hide_top_table ? $this->ht_decal_round : $this->ht_top_table + $this->ht_decal_round);
								$showpricebeforepagebreak		= 1;
							}	// if (($curY + $imglinesize['height']) > ($this->page_hauteur - ($heightforfooter + $heightforinfotot)))
							$posxpicture			= ($this->posxdesc - 1) + (($this->larg_desc - 2 - $imglinesize['width']) / 2);	// centre l'image dans la colonne
							$pdf->Image($realpatharray[$i], $posxpicture, $curY, $imglinesize['width'], $imglinesize['height'], '', '', '', 2, 300);	// Use 300 dpi
							$curY					+= $imglinesize['height'];	// $pdf->Image does not increase value return by getY, so we save it manually
						}	// if (isset($imglinesize['width']) && isset($imglinesize['height']))
						// Description of product line
						$pdf->startTransaction();
						pdf_InfraS_writelinedesc($pdf, $object, $i, $outputlangs, $this->larg_desc, $this->tab_hl, $this->posxdesc, $curY, $hideref, $hidedesc);
						$pageposafter				= $pdf->getPage();
						$pageposdesc				= $pdf->getPage();
						$posyafter					= $pdf->GetY();
						if ($pageposafter > $pageposbefore)	// There is a pagebreak
						{
							$pdf->rollbackTransaction(true);
							$pageposafter	= $pageposbefore;
							$pdf->setPageOrientation('', 1, $heightforfooter);	// The only function to edit the bottom margin of current page to set it.
							$pageposdesc	= $pdf->getPage();
							pdf_InfraS_writelinedesc($pdf, $object, $i, $outputlangs, $this->larg_desc, $this->tab_hl, $this->posxdesc, $curY, $hideref, $hidedesc);
							$pageposafter	= $pdf->getPage();
							$posyafter		= $pdf->GetY();
							if ($posyafter > ($this->page_hauteur - ($heightforfooter + $heightforinfotot)))	// There is no space left for total+free text
							{
								if ($i == ($nblignes - 1))	// No more lines, and no space left to show total, so we create a new page
								{
									$pdf->AddPage('', '', true);
									if (! empty($tplidx))			$pdf->useTemplate($tplidx);
									$pdf->setPage($pageposafter + 1);
								}	// if ($i == ($nblignes - 1))
							}	// if ($posyafter > ($this->page_hauteur - ($heightforfooter + $heightforinfotot)))
							else	$showpricebeforepagebreak	= 0;
						}	// if ($pageposafter > $pageposbefore)
						elseif ($posyafter > ($this->page_hauteur - ($heightforfooter + $heightforinfotot)))
						{
							$pdf->rollbackTransaction(true);
							$pageposafter	= $pageposbefore;
							$pdf->setPageOrientation('', 1, $heightforfooter);	// The only function to edit the bottom margin of current page to set it.
							$pageposdesc	= $pdf->getPage();
							pdf_InfraS_writelinedesc($pdf, $object, $i, $outputlangs, $this->larg_desc, $this->tab_hl, $this->posxdesc, $curY, $hideref, $hidedesc);
							$pageposafter	= $pdf->getPage();
							$posyafter		= $pdf->GetY();
							if ($i == ($nblignes - 1))	// No more lines, and no space left to show total, so we create a new page
							{
								$pdf->AddPage('', '', true);
								if (! empty($tplidx))				$pdf->useTemplate($tplidx);
								$pdf->setPage($pageposafter + 1);
							}	// if ($i == ($nblignes - 1))
						}	// elseif ($posyafter > ($this->page_hauteur - ($heightforfooter + $heightforinfotot)))
						else	$pdf->commitTransaction();	// No pagebreak
						$nexY										= $pdf->GetY();
						$pageposafter								= $pdf->getPage();
						$pdf->setPage($pageposbefore);
						$pdf->setTopMargin($this->marge_haute);
						$pdf->setPageOrientation('', 1, 0);	// The only function to edit the bottom margin of current page to set it.
						if ($pageposafter > $pageposbefore && empty($showpricebeforepagebreak))
						{
							if ($curY > ($this->page_hauteur - $heightforfooter - $this->tab_hl))
							{
								$pdf->setPage($pageposafter);
								$curY		= $tab_top_newpage + ($this->hide_top_table ? $this->ht_decal_round : $this->ht_top_table + $this->ht_decal_round);
							}	// if ($curY > ($this->page_hauteur - $heightforfooter - $this->tab_hl))
							else			$pdf->setPage($pageposdesc);
						}	// if ($pageposafter > $pageposbefore && empty($showpricebeforepagebreak))
						$pdf->SetFont('', '', $default_font_size - 1);   // On repositionne la police par defaut
						if (empty($this->hide_weight_vol))
						{
							$weighttxt	= '';
							$voltxt		= '';
							if ($object->lines[$i]->fk_product_type == 0)
							{
								if ($object->lines[$i]->weight)
									$weighttxt	= round($object->lines[$i]->weight * $object->lines[$i]->qty_shipped, 2).' '.measuring_units_string($object->lines[$i]->weight_units, "weight");
								if ($object->lines[$i]->volume)
									$voltxt		= round($object->lines[$i]->volume * $object->lines[$i]->qty_shipped, 2).' '.measuring_units_string($object->lines[$i]->volume_units ? $object->lines[$i]->volume_units : 0, "volume");
							}	//if ($object->lines[$i]->fk_product_type == 0)
							$pdf->writeHTMLCell($this->larg_weightvol, $this->tab_hl, $this->posxweightvol, $curY, $weighttxt.(($weighttxt && $voltxt) ? '<br>' : '').$voltxt, 0, 0, false, true, 'C');
						}	// if (empty($this->hide_weight_vol))
						if (empty($this->hide_ordered))
						{
						   $pdf->SetXY($this->posxqtyordered, $curY);
						   $pdf->MultiCell($this->larg_qtyordered, $this->tab_hl, $object->lines[$i]->qty_asked, '', 'C');
						}	//if (empty($this->hide_ordered))
						$pdf->SetXY($this->posxqtytoship, $curY);
						$pdf->MultiCell($this->larg_qtytoship, $this->tab_hl, $object->lines[$i]->qty_shipped, '', 'C');
						if (! empty($this->show_ref_col))
						{
							$ref									= pdf_getlineref($object, $i, $outputlangs, $hidedetails);
							$pdf->SetXY($this->posxref, $curY);
							$pdf->MultiCell($this->larg_ref, $this->tab_hl, $ref, 0, 'L');
						}	// if (! empty($this->show_ref_col))
						// Add dash or space between line
						if ($this->dash_between_line && $i < ($nblignes - 1))
						{
							$pdf->setPage($pageposafter);
							$pdf->SetLineStyle(array('color'=>array(128, 128 ,128)));
							$pdf->line($this->marge_gauche, $nexY + 1, $this->page_largeur - $this->marge_droite, $nexY + 1);
							$nexY	+= 2;
						}	// if ($this->dash_between_line && $i < ($nblignes - 1))
						else								$nexY	+= $this->tab_hl;
						// Detect if some page were added automatically and output _tableau for past pages
						while ($pagenb < $pageposafter)
						{
							$pdf->setPage($pagenb);
							$heightforfooter				= $this->_pagefoot($pdf, $object, $outputlangs, 0);
							if ($pagenb == 1)				$this->_tableau($pdf, $object, $tab_top, $this->page_hauteur - $tab_top - $heightforfooter, $outputlangs, $this->hide_top_table, 1, $pagenb);
							else							$this->_tableau($pdf, $object, $tab_top_newpage, $this->page_hauteur - $tab_top_newpage - $heightforfooter, $outputlangs, $this->hide_top_table, 1, $pagenb);
							$pagenb++;
							$pdf->setPage($pagenb);
							$pdf->setPageOrientation('', 1, 0);	// The only function to edit the bottom margin of current page to set it.
							if (empty($this->small_head2))	$this->_pagehead($pdf, $object, 0, $outputlangs);
							else							$this->_pagesmallhead($pdf, $object, 0, $outputlangs);
						}	// while ($pagenb < $pageposafter)
						if (isset($object->lines[$i + 1]->pagebreak) && $object->lines[$i + 1]->pagebreak)
						{
							$heightforfooter				= $this->_pagefoot($pdf, $object, $outputlangs, 0);
							if ($pagenb == 1)				$this->_tableau($pdf, $object, $tab_top, $this->page_hauteur - $tab_top - $heightforfooter, $outputlangs, $this->hide_top_table, 1, $pagenb);
							else							$this->_tableau($pdf, $object, $tab_top_newpage, $this->page_hauteur - $tab_top_newpage - $heightforfooter, $outputlangs, $this->hide_top_table, 1, $pagenb);
							// New page
							$pdf->AddPage();
							if (! empty($tplidx))			$pdf->useTemplate($tplidx);
							$pagenb++;
							if (empty($this->small_head2))	$this->_pagehead($pdf, $object, 0, $outputlangs);
							else							$this->_pagesmallhead($pdf, $object, 0, $outputlangs);
						}	// if (isset($object->lines[$i + 1]->pagebreak) && $object->lines[$i + 1]->pagebreak)
					}	// for ($i = 0 ; $i < $nblignes ; $i++)
					$bottomlasttab							= $this->page_hauteur - $heightforinfotot - $heightforfooter - 1;
					if ($pagenb == 1)						$this->_tableau($pdf, $object, $tab_top, $bottomlasttab - $tab_top, $outputlangs, $this->hide_top_table, 1, $pagenb);
					else									$this->_tableau($pdf, $object, $tab_top_newpage, $bottomlasttab - $tab_top_newpage, $outputlangs, $this->hide_top_table, 0, $pagenb);
					$posyinfo								= $this->_tableau_info($pdf, $object, $bottomlasttab, $outputlangs, 0);
					$posytot								= $this->_tableau_tot($pdf, $object, $bottomlasttab, $outputlangs, 0);
					if ($ht2_coltotal > 4)					$pdf->Line($this->posxtabtotal, $posytot + 2, $this->page_largeur - $this->marge_droite, $posytot + 2);
					$pdf->SetTextColor(0, 0, 0);
					$posyfreetext							= pdf_InfraS_free_text($pdf, $object, $this->posxtabtotal, $posytot + 3, $outputlangs, $this->emetteur, 'SHIPPING_FREE_TEXT', 0);
					if ($ht2_coltotal > 4)
					{
						$pdf->Line($this->posxtabtotal, $posyfreetext + 2, $this->page_largeur - $this->marge_droite, $posyfreetext + 2);
						$posysignarea						= $this->_signature_area($pdf, $object, $posyfreetext + 3, $outputlangs);
					}	// if ($ht2_coltotal > 4)
					else										$posysignarea	= $this->_signature_area($pdf, $object, $posyfreetext, $outputlangs);
					$posy										= $posyinfo > $posysignarea ? $posyinfo : $posysignarea;
					$this->_pagefoot($pdf, $object, $outputlangs, 0);
					if (method_exists($pdf, 'AliasNbPages'))	$pdf->AliasNbPages();
					$pdf->Close();
					$pdf->Output($file, 'F');
					// Add pdfgeneration hook
					$hookmanager->initHooks(array('pdfgeneration'));
					$parameters									= array('file'=>$file, 'object'=>$object, 'outputlangs'=>$outputlangs);
					global $action;
					$reshook=$hookmanager->executeHooks('afterPDFCreation',$parameters,$this,$action);    // Note that $action and $object may have been modified by some hooks
					if (! empty($this->main_umask))				@chmod($file, octdec($this->main_umask));
					$this->result								= array('fullpath' => $file);
					return 1;   // Pas d'erreur
				}	// if (file_exists($dir))
				else
				{
					$this->error=$langs->trans("ErrorCanNotCreateDir",$dir);
					return 0;
				}	// else	// if (file_exists($dir))
			}	// if ($conf->expedition->dir_output)
			else
			{
				$this->error=$langs->trans("ErrorConstantNotDefined","EXP_OUTPUTDIR");
				return 0;
			}	// else	// if ($conf->expedition->dir_output)
		}	// function write_file($object, $outputlangs, $srctemplatepath = '', $hidedetails = 0, $hidedesc = 0, $hideref = 0)

		/********************************************
		*	Show top header of page.
		*
		*	@param		PDF			$pdf     		Object PDF
		*	@param		Object		$object     	Object to show
		*	@param		int	    	$showaddress    0=no, 1=yes
		*	@param		Translate	$outputlangs	Object lang for output
		*	@param		string		$titlekey		Translation key to show as title of document
		*	@return		array		$hauteurhead	'totalhead'		= hight of header
		*											'hauteurcadre	= hight of frame
		********************************************/
		function _pagehead(&$pdf, $object, $showaddress, $outputlangs, $titlekey = "PDFInfrasExpeditionTitle")
		{
			global $conf, $langs, $hookmanager;

			if((float)DOL_VERSION >= 8.0)	$outputlangs->loadLangs(array("main", "bills", "companies", "propal", "orders", "commercial"));
			else
			{
				$outputlangs->load("main");
				$outputlangs->load("bills");
				$outputlangs->load("companies");
				$outputlangs->load("propal");
				$outputlangs->load("orders");
				$outputlangs->load("commercial");
			}	// else	// if((float)DOL_VERSION >= 8.0)
			$default_font_size	= pdf_getPDFFontSize($outputlangs);
			pdf_pagehead($pdf, $outputlangs, $this->page_hauteur);
			$pdf->SetDrawColor(128, 128, 128);
			$pdf->SetTextColor(0, 0, 0);
			$pdf->SetFont('', 'B', $default_font_size + 3);
			$w		= 100;
			$posy	= $this->marge_haute;
			$posx	= $this->page_largeur - $this->marge_droite - $w;
			$pdf->SetXY($this->marge_gauche, $posy);
			// Logo
			$logo	= $conf->mycompany->dir_output.'/logos/'.$this->emetteur->logo;
			if ($logo)
			{
				if (is_file($logo) && is_readable($logo))
				{
					$heightLogo	= pdf_getHeightForLogo($logo);
					$pdf->Image($logo, $this->marge_gauche, $posy, 0, $heightLogo);	// width=0 (auto)
				}	// if (is_file($logo) && is_readable($logo))
				else
				{
					$pdf->SetTextColor(200, 0, 0);
					$pdf->SetFont('', 'B', $default_font_size - 2);
					$pdf->MultiCell($w, $this->tab_hl, $outputlangs->transnoentities("ErrorLogoFileNotFound",$logo), 0, 'L');
					$pdf->MultiCell($w, $this->tab_hl, $outputlangs->transnoentities("ErrorGoToGlobalSetup"), 0, 'L');
					$pdf->SetTextColor(0, 0, 0);
				}	// else	// if (is_file($logo) && is_readable($logo))
			}	// if ($logo)
			else
			{
				$text=$this->emetteur->name;
				$pdf->MultiCell($w, $this->tab_hl, $outputlangs->convToOutputCharset($text), 0, 'L');
				$heightLogo = $this->tab_hl;
			}	// else// if ($logo)
			$heightLogo	+= $posy + $this->tab_hl;
			$pdf->SetFont('', 'B', $default_font_size * 2);
			$title		= $outputlangs->transnoentities($titlekey);
			$pdf->SetXY($posx, $posy);
			$pdf->MultiCell($w, $this->tab_hl * 2, $title, '', 'R');
			$pdf->SetFont('', 'B', $default_font_size - 1);
			$posy		+= $this->tab_hl * 2;
			$txtref		= $outputlangs->transnoentities("RefSending")." : ".$outputlangs->convToOutputCharset($object->ref);
			if ($object->statut == 0)
			{
				$pdf->SetTextColor(128, 0, 0);
				$txtref .= ' - '.$outputlangs->transnoentities("NotValidated");
				$pdf->SetTextColor(0, 0, 0);
			}	// if ($object->statut == 0)
			$pdf->SetXY($posx, $posy);
			$pdf->MultiCell($w, $this->tab_hl, $txtref, '', 'R');
			$pdf->SetFont('', '', $default_font_size - 2);
			$posy		+= $this->tab_hl;
			$txtdt		= $outputlangs->transnoentities("DateDeliveryPlanned")." : ".dol_print_date($object->date_delivery, "dayhour", false, $outputlangs, true);
			$pdf->SetXY($posx, $posy);
			$pdf->MultiCell($w, $this->tab_hl, $txtdt, '', 'R');
			if ($object->ref_customer)
			{
				$posy	+= $this->tab_hl - 0.5;
				$txtcc	= '';
				$txtcc	.= $outputlangs->transnoentities("RefCustomer")." : ".$outputlangs->convToOutputCharset($object->ref_customer);
				$pdf->SetXY($posx, $posy);
				$pdf->MultiCell($w, $this->tab_hl, $txtcc, '', 'R');
			}	// if ($object->ref_customer)			
			// Show list of linked objects
			$posy		= pdf_InfraS_writeLinkedObjects($pdf, $object, $outputlangs, $posx, $posy, $w, $this->tab_hl, 'R');
			$posy		+= 0.5;
			$posycadre	= ($this->use_iso_location ? 40 : ($heightLogo > $posy + $this->tab_hl ? $heightLogo : $posy + $this->tab_hl));
			if ($showaddress)
			{
				// Sender properties
				$carac_emetteur	= '';
				// Add internal contact of proposal if defined
				$arrayidcontact											= array();
				if (! empty($origin) && is_object($object->$origin))	$arrayidcontact	= $object->$origin->getIdContact('internal', 'SALESREPFOLL');
				if (count($arrayidcontact) > 0)
				{
					$object->fetch_user(reset($arrayidcontact));
					$carac_emetteur 		.= ($carac_emetteur ? "\n" : '' ).$outputlangs->convToOutputCharset($object->user->getFullName($outputlangs))."\n";
				}	// if (count($arrayidcontact) > 0)
				$carac_emetteur .= pdf_InfraS_build_address($outputlangs, $this->emetteur, $object->thirdparty, '', 0, 'source', null, 1);
				// Recipient properties
				// If SHIPPING contact defined, we use it
				$usecontact		= false;
				$arrayidcontact	= $object->getIdContact('external', 'SHIPPING');
				if (count($arrayidcontact) > 0)
				{
					$usecontact	= true;
					$result		= $object->fetch_contact($arrayidcontact[0]);
				}	// if (count($arrayidcontact) > 0)
				//Recipient name
				// On peut utiliser le nom de la societe du contact
				if ($usecontact && !empty($this->use_comp_name_contact))	$thirdparty	= $object->contact;
				else														$thirdparty	= $object->thirdparty;
				$carac_client_name	= pdf_InfraS_Build_Third_party_Name($thirdparty, $outputlangs, $this->includealias);
				$recepdetails		= $this->show_recep_details ? 'targetwithdetails' : 'target';
				$carac_client		= pdf_InfraS_build_address($outputlangs, $this->emetteur, $object->thirdparty, (!empty($object->contact) ? $object->contact : null), $usecontact, $recepdetails, $object, 1);
				// Show sender
				$largeurcadreS	= 92;
				if (! empty($this->invert_sender_recipient))	$posx	= $this->page_largeur - $this->marge_droite - $largeurcadreS;
				else											$posx	= $this->marge_gauche;
				$posxcadre	= $posx;
				$posy		= $posycadre;
				// Show sender name
				$pdf->SetFont('', 'B', $default_font_size);
				$pdf->SetXY($posx + 2, $posy + 1);
				$pdf->MultiCell($largeurcadreS - 4, $this->tab_hl, $outputlangs->convToOutputCharset($this->emetteur->name), 0, 'L');
				$posy	= $pdf->getY();
				// Show sender information
				$pdf->SetFont('', '', $default_font_size - 1);
				$pdf->SetXY($posx + 2, $posy);
				$pdf->MultiCell($largeurcadreS - 4, $this->tab_hl, $carac_emetteur, 0, 'L');
				$posyendsender	= $pdf->getY();
				//Show Recipient
				$largeurcadreR	= 92;
				if ($this->page_largeur < 210) $largeurcadreR = 84;	// To work with US executive format
				if (! empty($this->invert_sender_recipient))	$posx	= $this->marge_gauche;
				else											$posx	= $this->page_largeur - $this->marge_droite - $largeurcadreR;
				$posy	= $posycadre;
				// Show recipient name
				$pdf->SetFont('', 'B', $default_font_size);
				$pdf->SetXY($posx + 2, $posy + 1);
				$pdf->MultiCell($largeurcadreR - 4, $this->tab_hl, $carac_client_name, 0, 'L');
				$posy	= $pdf->getY();
				// Show recipient information
				$pdf->SetFont('', '', $default_font_size - 1);
				$pdf->SetXY($posx + 2, $posy);
				$pdf->MultiCell($largeurcadreR - 4, $this->tab_hl, $carac_client, 0, 'L');
				$posyendrecipient	= $pdf->getY();
				//Calcul hauteur des cadres
				if ($posyendsender > $posyendrecipient)	$hauteurcadre	= ($posyendsender - $posycadre) + 1;
				else									$hauteurcadre	= ($posyendrecipient - $posycadre) + 1;
				// Show sender frame
				$pdf->SetFont('', '', $default_font_size - 2);
				$pdf->SetXY($posxcadre + $this->Rounded_rect, $posycadre - 4);
				if (! $this->hide_recep_frame)			$pdf->MultiCell($largeurcadreS, $this->tab_hl + 1, $outputlangs->transnoentities("Sender").":", 0, 'L');
				$pdf->SetAlpha($this->alpha);
				$pdf->RoundedRect($posxcadre, $posycadre, $largeurcadreS, $hauteurcadre, $this->Rounded_rect, '1111', 'F' ,null , explode(',', $this->bg_color));
				$pdf->SetAlpha(1);
				// Show recipient frame
				if (! $this->hide_recep_frame)
				{
					$pdf->SetFont('', '', $default_font_size - 2);
					$pdf->SetXY($posx + $this->Rounded_rect, $posycadre - 4);
					$pdf->MultiCell($largeurcadreR, $this->tab_hl + 1, $outputlangs->transnoentities("Recipient").":", 0, 'L');
					$pdf->RoundedRect($posx, $posycadre, $largeurcadreR, $hauteurcadre, $this->Rounded_rect, '1111');
				}	// if (! $this->hide_recep_frame)
			}	// if ($showaddress)
			$hauteurhead = array('totalhead'=>$posycadre + $hauteurcadre, 'hauteurcadre'=>$hauteurcadre);
			return $hauteurhead;
		}	// function _pagehead(&$pdf, $object, $showaddress, $outputlangs, $titlekey = "PDFInfrasExpeditionTitle")

		/********************************************
		*	Show top small header of page.
		*
		*	@param		PDF			$pdf     		Object PDF
		*	@param		Object		$object     	Object to show
		*	@param		int	    	$showaddress    0=no, 1=yes
		*	@param		Translate	$outputlangs	Object lang for output
		*	@param		string		$titlekey		Translation key to show as title of document
		*	@return		void
		********************************************/
		function _pagesmallhead(&$pdf, $object, $showaddress, $outputlangs, $titlekey = "PDFInfrasExpeditionTitle")
		{
			global $conf, $langs, $hookmanager;
			
			$fromcompany	= $this->emetteur;
			$logo			= $conf->mycompany->dir_output.'/logos/'.$this->emetteur->logo;
			$title			= $outputlangs->transnoentities($titlekey);
			pdf_InfraS_pagesmallhead($pdf, $object, $showaddress, $outputlangs, $title, $fromcompany, $this->formatpage, $this->Rounded_rect, $logo);
		}	// function _pagesmallhead(&$pdf, $object, $showaddress, $outputlangs, $titlekey = "PDFInfrasExpeditionTitle")

		/********************************************
		*   Show table for lines
		*
		*	@param		PDF			$pdf     		Object PDF
		*	@param  	Object		$object     	Object to show
		*	@param		float		$tab_top		Top position of table
		*	@param		float		$tab_height		Height of table (rectangle)
		*	@param		Translate	$outputlangs	Langs object
		*	@param		int			$hidetop		1=Hide top bar of array and title, 0=Hide nothing, -1=Hide only title
		*	@param		int			$hidebottom		Hide bottom bar of array
		*	@return		void
		********************************************/
		function _tableau(&$pdf, $object, $tab_top, $tab_height, $outputlangs, $hidetop = 0, $hidebottom = 0, $pagenb)
		{
			global $conf;

			// Force to disable hidetop and hidebottom
			$hidebottom			= 0;
			if ($hidetop)		$hidetop	= -1;
			$default_font_size	= pdf_getPDFFontSize($outputlangs);
			$pdf->SetDrawColor(128, 128, 128);
			$pdf->SetTextColor(0, 0, 0);
			$pdf->SetFont('', '', $default_font_size - 2);
			// Output Rounded Rectangle
			if (empty($hidetop) || $pagenb == 1)
			{
				if (! empty($this->title_bg))
				{
					$bg_color	= explode(',', $this->bg_color);
					$pdf->SetDrawColor($bg_color[0], $bg_color[1], $bg_color[2]);
					$pdf->RoundedRect($this->marge_gauche, $tab_top, $this->larg_util_cadre, $this->ht_top_table, $this->Rounded_rect, '1111', 'DF', null, explode(',', $this->bg_color));
					$pdf->SetDrawColor(128, 128, 128);
				}	// if (! empty($this->title_bg))
				else	$pdf->RoundedRect($this->marge_gauche, $tab_top, $this->larg_util_cadre, $this->ht_top_table, $this->Rounded_rect, '1111');
				$pdf->RoundedRect($this->marge_gauche, $tab_top + $this->ht_top_table, $this->larg_util_cadre, $tab_height - $this->ht_top_table, $this->Rounded_rect, '1111');
			}	// if (empty($hidetop) || $pagenb == 1)
			else	$pdf->RoundedRect($this->marge_gauche, $tab_top, $this->larg_util_cadre, $tab_height, $this->Rounded_rect, '1111');
			// Show Watermarks
			$pdf->SetAlpha($this->watermark_i_opacity / 100);
			$filigrane	= $conf->mycompany->dir_output.'/logos/'.$this->image_watermark;
			if ($filigrane && is_readable($filigrane))
			{
				$imglinesize	= pdf_InfraS_getSizeForImage($filigrane, $this->larg_util_cadre, $tab_height);
				if (isset($imglinesize['width']) && isset($imglinesize['height']))
				{
					$posXfiligrane			= $this->marge_gauche + (($this->larg_util_cadre - $imglinesize['width']) / 2);	// centrer le filigrane
					if (empty($hidetop))	$pdf->Image($filigrane, $posXfiligrane, $tab_top + $this->ht_top_table + ($tab_height / 2) - ($imglinesize['height'] / 2), $imglinesize['width'], 0);	// width=0 (auto)
					else					$pdf->Image($filigrane, $posXfiligrane, $tab_top + ($tab_height / 2) - ($imglinesize['height'] / 2), $imglinesize['width'], 0);	// width=0 (auto)
				}	// if (isset($imglinesize['width']) && isset($imglinesize['height']))
			}	// if ($filigrane && is_readable($filigrane))
			$pdf->SetAlpha($this->watermark_t_opacity / 100);
			if ($object->statut == 0 && (! empty($this->draft_watermark)))
			{
				if (empty($hidetop))	pdf_InfraS_watermark($pdf, $outputlangs, $this->draft_watermark, $tab_top + $this->ht_top_table + ($tab_height / 2), $this->larg_util_cadre, $this->page_hauteur, 'mm');
				else					pdf_InfraS_watermark($pdf, $outputlangs, $this->draft_watermark, $tab_top + ($tab_height / 2), $this->larg_util_cadre, $this->page_hauteur, 'mm');
			}	// if ($object->statut == 0 && (! empty($this->draft_watermark)))
			$pdf->SetAlpha(1);
			// Show Folder mark
			if (! empty($this->fold_mark))
			{
				$pdf->Line(0, ($this->page_hauteur)/3, $this->fold_mark, ($this->page_hauteur)/3);
				$pdf->Line($this->page_largeur - $this->fold_mark, ($this->page_hauteur)/3, $this->page_largeur, ($this->page_hauteur)/3);
			}	// if (! empty($this->fold_mark))
			// Colonnes
			$pdf->SetFont('', 'B', $default_font_size - 1);
			$this->title_bg ? $pdf->SetTextColor($this->txtcolor[0], $this->txtcolor[1], $this->txtcolor[2]) : $pdf->SetTextColor(0, 0, 0);
			if (empty($this->hide_weight_vol))	$pdf->line($this->posxweightvol, $tab_top, $this->posxweightvol, $tab_top + $tab_height);
			if (empty($this->hide_ordered))		$pdf->line($this->posxqtyordered, $tab_top, $this->posxqtyordered, $tab_top + $tab_height);
			$pdf->line($this->posxqtytoship, $tab_top, $this->posxqtytoship, $tab_top + $tab_height);
			if (! empty($this->show_ref_col))	$pdf->line($this->posxdesc, $tab_top, $this->posxdesc, $tab_top + $tab_height);
			// En-têtes
			if (empty($hidetop) || $pagenb == 1)
			{
				if (empty($this->hide_weight_vol))	$pdf->SetXY($this->posxweightvol, $tab_top);
				$pdf->MultiCell($this->larg_weightvol, $this->ht_top_table, $outputlangs->transnoentities("WeightVolShort"), '', 'C', 0, 1, '', '', true, 0, false, true, $this->ht_top_table, 'M');
				if (empty($this->hide_ordered))
				{
					$pdf->SetXY($this->posxqtyordered, $tab_top);
					$pdf->MultiCell($this->larg_qtyordered, $this->ht_top_table, $outputlangs->transnoentities("PDFInfraSExpeditionOrdered"), '', 'C', 0, 1, '', '', true, 0, false, true, $this->ht_top_table, 'M');
				}	// if (empty($this->hide_ordered))
				$pdf->SetXY($this->posxqtytoship, $tab_top);
				$pdf->MultiCell($this->larg_qtytoship, $this->ht_top_table, $outputlangs->transnoentities("PDFInfraSExpeditionShipped"), '', 'C', 0, 1, '', '', true, 0, false, true, $this->ht_top_table, 'M');
				if (! empty($this->show_ref_col))
				{
					$pdf->SetXY($this->posxref, $tab_top);
					$pdf->MultiCell($this->larg_ref, $this->ht_top_table, $outputlangs->transnoentities("PDFInfrasRef"), '', 'C', 0, 1, '', '', true, 0, false, true, $this->ht_top_table, 'M');
				}	// if (! empty($this->show_ref_col))
			}	// if (empty($hidetop) || $pagenb == 1)
		}	// function _tableau(&$pdf, $object, $tab_top, $tab_height, $outputlangs, $hidetop = 0, $hidebottom = 0, $pagenb)

		/********************************************
		*	Show miscellaneous information (payment mode, payment term, ...)
		*
		*	@param		PDF			$pdf     		Object PDF
		*	@param		Object		$object			Object to show
		*	@param		int			$posy			Y
		*	@param		Translate	$outputlangs	Langs object
		*	@return		int			$posy			Position pour suite
		********************************************/
		function _tableau_info(&$pdf, $object, $posy, $outputlangs, $calculseul = 0)
		{
			global $conf;
			
			$pdf->startTransaction();
			$default_font_size							= pdf_getPDFFontSize($outputlangs);
			$posytabinfo								= $posy + 1;
			$tabinfo_hl									= $this->tab_hl;
			$pdf->SetFont('', '', $default_font_size - 1);
			$pdf->SetTextColor(0, 0, 0);
			$larg_tabinfo								= $this->larg_tabinfo;
			$larg_col1info								= 47;
			$larg_col2info								= $larg_tabinfo - $larg_col1info;
			$posxtabinfo								= $this->marge_gauche;
			$posxcol2info								= $posxtabinfo + $larg_col1info;
			if ($object->shipping_method_id > 0)
			{
				$pdf->SetFont('', 'B', $default_font_size - 2);
				$titre									= $outputlangs->transnoentities("SendingMethod").' : ';
				$pdf->SetXY($posxtabinfo, $posytabinfo);
				$pdf->MultiCell($larg_col1info, $tabinfo_hl, $titre, 0, 'L');
				$pdf->SetFont('', '', $default_font_size - 2);
				$label	= '';
				$code	= $outputlangs->getLabelFromKey($this->db, $object->shipping_method_id, 'c_shipment_mode', 'rowid', 'code');	// Get code using getLabelFromKey
				$label	.= $outputlangs->trans("SendingMethod".strtoupper($code));
				$pdf->SetXY($posxcol2info, $posytabinfo);
				$pdf->MultiCell($larg_col2info, $tabinfo_hl, $label, 0, 'L');
				$posytabinfo							= $pdf->GetY() + 1;
			}	//if ($object->shipping_method_id > 0)
			if (! empty($object->tracking_number))
			{
				$object->GetUrlTrackingStatus($object->tracking_number);
				$pdf->SetFont('', 'B', $default_font_size - 2);
				$titre									= $outputlangs->transnoentities("TrackingNumber").' : ';
				$pdf->SetXY($posxtabinfo, $posytabinfo);
				$pdf->MultiCell($larg_col1info, $tabinfo_hl, $titre, 0, 'L');
				$pdf->SetFont('', '', $default_font_size - 2);
				$pdf->SetXY($posxcol2info, $posytabinfo);
				$pdf->MultiCell($larg_col2info, $tabinfo_hl, $object->tracking_number, 0, 'L');
				$posytabinfo							= $pdf->GetY() + 1;
				if (! empty($object->tracking_url) && $object->tracking_url != $object->tracking_number)
				{
					$pdf->SetFont('', 'B', $default_font_size - 2);
					$titre								= $outputlangs->transnoentities("LinkToTrackYourPackage").' : ';
					$pdf->SetXY($posxtabinfo, $posytabinfo);
					$pdf->MultiCell($larg_col1info, $tabinfo_hl, $titre, 0, 'L');
					$pdf->SetFont('', '', $default_font_size - 2);
					$pdf->writeHTMLCell($larg_col2info, $tabinfo_hl, $posxcol2info, $posytabinfo, $object->tracking_url, 0, 1, false, true, 'L');
					$posytabinfo						= $pdf->GetY() + 1;
				}	// if (! empty($object->tracking_url) && $object->tracking_url != $object->tracking_number)
			}	// if (! empty($object->tracking_number))
			if ($calculseul)
			{
				$heightforinfo							= $posytabinfo - $posy;
				$pdf->rollbackTransaction(true);
				return $heightforinfo;
			}	// if ($calculseul)
			else
			{
				$pdf->commitTransaction();
				return $posytabinfo;
			}	// else // if ($calculseul)
		}	// function _tableau_info(&$pdf, $object, $posy, $outputlangs, $calculseul = 0)

		/********************************************
		*	Show total to pay
		*
		*	@param		PDF			$pdf            Object PDF
		*	@param		Facture		$object         Object invoice
		*	@param		int			$posy			y
		*	@param		Translate	$outputlangs	Objet langs
		*	@return		int							Position pour suite
		********************************************/
		function _tableau_tot(&$pdf, $object, $posy, $outputlangs, $calculseul = 0)
		{
			global $conf;
			
			$pdf->startTransaction();
			$default_font_size					= pdf_getPDFFontSize($outputlangs);
			$posytabtot							= $posy + 1;
			$tabtot_hl							= $this->tab_hl;
			$pdf->SetFont('', '', $default_font_size - 1);
			$bg_color							= explode(',', $this->bg_color);
			$pdf->SetDrawColor($bg_color[0], $bg_color[1], $bg_color[2]);
			$pdf->SetTextColor(0, 0, 0);
			// Tableau total
			$larg_tabtotal						= $this->larg_tabtotal;
			$larg_col4total						= $this->larg_qtytoship;
			$larg_col3total						= $this->larg_qtyordered;
			$larg_col2total						= $this->larg_weightvol;
			$larg_col1total						= $larg_tabtotal - $larg_col4total - $larg_col3total - $larg_col2total;
			$posxtabtotal						= $this->posxtabtotal;
			$posxcol4total						= $this->posxqtytoship;
			$posxcol3total						= $this->posxqtyordered;
			$posxcol2total						= $this->posxweightvol;
			$index								= 0;
			$totalWeighttoshow					= '';
			$totalVolumetoshow					= '';
			// Load dim data
			$tmparray							= $object->getTotalWeightVolume();
			$totalWeight						= round($tmparray['weight'], 2);
			$totalVolume						= $tmparray['volume'];
			$totalOrdered						= $tmparray['ordered'];
			$totalToShip						= $tmparray['toship'];
			// Set true Volume and volume_units not currently stored into database
			if ($object->trueWidth && $object->trueHeight && $object->trueDepth)
			{
				$object->trueVolume				= price(($object->trueWidth * $object->trueHeight * $object->trueDepth), 0, $outputlangs, 0, 0);
				$object->volume_units			= $object->size_units * 3;
			}	// if ($object->trueWidth && $object->trueHeight && $object->trueDepth)
			if ($totalWeight != '')				$totalWeighttoshow	= showDimensionInBestUnit($totalWeight, 0, "weight", $outputlangs);
			if ($totalVolume != '')				$totalVolumetoshow	= showDimensionInBestUnit($totalVolume, 0, "volume", $outputlangs);
			if ($object->trueWeight)			$totalWeighttoshow	= showDimensionInBestUnit($object->trueWeight, $object->weight_units, "weight", $outputlangs);
			if ($object->trueVolume)			$totalVolumetoshow	= showDimensionInBestUnit($object->trueVolume, $object->volume_units, "volume", $outputlangs);
			// Totaux
			$pdf->RoundedRect($posxtabtotal, $posytabtot + ($tabtot_hl * $index), $larg_tabtotal, $tabtot_hl, $this->Rounded_rect > $tabtot_hl / 2 ? $tabtot_hl / 2 : $this->Rounded_rect, '1111', 'DF' ,null , explode(',', $this->bg_color));
			$pdf->SetFont('', 'B', $default_font_size - 1);
			$this->title_bg ? $pdf->SetTextColor($this->txtcolor[0], $this->txtcolor[1], $this->txtcolor[2]) : $pdf->SetTextColor(0, 0, 0);
			$pdf->SetXY($posxtabtotal, $posytabtot + ($tabtot_hl * $index));
			$pdf->MultiCell($larg_col1total, $tabtot_hl, $outputlangs->transnoentities("Total"), 0, 'L', 0);
			// Total to ship
			$pdf->SetXY($posxcol4total, $posytabtot + ($tabtot_hl * $index));
			$pdf->MultiCell($larg_col4total, $tabtot_hl, $totalToShip, 0, 'C', 0);
			// Total ordered
			if (empty($this->hide_ordered))
			{
				$pdf->SetXY($posxcol3total, $posytabtot + ($tabtot_hl * $index));
				$pdf->MultiCell($larg_col3total, $tabtot_hl, $totalOrdered, 0, 'C', 0);
			}	// if (empty($this->hide_ordered))
			// Total weight
			if (empty($this->hide_weight_vol))
			{
				if ($totalWeighttoshow)
				{
					$pdf->SetXY($posxcol2total, $posytabtot + ($tabtot_hl * $index));
					$pdf->MultiCell($larg_col2total, $tabtot_hl, $totalWeighttoshow, 0, 'C', 0);
					$index++;
				}	// if ($totalWeighttoshow)
				// Total volume
				if ($totalVolumetoshow)
				{
					$pdf->RoundedRect($posxtabtotal, $posytabtot + ($tabtot_hl * $index), $larg_tabtotal, $tabtot_hl, $this->Rounded_rect > $tabtot_hl / 2 ? $tabtot_hl / 2 : $this->Rounded_rect, '1111', 'DF' ,null , explode(',', $this->bg_color));
					$pdf->SetXY($posxcol2total, $posytabtot + ($tabtot_hl * $index));
					$pdf->MultiCell($larg_col2total, $tabtot_hl, $totalVolumetoshow, 0, 'C', 0);
					$index++;
				}	// if ($totalVolumetoshow)
			}	// if (empty($this->hide_weight_vol))
			$pdf->SetFont('', '', $default_font_size - 1);
			$pdf->SetDrawColor(128, 128, 128);
			$pdf->SetTextColor(0, 0, 0);
			$posytabtot							= $pdf->GetY() + 1;
			if ($calculseul)
			{
				$heightfortot					= $posytabtot - $posy;
				$pdf->rollbackTransaction(true);
				return $heightfortot;
			}	// if ($calculseul)
			else
			{
				$pdf->commitTransaction();
				return $posytabtot;
			}	// else // if ($calculseul)
		}	// function _tableau_tot(&$pdf, $object, $posy, $outputlangs, $calculseul = 0)

		/********************************************
		*	Show area for the customer to sign
		*
		*	@param		PDF			$pdf            Object PDF
		*	@param		Facture		$object         Object invoice
		*	@param		int			$posy			y
		*	@param		Translate	$outputlangs	Objet langs
		*	@return		int							Position pour suite
		********************************************/
		function _signature_area(&$pdf, $object, $posy, $outputlangs)
		{
			$default_font_size	= pdf_getPDFFontSize($outputlangs);
			$signarea_top		= $posy + 1;
			$signarea_hl		= $this->tab_hl;
			$pdf->SetFont('', '', $default_font_size - 2);
			$pdf->SetTextColor(0, 0, 0);
			$larg_signarea		= $this->larg_tabtotal;
			$posxsignarea		= $this->posxtabtotal;
			$pdf->SetXY($posxsignarea + $this->Rounded_rect, $signarea_top);
			$pdf->MultiCell($larg_signarea, $signarea_hl, $outputlangs->transnoentities("ExpeditionCustomerSignature"), 0, 'L', 0);
			$pdf->RoundedRect($posxsignarea, $signarea_top + $signarea_hl, $larg_signarea, $this->ht_signarea, $this->Rounded_rect, '1111');
			return ($signarea_top + ($signarea_hl * 7));
		}	// function _signature_area(&$pdf, $object, $posy, $outputlangs)

		/********************************************
		*	Show footer of page. Need this->emetteur object
		*
		*	@param		PDF			$pdf     		The PDF factory
		*	@param		Translate	$outputlangs	Object lang for output
		*	@param		Societe		$fromcompany	Object company
		*	@param		int			$marge_basse	Margin bottom we use for the autobreak
		*	@param		int			$marge_gauche	Margin left
		*	@param		int			$page_hauteur	Page height
		*	@param		Object		$object			Object shown in PDF
		*	@param		int			$showdetails	Show company details into footer. This param seems to not be used by standard version.
													(10=minimum, 11 = 1 + Logos de pied, 20 = Show address and managers, 21 = Show 2 + Logos de pied)
		*	@param		int			$hidesupline	Completly hide the line up to footer (for some edition with only table)
		*	@param		int			$calculseul		Arrête la fonction au calcul de hauteur nécessaire
		*	@return		int							Return height of bottom margin including footer text
		********************************************/
		function _pagefoot(&$pdf, $object, $outputlangs, $calculseul)
		{
			global $conf, $langs;
			
			$showdetails									= 10;
			if (! empty($this->type_foot))	$showdetails	= 20;
			if (! empty($this->pied))		$showdetails	+= 1;
			return pdf_InfraS_pagefoot($pdf, $outputlangs, $this->emetteur, $this->formatpage, $showdetails, 0, $calculseul, $this->pied, $this->maxsizeimgfoot, $this->hidepagenum);
		}	// function _pagefoot(&$pdf, $object, $outputlangs, $calculseul)
	}	// class pdf_InfraS_expedition extends ModelePdfExpedition
?>