<?php
/*
 * Copyright (C) 2014-2016  Jean-François Ferry	<hello@librethic.io>
 * 				 2016       Christophe Battarel <christophe@altairis.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 *  \file       htdocs/core/triggers/interface_50_modTicketsup_TicketEmail.class.php
 *  \ingroup    core
 *  \brief      Fichier
 *  \remarks    Son propre fichier d'actions peut etre cree par recopie de celui-ci:
 *              - Le nom du fichier doit etre: interface_99_modMymodule_Mytrigger.class.php
 *                                           ou: interface_99_all_Mytrigger.class.php
 *              - Le fichier doit rester stocke dans core/triggers
 *              - Le nom de la classe doit etre InterfaceMytrigger
 *              - Le nom de la propriete name doit etre Mytrigger
 */
require_once DOL_DOCUMENT_ROOT.'/core/triggers/dolibarrtriggers.class.php';

/**
 *  Class of triggers for ticketsup module
 */
class InterfaceTicketEmail extends DolibarrTriggers
{
    public $db;

    /**
     *   Constructor
     *
     *   @param DoliDB $db Database handler
     */
    public function __construct($db)
    {
        $this->db = $db;

        $this->name = preg_replace('/^Interface/i', '', get_class($this));
        $this->family = "ticketsup";
        $this->description = "Triggers of the module ticketsup";
        $this->version = 'dolibarr'; // 'development', 'experimental', 'dolibarr' or version
        $this->picto = 'ticketsup@ticketsup';
    }

    /**
     *   Return name of trigger file
     *
     *   @return string      Name of trigger file
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     *   Return description of trigger file
     *
     *   @return string      Description of trigger file
     */
    public function getDesc()
    {
        return $this->description;
    }

    /**
     *   Return version of trigger file
     *
     *   @return string      Version of trigger file
     */
    public function getVersion()
    {
        global $langs;
        $langs->load("admin");

        if ($this->version == 'development') {
            return $langs->trans("Development");
        } elseif ($this->version == 'experimental') {
            return $langs->trans("Experimental");
        } elseif ($this->version == 'dolibarr') {
            return DOL_VERSION;
        } elseif ($this->version) {
            return $this->version;
        } else {
            return $langs->trans("Unknown");
        }
    }

    /**
     *      Function called when a Dolibarrr business event is done.
     *      All functions "run_trigger" are triggered if file is inside directory htdocs/core/triggers
     *
     *      @param  string    $action Event action code
     *      @param  Object    $object Object
     *      @param  User      $user   Object user
     *      @param  Translate $langs  Object langs
     *      @param  conf      $conf   Object conf
     *      @return int                     <0 if KO, 0 if no triggered ran, >0 if OK
     */
    public function runTrigger($action, $object, User $user, Translate $langs, Conf $conf)
    {
		$ok = 0;
		
    	switch ($action) {
    		case 'TICKET_ASSIGNED':
                //
                // Send email to assigned user
                //
                dol_syslog("Trigger '" . $this->name . "' for action '$action' launched by " . __FILE__ . ". id=" . $object->id);

                if ($object->fk_user_assign != $user->id) {
                    $userstat = new User($this->db);
                    $res = $userstat->fetch($object->fk_user_assign);
                    if ($res) {
                        // Mail assigned user
                        $sendto = $userstat->email;
                        $from = dolGetFirstLastname($user->firstname, $user->lastname) . '<' . $user->email . '>';

                        // Set headers for ticketfetcher module
                        if ($conf->ticketfetcher->enabled) {
                            dol_include_once('ticketfetcher/class/ticketfetcher.class.php');
                            dol_include_once('ticketfetcher/class/ticketfetcheraccount.class.php');
                            $fetcherstatic = new TicketFetcher();

                            // Get origin email reference
                            $MessageUid = $fetcherstatic->getMessageIdForTicket($object->id);
                            $moreinheader = "REFERENCES: ".$MessageUid."\n";
                            $trackid = 'ticketsup'.$object->id;

                            $accountId = $fetcherstatic->getAccountForTicket($object->id);
                            $tfAccountStatic = new TicketfetcherAccount($this->db);
                            if ($tfAccountStatic->fetch($accountId) > 0) {
                                $from = $tfAccountStatic->label. ' <'.$tfAccountStatic->userAccount->email.'>';
                                $label_title = $tfAccountStatic->label;
                            }
                        } else {
                            $trackid = 'ticketsup'.$object->id;
                          //$from = $conf->global->MAIN_INFO_SOCIETE_NOM.'<'.$conf->global->TICKETS_NOTIFICATION_EMAIL_FROM.'>';
                            $label_title = $conf->global->MAIN_INFO_SOCIETE_NOM;
                        }

	                    $subject = '[' . $label_title . '] ' . $langs->transnoentities('TicketAssignedToYou');
	                    $message = '<p>' . $langs->transnoentities('TicketAssignedEmailBody', $object->track_id, dolGetFirstLastname($user->firstname, $user->lastname)) . "</p>";
	                    $message .= '<ul><li>' . $langs->trans('Title') . ' : ' . $object->subject . '</li>';
	                    $message .= '<li>' . $langs->trans('Type') . ' : ' . $object->type_label . '</li>';
	                    $message .= '<li>' . $langs->trans('Category') . ' : ' . $object->category_label . '</li>';
	                    $message .= '<li>' . $langs->trans('Severity') . ' : ' . $object->severity_label . '</li>';
	                    // Extrafields
	                    if (is_array($object->array_options) && count($object->array_options) > 0) {
	                        foreach ($object->array_options as $key => $value) {
                                $message .= '<li>' . $langs->trans($key) . ' : ' . $value . '</li>';
	                        }
	                    }
	
	                    $message .= '</ul>';
	                    $message .= '<p>' . $langs->trans('Message') . ' : <br />' . $object->message . '</p>';
	                    $message .= '<p><a href="' . dol_buildpath('/ticketsup/card.php', 2) . '?track_id=' . $object->track_id . '">' . $langs->trans('SeeThisTicketIntomanagementInterface') . '</a><br /></p>';
	
	                    // Init to avoid errors
	                    $filepath = array();
	                    $filename = array();
	                    $mimetype = array();

                        $msgishtml = -1;

                        $message = dol_nl2br($message);

                        if (!empty($conf->global->TICKETS_DISABLE_MAIL_AUTOCOPY_TO)) {
                            $old_MAIN_MAIL_AUTOCOPY_TO = $conf->global->MAIN_MAIL_AUTOCOPY_TO;
                            $conf->global->MAIN_MAIL_AUTOCOPY_TO = '';
                        }
                        include_once DOL_DOCUMENT_ROOT . '/core/class/CMailFile.class.php';
	                    $mailfile = new CMailFile(
                            $subject,
                            $sendto,
                            $from,
                            $message,
                            $filepath,
                            $mimetype,
                            $filename,
                            $sendtocc,
                            '',
                            $deliveryreceipt,
                            $msgishtml,
                            $errors_to,
                            $css,
                            $trackid,
                            $moreinheader
                        );

	                    if ($mailfile->error) {
                            setEventMessage($mailfile->error, 'errors');
	                    } else {
	                        $result = $mailfile->sendfile();
	                    }
                        if (!empty($conf->global->TICKETS_DISABLE_MAIL_AUTOCOPY_TO)) {
                            $conf->global->MAIN_MAIL_AUTOCOPY_TO = $old_MAIN_MAIL_AUTOCOPY_TO;
                        }

                        $ok = 1;
	                }
	            }
	            break;
	            
	            
    		case 'TICKET_CREATE':
	            dol_syslog("Trigger '" . $this->name . "' for action '$action' launched by " . __FILE__ . ". id=" . $object->id);
	
	            // Init to avoid errors
	            $filepath = array();
	            $filename = array();
	            $mimetype = array();
	
	            $langs->load('ticketsup@ticketsup');
	
	            $object->fetch('', $object->track_id);

                $trackid = 'ticketsup'.$object->id;
                $from = $conf->global->MAIN_INFO_SOCIETE_NOM.' <'.$conf->global->TICKETS_NOTIFICATION_EMAIL_FROM.'>';
                $label_title = $conf->global->MAIN_INFO_SOCIETE_NOM;

                // Set headers for ticketfetcher module
                if ($conf->ticketfetcher->enabled) {
                    dol_include_once('ticketfetcher/class/ticketfetcher.class.php');
                    dol_include_once('ticketfetcher/class/ticketfetcheraccount.class.php');
                    $fetcherstatic = new TicketFetcher();

                    // Get origin email reference
                    $MessageUid = $fetcherstatic->getMessageIdForTicket($object->id);
                    if ($MessageUid) {
                        $moreinheader = "REFERENCES: ".$MessageUid."\n";

                        $accountId = $fetcherstatic->getAccountForTicket($object->id);
                        $tfAccountStatic = new TicketfetcherAccount($this->db);
                        if ($tfAccountStatic->fetch($accountId) > 0) {
                            $from = $tfAccountStatic->label. ' <'.$tfAccountStatic->userAccount->email.'>';
                            $label_title = $tfAccountStatic->label;
                            $trackid = 'ticketsup'.$object->id;
                        }
                    }
                }
	
	            /* Send email to admin */
	            $sendto = $conf->global->TICKETS_NOTIFICATION_EMAIL_TO;
	            $subject = '['.$label_title.'] '.$langs->transnoentities('TicketNewEmailSubjectAdmin');

	            $message_admin= '<p>' . $langs->transnoentities('TicketNewEmailBodyAdmin', $object->track_id) . '</p>';
	            $message_admin.='<ul><li>'.$langs->trans('Title').' : '.$object->subject.'</li>';
	            $message_admin.='<li>'.$langs->trans('Type').' : '.$object->type_label.'</li>';
	            $message_admin.='<li>'.$langs->trans('Category').' : '.$object->category_label.'</li>';
	            $message_admin.='<li>'.$langs->trans('Severity').' : '.$object->severity_label.'</li>';
	            $message_admin.='<li>'.$langs->trans('From').' : '.( $object->email_from ? $object->email_from : ( $object->fk_user_create > 0 ? $langs->trans('Internal') : '') ).'</li>';
	            // Extrafields
	            if (is_array($object->array_options) && count($object->array_options) > 0) {
	                foreach ($object->array_options as $key => $value) {
	                      $message_admin.='<li>'.$langs->trans($key).' : '.$value.'</li>';
	                }
	            }
	            $message_admin.='</ul>';
	
	            if ($object->fk_soc > 0) {
	                      $object->fetch_thirdparty();
	                      $message_admin.='<p>'.$langs->trans('Company'). ' : '.$object->thirdparty->name.'</p>';
	            }
	
	            $message_admin.='<p>'.$langs->trans('Message').' : <br />'.$object->message.'</p>';
	            $message_admin.='<p><a href="'.dol_buildpath('/ticketsup/card.php', 2).'?track_id='.$object->track_id.'">'.$langs->trans('SeeThisTicketIntomanagementInterface').'</a></p>';

                if ($conf->ticketfetcher->enabled) {
                    // fingerprint into text
                    $add_header                = TicketFetcher::getContentHeader($langs);
                    $add_footer                = TicketFetcher::getContentFooter($langs);

                    $message_admin = $add_header . $message_admin;
                    $message_admin.= "Ticket-ID:".$object->id."-".$object->track_id."\n";
                    $message_admin.= $add_footer;
                }
                
	            $replyto = $from;

                $message_admin = dol_nl2br($message_admin);

                if (!empty($conf->global->TICKETS_DISABLE_MAIL_AUTOCOPY_TO)) {
                    $old_MAIN_MAIL_AUTOCOPY_TO = $conf->global->MAIN_MAIL_AUTOCOPY_TO;
                    $conf->global->MAIN_MAIL_AUTOCOPY_TO = '';
                }
                include_once DOL_DOCUMENT_ROOT.'/core/class/CMailFile.class.php';
	            $mailfile = new CMailFile(
                    $subject,
                    $sendto,
                    $from,
                    $message_admin,
                    $filepath,
                    $mimetype,
                    $filename,
                    $sendtocc,
                    '',
                    $deliveryreceipt,
                    -1,
                    $errors_to,
                    $css,
                    $trackid,
                    $moreinheader
                );
	            if ($mailfile->error) {
                    dol_syslog($mailfile->error, LOG_DEBUG);
	            } else {
	                     $result=$mailfile->sendfile();
	            }
                if (!empty($conf->global->TICKETS_DISABLE_MAIL_AUTOCOPY_TO)) {
                    $conf->global->MAIN_MAIL_AUTOCOPY_TO = $old_MAIN_MAIL_AUTOCOPY_TO;
                }

                /* Send email to customer */
	            $sendto = '';
                // External user with email
                if ($user->socid && !empty($user->email)) {
                    $sendto = $user->email;
                }
	            if (empty($user->socid) && empty($user->email)) {
                    $object->fetch_thirdparty();
                    $sendto = $object->thirdparty->email;
	            }

                if (empty($sendto) && !empty($object->origin_email)) {
                    $sendto = $object->origin_email;
                }

                if (empty($sendto) && !empty($object->thirdparty->email)) {
                    $sendto = $object->thirdparty->email;
                }
                // Ticket creation from API or fetcher
                if (empty($object->notify_tiers_at_create) && empty($conf->global->TICKETS_DISABLE_ALL_MAILS)) {
                    $object->notify_tiers_at_create = 1;
                }

	            if ($sendto && $object->notify_tiers_at_create) {
		            $subject = '['.$label_title.'] '.$langs->transnoentities('TicketNewEmailSubjectCustomer');
		            $message_customer= '<p>' . $langs->transnoentities('TicketNewEmailBodyCustomer', $object->track_id).'</p>';
		            $message_customer.='<ul><li>'.$langs->trans('Title').' : '.$object->subject.'</li>';
		            $message_customer.='<li>'.$langs->trans('Type').' : '.$object->type_label.'</li>';
		            $message_customer.='<li>'.$langs->trans('Category').' : '.$object->category_label.'</li>';
		            $message_customer.='<li>'.$langs->trans('Severity').' : '.$object->severity_label.'</li>';
		            // Extrafields
		            if ($conf->global->TICKETS_EXTRAFIELDS_PUBLIC) {
		                if (is_array($object->array_options) && count($object->array_options) > 0) {
		                    foreach ($object->array_options as $key => $value) {
                                $message_customer.='<li>'.$langs->trans($key).' : '.$value.'</li>';
		                    }
		                }
		            }
		            $message_customer.='</ul>';
		            $message_customer.='<p>'.$langs->trans('Message').' : <br />'.$object->message.'</p>';
                    if (!empty($conf->global->TICKETS_ENABLE_PUBLIC_INTERFACE)) {
    		            $url_public_ticket = ($conf->global->TICKETS_URL_PUBLIC_INTERFACE?$conf->global->TICKETS_URL_PUBLIC_INTERFACE.'/':dol_buildpath('/ticketsup/public/view.php', 2)).'?track_id='.$object->track_id;
    		            $message_customer.='<p>' . $langs->trans('TicketNewEmailBodyInfosTrackUrlCustomer') . ' : <a href="'.$url_public_ticket.'">'.$url_public_ticket.'</a></p>';
                    }
                    if (! $conf->ticketfetcher->enabled) {
                        $message_customer.='<p>'.$langs->trans('TicketEmailPleaseDoNotReplyToThisEmail').'</p>';
                    } else {
                        // fingerprint into text
                        $add_header                = TicketFetcher::getContentHeader($langs);
                        $add_footer                = TicketFetcher::getContentFooter($langs);

                        $message_customer = $add_header . $message_customer;
                        $message_customer.= "Ticket-ID:".$object->id."-".$object->track_id."\n";
                        $message_customer.= $add_footer;
                    }
        
		            $replyto = $from; // TODO: set headers after CmailFile constructor
		
		            // Init to avoid errors
		            $filepath = array();
		            $filename = array();
		            $mimetype = array();

                    $msgishtml = -1;

                    $message_customer = dol_nl2br($message_customer);

                    if (!empty($conf->global->TICKETS_DISABLE_MAIL_AUTOCOPY_TO)) {
                        $old_MAIN_MAIL_AUTOCOPY_TO = $conf->global->MAIN_MAIL_AUTOCOPY_TO;
                        $conf->global->MAIN_MAIL_AUTOCOPY_TO = '';
                    }
                    include_once DOL_DOCUMENT_ROOT.'/core/class/CMailFile.class.php';
		            $mailfile = new CMailFile(
                        $subject,
                        $sendto,
                        $from,
                        $message_customer,
                        $filepath,
                        $mimetype,
                        $filename,
                        $sendtocc,
                        '',
                        $deliveryreceipt,
                        $msgishtml,
                        $errors_to,
                        $css,
                        $trackid,
                        $moreinheader
                    );
		            if ($mailfile->error) {
                        dol_syslog($mailfile->error, LOG_DEBUG);
		            } else {
		                      $result=$mailfile->sendfile();
		            }
                    if (!empty($conf->global->TICKETS_DISABLE_MAIL_AUTOCOPY_TO)) {
                        $conf->global->MAIN_MAIL_AUTOCOPY_TO = $old_MAIN_MAIL_AUTOCOPY_TO;
                    }
                } else {
                    dol_syslog("Trigger '" . $this->name ."' No sendto parametres", LOG_WARNING);
                }
                $ok = 1;
	            
	            break;
	            
            case 'TICKET_DELETE':
            	dol_syslog("Trigger '" . $this->name . "' for action '$action' launched by " . __FILE__ . ". id=" . $object->id);
            	break;
            
           	case 'TICKET_MODIFY':
           		dol_syslog("Trigger '" . $this->name . "' for action '$action' launched by " . __FILE__ . ". id=" . $object->id);
           		break;
            
           	case 'TICKET_MARK_READ':
           		dol_syslog("Trigger '" . $this->name . "' for action '$action' launched by " . __FILE__ . ". id=" . $object->id);
           		break;
           		
           	case 'TICKET_CLOSED':
           		dol_syslog("Trigger '" . $this->name . "' for action '$action' launched by " . __FILE__ . ". id=" . $object->id);
           		break;
    	}
       

        return $ok;
    }
}
