<?php
/* Copyright (C) Yassine Belkaid y.belkaid@nextconcept.ma
 *
 */

/**
 * \file    btp/tender.class.php
 * \ingroup btp
 * \brief   This file is btp CRUD class file
 *
 */

require_once DOL_DOCUMENT_ROOT . '/btp/db/nxdb.class.php';

/**
 * Class btp
 *
 * Contract Management class
 */
class Tender extends NXdb
{
	/**
	 * @var string Error code (or message)
	 * @deprecated
	 * @see btp::errors
	 */
	public $error;
	/**
	 * @var string[] Error codes (or messages)
	 */
	public $errors = array();
	/**
	 * @var string Id to identify managed objects
	 */
	public $element = 'tender';
	/**
	 * @var string Name of table without prefix where object is stored
	 */
	public $table_element = 'tender';

	/**
	 * @var int ID
	 */
	public $id;
	public $ref;
	public $statut;
	public $fk_createdby;
	// public $fk_classification;
	public $fk_third_party;
	public $fk_bank;
	public $created_at;
	public $updated_at;
	public $created_date;
	public $completion_time;
	public $date_bid_open;
	public $subject;
	public $type; // 1 -> marche normale, 2 -> marche cadre 

	public $now;
	public $lines = array();
 
	/**
	 * Constructor
	 *
	 * @param DoliDb $db Database handler
	 */
	public function __construct(DoliDBMysqli $db)
	{
		$this->db 		 = $db;
		$this->now 		 = new \DateTime("now");
		$this->now 		 = $this->now->format('Y-m-d H:i:s');
		$this->updated_at = $this->now;
		return 1;
	}

	/**
	 * Create object into database
	 *
	 * @return int   id of last inserted id, otherwise -1 if error arised 
	 */
	public function create()
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		// Clean parameters
		$this->fk_createdby 		= (int) $this->fk_createdby ? $this->db->escape($this->fk_createdby) : 0;
		$this->fk_bank 				= (int)$this->fk_bank ? $this->db->escape($this->fk_bank) : 0;
		$this->fk_third_party		= (int)$this->fk_third_party ? $this->db->escape($this->fk_third_party) : 0;
		$this->ref					= $this->ref ? $this->db->escape($this->ref) : '';
		$this->created_at 			= $this->now;
		$this->created_date			= $this->created_date ? $this->db->escape($this->created_date) : $this->now;
		$this->completion_time		= $this->completion_time ? $this->db->escape($this->completion_time) : null;
		$this->date_bid_open		= $this->date_bid_open ? $this->db->escape($this->date_bid_open) : null;
		$this->subject				= $this->subject ? $this->db->escape($this->subject) : null;
		$this->amount				= $this->amount ? $this->db->escape($this->amount) : null;

		// Insert request
		$sql = 'INSERT INTO ' . MAIN_DB_PREFIX . $this->table_element . ' (fk_createdby, fk_third_party, fk_bank, created_at, updated_at, created_date, completion_time, date_bid_open, ref, subject, amount) VALUES (';
		$sql .= $this->fk_createdby.', '.$this->fk_third_party.', '.$this->fk_bank.', "'.$this->created_at.'", "'.$this->updated_at .'", "'.$this->created_date.'", "'.$this->completion_time.'", "'.$this->date_bid_open.'", "'.$this->ref.'", "'.$this->subject.'", '.$this->amount.')';

		$this->db->begin();
		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->error = 'Error tender ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return $this->getLasInsrtedId();
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param int    $id  Id object
	 * @param string $ref Ref
	 *
	 * @return int <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id, $ref = null)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;
		
		if (null !== $ref) {
			$sql .= ' WHERE ref = ' . '\'' . $this->db->escape($ref) . '\'';
		} else {
			$sql .= ' WHERE rowid = ' . (int)$id;
		}

		$resql = $this->db->query($sql);
		if ($resql) {
			$numrows = $this->db->num_rows($resql);
			if ($numrows) {
				$obj 					 = $this->db->fetch_object($resql);
				$this->id 				 = $obj->rowid;
				$this->ref 			 	 = $obj->ref;
				$this->statut  		 	 = $obj->statut;
				$this->fk_createdby 	 = $obj->fk_createdby;
				$this->fk_third_party    = $obj->fk_third_party;
				$this->fk_bank 			 = $obj->fk_bank;
				$this->created_date		 = $this->db->jdate($obj->created_date);
				$this->date_bid_open	 = $this->db->jdate($obj->date_bid_open);
				$this->completion_time	 = $obj->completion_time;
				$this->subject 			 = $obj->subject;
				$this->amount 			 = (float) $obj->amount;
				$this->updated_at 		 = $this->db->jdate($obj->updated_at);
				$this->created_at 		 = $this->db->jdate($obj->created_at);

				$this->db->free($resql);

				return 1;
			}
			else {
				return 0;
			}
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param string $sortorder Sort Order
	 * @param string $sortfield Sort field
	 * @param int    $limit     offset limit
	 * @param int    $offset    offset limit
	 * @param array  $filter    filter array
	 * @param string $filtermode filter mode (AND or OR)
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function fetchAll($sortorder = 'DESC', $sortfield = '', $limit = 0, $offset = 0, $filter = '')
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;

		if (!empty($filter)) {
			$sql .= ' WHERE '. substr($filter, 4);
		}
		
		if (!empty($sortfield)) {
			$sql .= $this->db->order($sortfield, $sortorder);
		}
         if(!empty($limit)){
         $sql.= " limit  ".$limit ;
		}
		$this->lines = array();
		$resql 		 = $this->db->query($sql);
		
		if ($resql) {
			$num = $this->db->num_rows($resql);

			while ($obj = $this->db->fetch_object($resql)) {
				$line = new stdClass;

				$line->id 				 = $obj->rowid;
				$line->ref 			 	 = $obj->ref;
				$line->statut 			 = $obj->statut;
				$line->fk_createdby 	 = $obj->fk_createdby;
				$line->fk_third_party    = $obj->fk_third_party;
				$line->fk_bank 			 = $obj->fk_bank;
				$line->created_date		 = $this->db->jdate($obj->created_date);
				$line->date_bid_open	 = $this->db->jdate($obj->date_bid_open);
				$line->completion_time	 = $obj->completion_time;
				$line->subject 			 = $obj->subject;
				$line->amount 			 = (float) $obj->amount;
				$line->updated_at 		 = $this->db->jdate($obj->updated_at);
				$line->created_at 		 = $this->db->jdate($obj->created_at);
				$this->lines[] 			 = $line;
			}
			$this->db->free($resql);

			return $num;
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Update object into database
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function update($id, array $data)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		if (!$id || $id <= 0 || !is_array($data))
			return false;

		// Update request
		$sql = 'UPDATE ' . MAIN_DB_PREFIX . $this->table_element . ' SET ';

		if (count($data) && is_array($data))
			foreach ($data as $key => $val) {
				$val = is_numeric($val) ? $val : '"'. $val .'"';
				$sql .= '`'. $key .'` = '. $val .',';
			}

		$sql  = substr($sql, 0, -1);
		$sql .= ' WHERE rowid = ' . (int)$id;

		if($data['archived'] == 1){
			$this->archive_market($id);
			
		}
        
		$this->db->begin();

		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();
			
			$this->error 	= 'Error ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}


	public function archive_market($id)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		if (!$id || $id <= 0 )
			return false;

		// Update request
			$sql .= 'UPDATE ' . MAIN_DB_PREFIX . 'market SET ';
			$sql .= '`archived` = 1 ';
			$sql .= ' WHERE `fk_tender` = ' . (int)$id;

		$this->db->begin();

		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();
			
			$this->error 	= 'Error ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}

	}



	/**
	 * Delete object in database
	 *
	 * @param User $user      User that deletes
	 * @param bool $notrigger false=launch triggers after, true=disable triggers
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function delete($id)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		if (!$id)
			return false;

		$this->db->begin();

		$sql 	= 'DELETE FROM ' . MAIN_DB_PREFIX . $this->table_element .' WHERE rowid = ' . (int)$id;
		$resql 	= $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}

	/**
     *	Return clicable name (with picto eventually)
     *
     *  @author Yassine Belkaid <y.belkaid@nextconcept.ma>
     *	@param		int			$withpicto		0=_No picto, 1=Includes the picto in the linkn, 2=Picto only
     *	@return		string						String with URL
     */
    public function getNomUrl($withpicto = 0,  $id = null, $ref = null)
    {
        global $langs;

        $result	= '';
        $setRef = (null !== $ref) ? $ref : '';
        $id  	= ($id  ?: '');
        $label  = $langs->trans("Show").': '. $setRef;

        $link 	 = '<a href="'.DOL_URL_ROOT.'/btp/list.php?id='. (int)$id .'" title="'.dol_escape_htmltag($label, 1).'" class="classfortooltip nowrap_cls">';
        $linkend ='</a>';
        $picto   = $this->defaultPicto;

        if ($withpicto) $result.=($link.img_object($label, $picto, 'class="classfortooltip"').$linkend);
        if ($withpicto && $withpicto != 2) $result.=' ';
        if ($withpicto != 2) $result.=$link.$setRef.$linkend;

        $result = $link."<div class='icon-tender mainvmenu'></div>  ".$setRef.$linkend;
        return $result;
    }
	
	/**
	 * Get AO type
	 *
	 * @param integer $type type value of marche normal/cadre
	 *
	 * @return string
	 */
    public static function getMarcheType($type)
    {
    	global $langs;

    	switch ((int)$type) {
    		case 1:
    			return $langs->trans('MarcheNormal');
    			break;
    		case 2:
    			return $langs->trans('MarcheCadre');
    			break;
    	}
    }

    /**
	 * Get All AO types
	 *
	 * @return array
	 */
    public static function getMarcheTypes()
    {
    	global $langs;

    	return array(
    		1 => $langs->trans('MarcheNormal'),
    		2 => $langs->trans('MarcheCadre')
    	);
    }

    /**
     * Get status level : Drfat, Approved or Archived
     *
     * @param integer $statut value of status
     *
     * @return string
     */
    public function statusLevel($statut)
	{
		global $langs;

		switch ($statut) {
			case 0:
				return $langs->trans('Draft').' '.img_picto($langs->trans('Draft'),'statut0'); // Draft
				break;
			
			case 1:
				return $langs->trans('Approved').' '.img_picto($langs->trans('Approved'), 'statut6');
				break;

			case 2:
				return $langs->trans('Archived').' '.img_picto($langs->trans('Archived'),'statut5');
				break;
		}

		return false;
    }

    /**
     * Create general information about AO
     */
    public function getTenderHeader()
    {
    	global $langs;
    	$html = '<table class="border" width="100%">';
		$html .= '<tr><td width="25%">'.$langs->trans("Subject").'</td>';
		$html .= '<td colspan="2">'. $this->subject .'</td></tr>';
		// $html .= '<tr><td width="25%">'.$langs->trans("Label").'</td>';
		// $html .= '<td colspan="2">'. $this->label .'</td></tr>';
		$html .= '<tr><td width="25%">'.$langs->trans("Ref").'</td>';
		$html .= '<td colspan="2">'. $this->ref .' &nbsp;&nbsp;&nbsp;&nbsp;'. $langs->trans("ReleaseDate").': '. dol_print_date($this->created_date, 'day') .'</td></tr>';
		$html .= '</table><br />';

		return $html;
    }

    public function getTenders($filter = ' AND statut = 1')
    {
    	$this->fetchAll($sortorder = 'DESC', $sortfield = 'rowid',0,0, $filter);
    	$tenders = array();

    	foreach ($this->lines as $key => $line) {
    		$tenders[$line->id] = $line->ref;
    	}

    	return $tenders;
    }

    public function load_state_board()
	{
		global $conf, $user;

		$this->nb=array();

		$sql = "SELECT count(t.rowid) as nb";
		$sql.= " FROM ".MAIN_DB_PREFIX. $this->table_element ." as t";

		$resql=$this->db->query($sql);
		
		if ($resql) {
			while ($obj=$this->db->fetch_object($resql)) {
				$this->nb["tenders"]=$obj->nb;
			}
            $this->db->free($resql);
			return 1;
		}
		else {
			dol_print_error($this->db);
			$this->error=$this->db->error();
			return -1;
		}
	}

	/**
     * Load indicators for dashboard (this->nbtodo and this->nbtodolate)
     *
     * @param	User	$user   Objet user
     * @return WorkboardResponse|int <0 if KO, WorkboardResponse if OK
     */
    public function load_board($tender = null)
    {
        global $conf, $user, $langs;

		require_once DOL_DOCUMENT_ROOT . '/btp/tender/releases/class/releases.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/tender/interimbail/class/interimbail.class.php';
        $releases 	 = new Releases($this->db);
        $interimbail = new InterimBail($this->db);

        return array($releases->getStatData($tender), $interimbail->getStatData($tender));
    }
}
