<?php
/* Copyright (C) Yassine Belkaid y.belkaid@nextconcept.ma
 *
 */

/**
 * \file    btp/market/works/class/works.class.php
 * \ingroup works
 * \brief   This file is works CRUD class file
 *
 */

require_once DOL_DOCUMENT_ROOT . '/btp/db/nxdb.class.php';

/**
 * Class btp
 *
 * MarketWorks Management class
 */
class MarketWorks extends NXdb
{
	/**
	 * @var string Error code (or message)
	 * @deprecated
	 * @see btp::errors
	 */
	public $error;
	/**
	 * @var string[] Error codes (or messages)
	 */
	public $errors = array();
	/**
	 * @var string Id to identify managed objects
	 */
	public $element = 'td_works';
	/**
	 * @var string Name of table without ptask_marcheix where object is stored
	 */
	public $table_element 	 = 'td_works';

	/**
	 * @var int ID
	 */
	public $id;
	public $fk_createdby;
	public $fk_market;
	public $fk_unit_type;
	public $task;
	public $price;
	public $qty;
	public $created_at;
	public $updated_at;

	public $now;
	public $lines = array();
	/**
	 * Constructor
	 *
	 * @param DoliDb $db Database handler
	 */
	public function __construct(DoliDBMysqli $db)
	{
		$this->db 		 = $db;
		$this->now 		 = new \DateTime("now");
		$this->now 		 = $this->now->format('Y-m-d H:i:s');
		$this->updated_at= $this->now;
		return 1;
	}

	/**
	 * Create object into database
	 *
	 * @return int   id of last inserted id, otherwise -1 if error arised 
	 */
	public function create()
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		// Clean parameters
		$this->fk_createdby = $this->fk_createdby ? $this->db->escape($this->fk_createdby) : 0;
		$this->fk_market 	= $this->fk_market ? $this->db->escape($this->fk_market) : 0;
		$this->fk_unit_type	= $this->fk_unit_type ? $this->db->escape((int)$this->fk_unit_type) : 0;
		$this->qty			= $this->qty ? $this->db->escape((int)$this->qty) : 0;
		$this->price 		= $this->price ? $this->db->escape((float)$this->price) : 0;
		$this->task 		= $this->task ? $this->db->escape($this->task) : null;
		$this->created_at 	= $this->now;

		// Insert request
		$sql = 'INSERT INTO ' . MAIN_DB_PREFIX . $this->table_element . ' (qty, fk_createdby, fk_market, created_at, updated_at, fk_unit_type, price, task) VALUES (';
		$sql .= $this->qty .",".$this->fk_createdby.", ".$this->fk_market.", '".$this->created_at."', '".$this->updated_at ."', ".$this->fk_unit_type.", ". $this->price .", '". $this->task ."')";

		$this->db->begin();
		$resql = $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error works ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return $this->getLasInsrtedId();
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param int    $id  Id object
	 * @param string $task task
	 *
	 * @return int <0 if KO, 0 if not found, >0 if OK
	 */
	public function fetch($id, $fk_market = null)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;

		if (null !== $fk_market) {
			$sql .= ' WHERE fk_market = ' . $fk_market;
		} else {
			$sql .= ' WHERE rowid = ' . $id;
		}

		$resql = $this->db->query($sql);
		if ($resql) {
			$numrows = $this->db->num_rows($resql);

			if ($numrows) {
				$obj 				 = $this->db->fetch_object($resql);
				$this->id 			 = $obj->rowid;
				$this->fk_createdby  = $obj->fk_createdby;
				$this->fk_market 	 = $obj->fk_market;
				$this->fk_unit_type	 = $obj->fk_unit_type;
				$this->price 		 = $obj->price;
				$this->qty  		 = $obj->qty;
				$this->task  		 = $obj->task;
				$this->updated_at 	 = $this->db->jdate($obj->updated_at);
				$this->created_at 	 = $this->db->jdate($obj->created_at);
			}

			$this->db->free($resql);

			if ($numrows) {
				return 1;
			} else {
				return 0;
			}
		} else {
			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Load object in memory from the database
	 *
	 * @param string $sortorder Sort Order
	 * @param string $sortfield Sort field
	 * @param int    $limit     offset limit
	 * @param int    $offset    offset limit
	 * @param array  $filter    filter array
	 * @param string $filtermode filter mode (AND or OR)
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function fetchAll($mid = null, $sortorder = '', $sortfield = '', $limit = 0, $offset = 0, $filter = '', $ignore_id = false)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$sql = 'SELECT * FROM ' . MAIN_DB_PREFIX . $this->table_element;

		if (!empty($filter)) {
			$sql .= ' WHERE '. substr($filter, 4) .' AND fk_market = '. $mid;
		} else {
			if (!$ignore_id)
				$sql .= ' WHERE fk_market = '. $mid;
		}

		if (!empty($sortfield)) {
			$sql .= $this->db->order($sortfield, $sortorder);
		}
        
         if(!empty($limit)){
         $sql.= " limit  ".$limit ;
		}
		$this->lines = array();
		$resql 		 = $this->db->query($sql);
		if ($resql) {
			$num = $this->db->num_rows($resql);

			while ($obj = $this->db->fetch_object($resql)) {
				$line = new stdClass;

				$line->id 			= $obj->rowid;
				$line->fk_createdby = $obj->fk_createdby;
				$line->fk_market 	= $obj->fk_market;
				$line->fk_unit_type = $obj->fk_unit_type;
				$line->price 		= $obj->price;
				$line->qty 		 	= $obj->qty;
				$line->task 	 	= $obj->task;
				$line->updated_at 	= $this->db->jdate($obj->updated_at);
				$line->created_at 	= $this->db->jdate($obj->created_at);
				$this->lines[] 		= $line;
			}
			$this->db->free($resql);

			return $num;
		} else {
			$this->error 	= 'Error ' . $this->db->lasterror();
			$this->errors[] = $this->error;
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		}
	}

	/**
	 * Update object into database
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function update($id, array $data)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		if (!$id)
			return false;

		if (count($data) && is_array($data)) {
			// Update request
			$sql = 'UPDATE ' . MAIN_DB_PREFIX . $this->table_element . ' SET updated_at = "'. $this->now .'",';

			foreach ($data as $key => $val) {
				$val = is_numeric($val) ? $val : '"'. $this->db->escape($val) .'"';
				$sql .= ' `'. $key. '` = '. $val .',';
			}

			$sql  = substr($sql, 0, -1);
			$sql .= ' WHERE rowid = ' . $id;

			$this->db->begin();

			$resql = $this->db->query($sql);

			if (!$resql) {
				$this->db->rollback();
				
				$this->error = 'Error ' . $this->db->lasterror();
				$this->errors[] = $this->error;
				dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

				return -1;
			} else {
				$this->db->commit();

				return 1;
			}
		}
	}

	/**
	 * Delete object in database
	 *
	 * @param User $user      User that deletes
	 * @param bool $notrigger false=launch triggers after, true=disable triggers
	 *
	 * @return int <0 if KO, >0 if OK
	 */
	public function delete($id)
	{
		dol_syslog(__METHOD__, LOG_DEBUG);

		$this->db->begin();

		$sql 	= 'DELETE FROM ' . MAIN_DB_PREFIX . $this->table_element .' WHERE rowid = ' . $id;
		$resql 	= $this->db->query($sql);

		if (!$resql) {
			$this->db->rollback();

			$this->errors[] = 'Error ' . $this->db->lasterror();
			dol_syslog(__METHOD__ . ' ' . join(',', $this->errors), LOG_ERR);

			return -1;
		} else {
			$this->db->commit();

			return 1;
		}
	}

	/**
     *	Return clicable name (with picto eventually)
     *
     *  @author Yassine Belkaid <y.belkaid@nextconcept.ma>
     *	@param		int			$withpicto		0=_No picto, 1=Includes the picto in the linkn, 2=Picto only
     *	@return		string						String with URL
     */
    public function getNomUrl($withpicto=0,  $id, $mid = null)
    {
        global $langs;

        $result	= '';
        $settask	= '';

        $id  	= ($id  ?: '');
        $label  = $langs->trans("Show").': '. $id;

        $link 	 = '<a href="'.DOL_URL_ROOT.'/btp/market/works/index.php?id='. $id .'&mid='. $mid .'" class="classfortooltip" title="'. $label .'">';
        $linkend ='</a>';
        $picto   = $this->defaultPicto;

        if ($withpicto) $result.=($link.img_object($label, $picto, 'class="classfortooltip"').$linkend);
        if ($withpicto && $withpicto != 2) $result.=' ';
        if ($withpicto != 2) $result.=$link.$label.$linkend;
        return $result;
    }

    public function getStatData($market_id = false)
    {
    	global $langs;
    	$this->fetchAll(($market_id ?: null), '', '', 0, 0, '', ($market_id ? false : true));
    	$countAmount = 0;

    	foreach ($this->lines as $key => $obj)
    		$countAmount += ($obj->price*$obj->qty);

    	$object 		= new stdClass();
    	$object->number = count($this->lines);
    	$object->cost 	= $countAmount;
    	$object->label 	= $langs->trans('Works');
    	$object->url 	= (int)$market_id ? DOL_URL_ROOT.'/btp/market/works/index.php?mid='.(int)$market_id : '';

    	return $object;
    }

    /**
     * Load indicators for dashboard (this->nbtodo and this->nbtodolate)
     *
     * @param	User	$user   Objet user
     * @return WorkboardResponse|int <0 if KO, WorkboardResponse if OK
     */
    public function load_board($work_id = false, $market_id = 0)
    {
        global $conf, $user, $langs;

		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_supplies/class/work_supplies.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_employees/class/work_employees.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_equipments/class/work_equipments.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_yields/class/work_yields.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_coefficients/class/work_coefficients.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_metres/class/work_metres.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_subcontracting/class/work_subcontracting.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_delays/class/work_delays.class.php';
		require_once DOL_DOCUMENT_ROOT . '/btp/market/work_advancement/class/work_advancement.class.php';

        $marketWorkSupplies  = new MarketWorkSupplies($this->db);
        $marketWorkEmployees = new MarketWorkEmployees($this->db);
        $marketWorkEquipments= new MarketWorkEquipments($this->db);
        $marketWorkYields  	 = new MarketWorkYields($this->db);
        $marketWorkCoefficients = new MarketWorkCoefficients($this->db);
        $marketWorkMetres 	 = new MarketWorkMetres($this->db);
        $marketWorkSubContracting = new MarketWorkSubContracting($this->db);
        $marketWorkDelays 	 = new MarketWorkDelays($this->db);
        $workAdvancement 	 = new WorkAdvancement($this->db);

        if ($work_id) {
        	$this->fetch($work_id);
        	$market_id = $this->fk_market;
        }

        return array($marketWorkSupplies->getStatData($work_id, $market_id), $marketWorkEmployees->getStatData($work_id, $market_id), $marketWorkEquipments->getStatData($work_id, $market_id), $marketWorkYields->getStatData($work_id, $market_id),
        	$marketWorkCoefficients->getStatData($work_id, $market_id), $marketWorkMetres->getStatData($work_id, $market_id),
        	$marketWorkSubContracting->getStatData($work_id, $market_id), $marketWorkDelays->getStatData($work_id, $market_id), 
        	$workAdvancement->getStatData($work_id, $market_id));
    }

    public function getNames($mid = null)
    {
    	if (!$mid)
    		return false;

    	$this->fetchAll($mid, '', '', 0, 0, '', false);
    	$names = array();

    	if (count($this->lines))
    	foreach ($this->lines as $key => $line) {
			$names[$line->id] = $line->task;
    	}

    	return $names;
    }
}
